@extends('layouts.horizontal', ['title' => 'Flexi-Pricing Configuration', 'topbarTitle' => 'Flexi-Pricing'])

@section('css')
@vite(['node_modules/datatables.net-bs5/css/dataTables.bootstrap5.min.css',
'node_modules/datatables.net-responsive-bs5/css/responsive.bootstrap5.min.css',
'node_modules/select2/dist/css/select2.min.css'])
@endsection

@section('content')

<div class="flexi-pricing">
    <!-- start page title -->
    <div class="row mb-4">
        <div class="col-12">
            <div class="page-title-box d-flex align-items-center justify-content-between">
                <h2 class="mb-0">Flexi-Pricing Configuration</h2>
                <div class="page-title-right d-flex align-items-center gap-3">
                    @php
                        $breadcrumb = getCurrentMenuBreadcrumb();
                    @endphp
                    @if(!empty($breadcrumb))
                        <nav aria-label="breadcrumb">
                            <ol class="breadcrumb mb-0 py-2">
                                @foreach($breadcrumb as $index => $item)
                                    @if($index === count($breadcrumb) - 1)
                                        <li class="breadcrumb-item active" aria-current="page">{{ $item }}</li>
                                    @else
                                        <li class="breadcrumb-item"><a href="javascript: void(0);">{{ $item }}</a></li>
                                    @endif
                                @endforeach
                            </ol>
                        </nav>
                    @endif
                </div>
            </div>
        </div>
    </div>
    <!-- end page title -->

    @if($business->flexi_pricing_enabled != 'Yes')
    <div class="row">
        <div class="col-12">
            <div class="alert alert-warning">
                <strong>Flexi-Pricing is disabled!</strong> Please enable it in Branch Settings > Modules section.
            </div>
        </div>
    </div>
    @endif

    <div class="row mb-3">
        <div class="col-12">
            <div class="card">
                <div class="card-body">
                    <div class="d-flex justify-content-between align-items-center">
                        <h4 class="card-title mb-0">Business Time Slots</h4>
                        <button type="button" class="btn btn-primary" onclick="openBusinessTimeslotsModal()">
                            <i class="ri-add-line me-1"></i> Manage Time Slots
                        </button>
                    </div>
                    <p class="text-muted mb-0">Time slots are universal and can be used across all mappings.</p>
                </div>
            </div>
        </div>
    </div>

    <div class="row">
        <div class="col-12">
            <div class="card">
                <div class="card-body">
                    <div class="d-flex justify-content-between align-items-center mb-3">
                        <h4 class="card-title mb-0">Category Mappings</h4>
                        <button type="button" class="btn btn-primary" onclick="openMappingModal()">
                            <i class="ri-add-line me-1"></i> New Mapping
                        </button>
                    </div>
                    <div class="table-responsive">
                        <table id="mappings-table" class="table table-hover">
                            <thead>
                                <tr>
                                    <th>ID</th>
                                    <th>Mapping Name</th>
                                    <th>Package Categories</th>
                                    <th>Status</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody id="mappings-tbody">
                                <!-- Populated via AJAX -->
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Mapping Modal -->
<div class="modal fade" id="mappingModal" tabindex="-1" aria-labelledby="mappingModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="mappingModalLabel">Create/Edit Mapping</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
                <form id="mappingForm">
                    <input type="hidden" id="mapping_id" name="mapping_id">
                    <div class="mb-3">
                        <label for="mapping_name" class="form-label">Mapping Name <span class="text-danger">*</span></label>
                        <input type="text" class="form-control" id="mapping_name" name="mapping_name" required>
                    </div>
                    <div class="mb-3">
                        <label for="package_categories" class="form-label">Package Categories <span class="text-danger">*</span></label>
                        <select class="form-select select2" id="package_categories" name="package_category_ids[]" multiple required>
                            <!-- Populated via AJAX -->
                        </select>
                    </div>
                </form>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                <button type="button" class="btn btn-primary" onclick="saveMapping()">Save Mapping</button>
            </div>
        </div>
    </div>
</div>

<!-- Mapping Details Modal -->
<div class="modal fade" id="mappingDetailsModal" tabindex="-1" aria-labelledby="mappingDetailsModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-xl">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="mappingDetailsModalLabel">Mapping Details</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
                <ul class="nav nav-tabs" role="tablist">
                    <li class="nav-item" role="presentation">
                        <button class="nav-link active" data-bs-toggle="tab" data-bs-target="#floors-tab" type="button">Floors</button>
                    </li>
                    <li class="nav-item" role="presentation">
                        <button class="nav-link" data-bs-toggle="tab" data-bs-target="#bands-tab" type="button">Pricing Bands</button>
                    </li>
                </ul>
                <div class="tab-content mt-3">
                    <!-- Floors Tab -->
                    <div class="tab-pane fade show active" id="floors-tab" role="tabpanel">
                        <div class="mb-3">
                            <button type="button" class="btn btn-sm btn-primary" onclick="loadFloors()">Refresh</button>
                        </div>
                        <div id="floors-content">
                            <!-- Populated via AJAX -->
                        </div>
                    </div>
                    <!-- Pricing Bands Tab -->
                    <div class="tab-pane fade" id="bands-tab" role="tabpanel">
                        <div class="mb-3">
                            <button type="button" class="btn btn-sm btn-primary" onclick="loadBands()">Refresh</button>
                        </div>
                        <div id="bands-content">
                            <!-- Populated via AJAX -->
                        </div>
                    </div>
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
            </div>
        </div>
    </div>
</div>

<!-- Business Timeslots Modal -->
<div class="modal fade" id="businessTimeslotsModal" tabindex="-1" aria-labelledby="businessTimeslotsModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="businessTimeslotsModalLabel">Manage Business Time Slots</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
                <div class="alert alert-info">Time slots are universal and can be used across all mappings.</div>
                <div class="mb-3">
                    <button type="button" class="btn btn-sm btn-primary" onclick="addBusinessTimeslotRow()">Add Time Slot</button>
                </div>
                <div id="business-timeslots-content">
                    <!-- Populated via AJAX -->
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
            </div>
        </div>
    </div>
</div>

@endsection

@section('scripts')
<script>
let baseURL = '{{ url("") }}'; // Use relative paths
let currentMappingId = null;

$(document).ready(function() {
    loadMappings();
    
    // Initialize select2 with multiple selection without CTRL
    $('#package_categories').select2({
        dropdownParent: $('#mappingModal'),
        placeholder: 'Select Package Categories',
        closeOnSelect: false, // Don't close dropdown when selecting
        allowClear: true
    });
});

function loadMappings() {
    $.ajax({
        url: baseURL + '/flexi-pricing/mappings',
        type: 'GET',
        success: function(response) {
            if (response.success) {
                let html = '';
                if (response.data && response.data.length > 0) {
                    response.data.forEach(function(mapping) {
                        // Handle both camelCase and snake_case relationship names
                        let packages = mapping.category_mapping_packages || mapping.categoryMappingPackages || [];
                        
                        let categories = [];
                        if (packages && packages.length > 0) {
                            categories = packages.map(p => {
                                let pkg = p.package_category || p.packageCategory;
                                if (!pkg) {
                                    return 'Unknown';
                                }
                                let packageType = 'Unknown';
                                if (pkg.package_type && pkg.package_type.service_type) {
                                    packageType = pkg.package_type.service_type;
                                } else if (pkg.packageType && pkg.packageType.service_type) {
                                    packageType = pkg.packageType.service_type;
                                }
                                return packageType + ' - ' + (pkg.service_category || 'Unnamed');
                            });
                        }
                        
                        html += `
                            <tr>
                                <td>${mapping.id_mapping}</td>
                                <td>${mapping.mapping_name}</td>
                                <td>${categories.length > 0 ? categories.join(', ') : 'None'}</td>
                                <td><span class="badge badge-outline-${mapping.active ? 'success' : 'danger'}">${mapping.active ? 'Active' : 'Inactive'}</span></td>
                                <td>
                                    <button class="btn btn-sm btn-primary me-1" onclick="openMappingDetailsModal(${mapping.id_mapping})">Configure</button>
                                    <button class="btn btn-sm btn-danger" onclick="deleteMapping(${mapping.id_mapping})">Delete</button>
                                </td>
                            </tr>
                        `;
                    });
                } else {
                    html = '<tr><td colspan="5" class="text-center">No mappings found. Click "New Mapping" to create one.</td></tr>';
                }
                $('#mappings-tbody').html(html);
            } else {
                toastr.error('Failed to load mappings: ' + (response.message || 'Unknown error'));
            }
        },
        error: function(xhr) {
            let errorMsg = 'Error loading mappings';
            if (xhr.responseJSON && xhr.responseJSON.message) {
                errorMsg += ': ' + xhr.responseJSON.message;
            }
            toastr.error(errorMsg);
            $('#mappings-tbody').html('<tr><td colspan="5" class="text-center text-danger">Error loading mappings. Please refresh the page.</td></tr>');
        }
    });
}

function openMappingModal(mappingId = null) {
    $('#mappingForm')[0].reset();
    $('#mapping_id').val(mappingId || '');
    $('#mappingModalLabel').text(mappingId ? 'Edit Mapping' : 'Create Mapping');
    
    // Load package categories
    $.ajax({
        url: baseURL + '/flexi-pricing/mappings/' + (mappingId || '0'),
        type: 'GET',
        success: function(response) {
            if (response.success && response.availablePackages) {
                let html = '';
                response.availablePackages.forEach(function(pkg) {
                    let displayText = (pkg.service_type || 'Unknown') + ' - ' + (pkg.service_category || 'Unnamed');
                    html += `<option value="${pkg.id_package_category}">${displayText}</option>`;
                });
                $('#package_categories').html(html);
                
                if (mappingId && response.mapping) {
                    $('#mapping_name').val(response.mapping.mapping_name);
                    let selectedIds = response.mapping.category_mapping_packages.map(p => p.package_category_id);
                    $('#package_categories').val(selectedIds).trigger('change');
                } else {
                    // Clear selection for new mapping
                    $('#package_categories').val(null).trigger('change');
                }
            } else {
                toastr.error('Failed to load package categories');
            }
        },
        error: function(xhr) {
            toastr.error('Error loading package categories');
        }
    });
    
    // Show modal using Bootstrap 5
    var modalElement = document.getElementById('mappingModal');
    var modal = new bootstrap.Modal(modalElement);
    modal.show();
}

function saveMapping() {
    let formData = {
        mapping_name: $('#mapping_name').val(),
        mapping_id: $('#mapping_id').val() || null,
        package_category_ids: $('#package_categories').val()
    };
    
    if (!formData.mapping_name || !formData.mapping_name.trim()) {
        toastr.error('Please enter a mapping name');
        return;
    }
    
    if (!formData.package_category_ids || formData.package_category_ids.length === 0) {
        toastr.error('Please select at least one package category');
        return;
    }
    
    $.ajax({
        url: baseURL + '/flexi-pricing/mappings',
        type: 'POST',
        data: formData,
        headers: {'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content')},
        success: function(response) {
            if (response.success) {
                toastr.success(response.message);
                // Close modal using Bootstrap 5 API
                var modalElement = document.getElementById('mappingModal');
                var modal = bootstrap.Modal.getInstance(modalElement);
                if (modal) {
                    modal.hide();
                }
                loadMappings();
            } else {
                toastr.error(response.message);
            }
        },
        error: function(xhr) {
            let errorMsg = xhr.responseJSON?.message || 'Error saving mapping';
            toastr.error(errorMsg);
        }
    });
}

function deleteMapping(id) {
    Swal.fire({
        title: 'Are you sure?',
        text: "You won't be able to revert this!",
        icon: 'warning',
        showCancelButton: true,
        confirmButtonText: 'Yes, delete it!',
        cancelButtonText: 'Cancel',
        customClass: {
            confirmButton: 'btn btn-danger me-2',
            cancelButton: 'btn btn-secondary'
        },
        buttonsStyling: false
    }).then((result) => {
        if (result.isConfirmed) {
            $.ajax({
                url: baseURL + '/flexi-pricing/mappings/' + id,
                type: 'DELETE',
                headers: {'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content')},
                success: function(response) {
                    if (response.success) {
                        Swal.fire({
                            title: 'Deleted!',
                            text: response.message,
                            icon: 'success',
                            customClass: {
                                confirmButton: 'btn btn-primary'
                            },
                            buttonsStyling: false
                        });
                        loadMappings();
                    } else {
                        Swal.fire({
                            title: 'Error!',
                            text: response.message,
                            icon: 'error',
                            customClass: {
                                confirmButton: 'btn btn-primary'
                            },
                            buttonsStyling: false
                        });
                    }
                },
                error: function(xhr) {
                    let errorMsg = xhr.responseJSON?.message || 'Error deleting mapping';
                    Swal.fire({
                        title: 'Error!',
                        text: errorMsg,
                        icon: 'error',
                        customClass: {
                            confirmButton: 'btn btn-primary'
                        },
                        buttonsStyling: false
                    });
                }
            });
        }
    });
}

function openMappingDetailsModal(mappingId) {
    currentMappingId = mappingId;
    // Show modal using Bootstrap 5 API
    var modalElement = document.getElementById('mappingDetailsModal');
    var modal = new bootstrap.Modal(modalElement);
    modal.show();
    loadMappingDetails(mappingId);
}

function loadMappingDetails(mappingId) {
    $.ajax({
        url: baseURL + '/flexi-pricing/mappings/' + mappingId,
        type: 'GET',
        success: function(response) {
            if (response.success) {
                $('#mappingDetailsModalLabel').text('Mapping Details: ' + response.mapping.mapping_name);
                // Store business timeslots globally for bands dropdown
                if (response.businessTimeslots) {
                    window.availableTimeslots = response.businessTimeslots.map(ts => ({
                        id: ts.id_business_timeslots,
                        name: ts.timeslot_name
                    }));
                }
                loadFloors();
                loadBands();
            }
        }
    });
}

function openBusinessTimeslotsModal() {
    // Load business timeslots
    $.ajax({
        url: baseURL + '/flexi-pricing/business-timeslots',
        type: 'GET',
        success: function(response) {
            if (response.success) {
                let html = '<div id="business-timeslots-list">';
                if (response.timeslots && response.timeslots.length > 0) {
                    response.timeslots.forEach(function(ts) {
                        html += createBusinessTimeslotRow(ts.id_business_timeslots, ts.timeslot_name, ts.start_time, ts.end_time);
                    });
                } else {
                    html += createBusinessTimeslotRow();
                }
                html += '</div><button class="btn btn-primary mt-2" onclick="saveBusinessTimeslots()">Save Time Slots</button>';
                $('#business-timeslots-content').html(html);
                
                // Show modal
                var modalElement = document.getElementById('businessTimeslotsModal');
                var modal = new bootstrap.Modal(modalElement);
                modal.show();
            }
        },
        error: function(xhr) {
            toastr.error('Error loading time slots');
        }
    });
}

function createBusinessTimeslotRow(id = null, name = '', start = '', end = '') {
    return `
        <div class="row mb-2 business-timeslot-row" data-id="${id || ''}">
            <div class="col-md-4">
                <input type="text" class="form-control business-timeslot-name" placeholder="Time Slot Name" value="${name}">
            </div>
            <div class="col-md-3">
                <input type="time" class="form-control business-timeslot-start" value="${start}">
            </div>
            <div class="col-md-3">
                <input type="time" class="form-control business-timeslot-end" value="${end}">
            </div>
            <div class="col-md-2">
                <button class="btn btn-sm btn-danger" onclick="removeBusinessTimeslotRow(this)">Remove</button>
            </div>
        </div>
    `;
}

function addBusinessTimeslotRow() {
    $('#business-timeslots-list').append(createBusinessTimeslotRow());
}

function removeBusinessTimeslotRow(btn) {
    $(btn).closest('.business-timeslot-row').remove();
}

function saveBusinessTimeslots() {
    let timeslots = [];
    $('.business-timeslot-row').each(function() {
        let id = $(this).data('id');
        let name = $(this).find('.business-timeslot-name').val();
        let start = $(this).find('.business-timeslot-start').val();
        let end = $(this).find('.business-timeslot-end').val();
        
        if (name && start && end) {
            timeslots.push({
                id: id || null,
                timeslot_name: name,
                start_time: start,
                end_time: end
            });
        }
    });
    
    $.ajax({
        url: baseURL + '/flexi-pricing/business-timeslots',
        type: 'POST',
        data: {
            timeslots: timeslots
        },
        headers: {'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content')},
        success: function(response) {
            if (response.success) {
                toastr.success(response.message);
                // Update global timeslots for bands
                if (response.timeslots) {
                    window.availableTimeslots = response.timeslots.map(ts => ({
                        id: ts.id_business_timeslots,
                        name: ts.timeslot_name
                    }));
                }
                // Close modal
                var modalElement = document.getElementById('businessTimeslotsModal');
                var modal = bootstrap.Modal.getInstance(modalElement);
                if (modal) {
                    modal.hide();
                }
            } else {
                toastr.error(response.message);
            }
        },
        error: function(xhr) {
            let errorMsg = xhr.responseJSON?.message || 'Error saving time slots';
            toastr.error(errorMsg);
        }
    });
}

function loadFloors() {
    if (!currentMappingId) return;
    
    $.ajax({
        url: baseURL + '/flexi-pricing/mappings/' + currentMappingId,
        type: 'GET',
        success: function(response) {
            if (response.success) {
                let html = '<table class="table table-bordered"><thead><tr><th>Floor Name</th><th>Capacity</th><th>Active</th></tr></thead><tbody>';
                
                response.availableFloors.forEach(function(floor) {
                    let mappingFloor = response.mapping.mapping_floors.find(mf => mf.business_floor_id == floor.id_business_floors);
                    let checked = mappingFloor && mappingFloor.active ? 'checked' : '';
                    html += `
                        <tr>
                            <td>${floor.floor_name}</td>
                            <td>${floor.floor_workstations || 0}</td>
                            <td>
                                <input type="checkbox" class="form-check-input floor-checkbox" 
                                    data-floor-id="${floor.id_business_floors}" ${checked}>
                            </td>
                        </tr>
                    `;
                });
                
                html += '</tbody></table><button class="btn btn-primary mt-2" onclick="saveFloors()">Save Floors</button>';
                $('#floors-content').html(html);
            }
        }
    });
}

function saveFloors() {
    let floors = [];
    $('.floor-checkbox').each(function() {
        floors.push({
            business_floor_id: $(this).data('floor-id'),
            active: $(this).is(':checked') ? 1 : 0
        });
    });
    
    $.ajax({
        url: baseURL + '/flexi-pricing/floors',
        type: 'POST',
        data: {
            mapping_id: currentMappingId,
            floors: floors
        },
        headers: {'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content')},
        success: function(response) {
            if (response.success) {
                toastr.success(response.message);
                loadFloors();
            } else {
                toastr.error(response.message);
            }
        }
    });
}

function loadBands() {
    if (!currentMappingId) return;
    
    $.ajax({
        url: baseURL + '/flexi-pricing/mappings/' + currentMappingId,
        type: 'GET',
        success: function(response) {
            if (response.success) {
                let html = '<div class="alert alert-info">Configure pricing bands for each day. Bands must cover 0-100% with no gaps.</div>';
                html += '<table class="table table-bordered"><thead><tr><th>Day</th><th>Time Slot</th><th>Min %</th><th>Max %</th><th>Discount Type</th><th>Discount</th><th>Actions</th></tr></thead><tbody id="bands-tbody"></tbody></table>';
                html += '<button class="btn btn-primary mt-2" onclick="addBandRow()">Add Band</button>';
                html += '<button class="btn btn-success mt-2 ms-2" onclick="saveBands()">Save All Bands</button>';
                
                $('#bands-content').html(html);
                
                // Store business timeslots for dropdown
                if (response.businessTimeslots && response.businessTimeslots.length > 0) {
                    window.availableTimeslots = response.businessTimeslots.map(ts => ({
                        id: ts.id_business_timeslots,
                        name: ts.timeslot_name
                    }));
                } else {
                    window.availableTimeslots = [];
                }
                
                // Load existing bands grouped by day and timeslot
                let bandsByKey = {};
                if (response.mapping.pricing_bands) {
                    response.mapping.pricing_bands.forEach(function(band) {
                        if (band.active == 1) { // Only show active bands
                            let key = band.day + '_' + (band.timeslot_id || 'null');
                            if (!bandsByKey[key]) bandsByKey[key] = [];
                            bandsByKey[key].push(band);
                        }
                    });
                }
                
                // Render bands
                for (let day = 1; day <= 7; day++) {
                    let dayName = ['Sunday', 'Monday', 'Tuesday', 'Wednesday', 'Thursday', 'Friday', 'Saturday'][day - 1];
                    let dayKey = day + '_null';
                    let dayBands = bandsByKey[dayKey] || [];
                    
                    if (dayBands.length > 0) {
                        dayBands.forEach(function(band) {
                            $('#bands-tbody').append(createBandRow(day, dayName, null, band, 'All Day'));
                        });
                    }
                    
                    // Add timeslot bands if any (using business timeslots)
                    if (response.businessTimeslots) {
                        response.businessTimeslots.forEach(function(ts) {
                            let tsKey = day + '_' + ts.id_business_timeslots;
                            let tsBands = bandsByKey[tsKey] || [];
                            tsBands.forEach(function(band) {
                                $('#bands-tbody').append(createBandRow(day, dayName, ts.id_business_timeslots, band, ts.timeslot_name));
                            });
                        });
                    }
                }
            }
        }
    });
}

function createBandRow(day = null, dayName = null, timeslotId = null, band = null, timeslotName = null) {
    // If called without parameters, create empty row for new band
    if (day === null) {
        day = 1; // Default to Sunday
        dayName = 'Sunday';
        timeslotId = null;
        timeslotName = 'All Day';
    }
    
    let bandId = band ? band.id_pricing_bands : '';
    let minPct = band ? band.min_percentage : '';
    let maxPct = band ? band.max_percentage : '';
    let discType = band ? band.discount_type : 'percentage';
    let discount = band ? band.discount : '';
    
    // Build day dropdown
    let dayOptions = '';
    const dayNames = ['Sunday', 'Monday', 'Tuesday', 'Wednesday', 'Thursday', 'Friday', 'Saturday'];
    for (let i = 1; i <= 7; i++) {
        dayOptions += `<option value="${i}" ${i == day ? 'selected' : ''}>${dayNames[i-1]}</option>`;
    }
    
    // Build timeslot dropdown
    let timeslotOptions = '<option value="">All Day</option>';
    if (window.availableTimeslots && window.availableTimeslots.length > 0) {
        window.availableTimeslots.forEach(function(ts) {
            timeslotOptions += `<option value="${ts.id}" ${ts.id == timeslotId ? 'selected' : ''}>${ts.name}</option>`;
        });
    }
    
    return `
        <tr class="band-row" data-band-id="${bandId}" data-day="${day}" data-timeslot-id="${timeslotId || ''}">
            <td>
                <select class="form-select form-select-sm band-day" onchange="updateBandDay(this)">
                    ${dayOptions}
                </select>
            </td>
            <td>
                <select class="form-select form-select-sm band-timeslot" onchange="updateBandTimeslot(this)">
                    ${timeslotOptions}
                </select>
            </td>
            <td><input type="number" class="form-control form-control-sm band-min" value="${minPct}" min="0" max="100" step="0.01"></td>
            <td><input type="number" class="form-control form-control-sm band-max" value="${maxPct}" min="0" max="100" step="0.01"></td>
            <td>
                <select class="form-select form-select-sm band-type">
                    <option value="percentage" ${discType == 'percentage' ? 'selected' : ''}>Percentage</option>
                    <option value="fixed" ${discType == 'fixed' ? 'selected' : ''}>Fixed</option>
                </select>
            </td>
            <td><input type="number" class="form-control form-control-sm band-discount" value="${discount}" min="0" step="0.01"></td>
            <td><button class="btn btn-sm btn-danger" onclick="removeBandRow(this)">Remove</button></td>
        </tr>
    `;
}

function updateBandDay(select) {
    let row = $(select).closest('.band-row');
    let day = parseInt($(select).val());
    row.attr('data-day', day);
}

function updateBandTimeslot(select) {
    let row = $(select).closest('.band-row');
    let timeslotId = $(select).val() || null;
    row.attr('data-timeslot-id', timeslotId || '');
}

function addBandRow() {
    // Add new row directly to table
    $('#bands-tbody').append(createBandRow());
}

function removeBandRow(btn) {
    $(btn).closest('.band-row').remove();
}

function saveBands() {
    let bands = [];
    $('.band-row').each(function() {
        let day = parseInt($(this).find('.band-day').val());
        let timeslotId = $(this).find('.band-timeslot').val() || null;
        let minPct = parseFloat($(this).find('.band-min').val());
        let maxPct = parseFloat($(this).find('.band-max').val());
        let discType = $(this).find('.band-type').val();
        let discount = parseFloat($(this).find('.band-discount').val());
        
        // Validate required fields
        if (isNaN(day) || day < 1 || day > 7) {
            toastr.error('Please select a valid day for all bands');
            return false;
        }
        
        if (isNaN(minPct) || isNaN(maxPct) || isNaN(discount)) {
            toastr.error('Please fill in all percentage and discount fields');
            return false;
        }
        
        if (minPct < 0 || minPct > 100 || maxPct < 0 || maxPct > 100) {
            toastr.error('Percentages must be between 0 and 100');
            return false;
        }
        
        if (discount < 0) {
            toastr.error('Discount cannot be negative');
            return false;
        }
        
        bands.push({
            day: day,
            timeslot_id: timeslotId,
            min_percentage: minPct,
            max_percentage: maxPct,
            discount_type: discType,
            discount: discount
        });
    });
    
    if (bands.length === 0) {
        toastr.error('Please add at least one pricing band');
        return;
    }
    
    $.ajax({
        url: baseURL + '/flexi-pricing/pricing-bands',
        type: 'POST',
        data: {
            mapping_id: currentMappingId,
            bands: bands
        },
        headers: {'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content')},
        success: function(response) {
            if (response.success) {
                toastr.success(response.message);
                loadBands();
            } else {
                // Show detailed error messages
                let errorMsg = response.message || 'Error saving pricing bands';
                if (response.validation_errors && Array.isArray(response.validation_errors)) {
                    errorMsg += '<br><ul style="text-align: left; margin-top: 10px;">';
                    response.validation_errors.forEach(function(err) {
                        errorMsg += '<li>' + err + '</li>';
                    });
                    errorMsg += '</ul>';
                }
                toastr.error(errorMsg, '', {
                    timeOut: 10000,
                    extendedTimeOut: 10000,
                    closeButton: true,
                    progressBar: true
                });
            }
        },
        error: function(xhr) {
            let errorMsg = 'Error saving pricing bands';
            if (xhr.responseJSON) {
                if (xhr.responseJSON.message) {
                    errorMsg = xhr.responseJSON.message;
                }
                if (xhr.responseJSON.validation_errors && Array.isArray(xhr.responseJSON.validation_errors)) {
                    errorMsg += '<br><ul style="text-align: left; margin-top: 10px;">';
                    xhr.responseJSON.validation_errors.forEach(function(err) {
                        errorMsg += '<li>' + err + '</li>';
                    });
                    errorMsg += '</ul>';
                }
            }
            toastr.error(errorMsg, '', {
                timeOut: 10000,
                extendedTimeOut: 10000,
                closeButton: true,
                progressBar: true
            });
        }
    });
}
</script>
@endsection

