@extends('layouts.horizontal', ['title' => 'Service Floors', 'topbarTitle' => 'Service Floors'])

@section('css')
@vite([
    'node_modules/datatables.net-bs5/css/dataTables.bootstrap5.min.css',
    'node_modules/datatables.net-responsive-bs5/css/responsive.bootstrap5.min.css',
    'node_modules/datatables.net-fixedcolumns-bs5/css/fixedColumns.bootstrap5.min.css',
    'node_modules/datatables.net-fixedheader-bs5/css/fixedHeader.bootstrap5.min.css',
    'node_modules/datatables.net-buttons-bs5/css/buttons.bootstrap5.min.css',
    'node_modules/datatables.net-select-bs5/css/select.bootstrap5.min.css',
    'node_modules/select2/dist/css/select2.min.css'
])
@endsection

@section('content')
<div class="container-fluid">
    <div class="row mb-4">
        <div class="col-12">
            <div class="page-title-box d-flex align-items-center justify-content-between">
                <h2 class="mb-0">Service Floors</h2>
                <div class="page-title-right d-flex align-items-center gap-3">
                    @php
                        $breadcrumb = getCurrentMenuBreadcrumb();
                    @endphp
                    @if(!empty($breadcrumb))
                        <nav aria-label="breadcrumb">
                            <ol class="breadcrumb mb-0 py-2">
                                @foreach($breadcrumb as $index => $item)
                                    @if($index === count($breadcrumb) - 1)
                                        <li class="breadcrumb-item active" aria-current="page">{{ $item }}</li>
                                    @else
                                        <li class="breadcrumb-item"><a href="javascript: void(0);">{{ $item }}</a></li>
                                    @endif
                                @endforeach
                            </ol>
                        </nav>
                    @endif
                </div>
            </div>
        </div>
    </div>

    <div class="row">
        <div class="col-12">
            <div class="card">
                <div class="card-body">
                    <table id="floors_table" class="table table-bordered dt-responsive fs-13 w-100">
                        <thead>
                            <tr>
                                <th>ID</th>
                                <th>Floor Name</th>
                                <th>Workstations</th>
                                <th>Floor Manager</th>
                                <th>Status</th>
                                <th class="noprint">Action</th>
                            </tr>
                        </thead>
                        <tbody>
                            @foreach($floors as $floor)
                                <tr>
                                    <td>{{ $floor->id_business_floors }}</td>
                                    <td>{{ $floor->floor_name }}</td>
                                    <td>{{ $floor->floor_workstations }}</td>
                                    <td>{{ $floor->floormanager_name }}</td>
                                    <td>
                                        @php($status = $floor->floor_status ?? 'Active')
                                        <span class="badge {{ $status === 'Active' ? 'bg-success' : 'bg-secondary' }}">
                                            {{ $status }}
                                        </span>
                                    </td>
                                    <td>
                                        <button
                                            onclick="openEditFloorModal({{ $floor->id_business_floors }})"
                                            class="btn btn-sm btn-ghost-info me-2 rounded-circle btn-icon edit-floor-btn"
                                            data-id="{{ $floor->id_business_floors }}"
                                            data-floor_name="{{ $floor->floor_name }}"
                                            data-floor_workstations="{{ $floor->floor_workstations }}"
                                            data-floormanager_id="{{ $floor->floormanager_id }}"
                                            data-floor_status="{{ $floor->floor_status }}"
                                        >
                                            <iconify-icon icon="line-md:edit-twotone" class="fs-22"></iconify-icon>
                                        </button>
                                    </td>
                                </tr>
                            @endforeach
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </div>
</div>

{{-- Add/Edit Modal --}}
<div class="modal fade" id="floorModal" tabindex="-1" aria-labelledby="floorModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-sm">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="floorModalLabel"></h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <form id="floorModalForm">
                @csrf
                <div class="modal-body">
                    <input type="hidden" id="id_business_floors" name="id_business_floors">

                    <div class="mb-3">
                        <label for="floor_name" class="form-label">Floor Name</label>
                        <input type="text" class="form-control" id="floor_name" name="floor_name" required>
                    </div>

                    <div class="mb-3">
                        <label for="floormanager_id" class="form-label">Floor Manager</label>
                        <select class="form-select" id="floormanager_id" name="floormanager_id">
                            <option value="">Select Staff</option>
                            @foreach($staffMembers as $s)
                                <option value="{{ $s->id_staff }}">{{ $s->staff_fullname }}</option>
                            @endforeach
                        </select>
                    </div>

                    <div class="mb-3">
                        <label for="floor_workstations" class="form-label">Workstations</label>
                        <input type="number" class="form-control" id="floor_workstations" name="floor_workstations" min="0" value="0" required>
                    </div>

                    <div class="mb-3">
                        <label for="floor_status" class="form-label">Status</label>
                        <select class="form-select" id="floor_status" name="floor_status" required>
                            <option value="Active">Active</option>
                            <option value="InActive">InActive</option>
                        </select>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                    <button type="button" onclick="save_floor()" class="btn btn-primary" id="saveFloorBtn">Save changes</button>
                </div>
            </form>
        </div>
    </div>
</div>
{{-- End Modal --}}
@endsection

@section('scripts')
<script>
    let floorsTable;

    $(document).ready(function() {
        initFloorsTable();
        $('#floormanager_id').select2({ width: '100%', dropdownParent: $('#floorModal') });
    });

    function initFloorsTable() {
        if ($.fn.DataTable.isDataTable('#floors_table')) {
            $('#floors_table').DataTable().clear().destroy();
        }

        floorsTable = $('#floors_table').DataTable({
            responsive: true,
            fixedHeader: true,
            lengthChange: true,
            pageLength: 10,
            searching: true,
            info: true,
            paging: true,
            buttons: [
                { extend: "copy",  className: "btn btn-sm btn-soft-primary" },
                { extend: "csv",   className: "btn btn-sm btn-soft-primary" },
                { extend: "print", className: "btn btn-sm btn-soft-primary" },
                { extend: "pdf",   className: "btn btn-sm btn-soft-primary" }
            ],
            layout: {
                topStart: "buttons",
                topEnd: function() {
                    // Return a flex container with search input and button
                    return '<div class="d-flex align-items-center gap-2" style="justify-content: flex-end; width: 100%;"><label class="dataTables_filter" style="margin: 0;"><input type="search" class="form-control form-control-sm" placeholder="Search..." aria-controls="floors_table" id="floors_table_search"></label><button class="btn btn-sm btn-outline-success" id="addFloorBtn" onclick="openAddFloorModal()"><iconify-icon icon="mdi:plus" class="me-1 fs-22"></iconify-icon> Add Floor</button></div>';
                },
                bottomStart: {
                    pageLength: { menu: [[10, 25, 50, 100, 1000], [10, 25, 50, 100, 1000]] },
                    info: {}
                },
                bottomEnd: "paging"
            },
            initComplete: function(settings) {
                // Manually wire up the search input to DataTables
                var searchInput = $('#floors_table_search');
                if (searchInput.length) {
                    searchInput.on('keyup', function() {
                        floorsTable.search(this.value).draw();
                    });
                }
            }
        });
    }

    function reloadFloorsTable() {
        $.ajax({
            url: window.location.href,
            type: 'GET',
            success: function(response) {
                const $newContent = $(response);
                const $newTableBody = $newContent.find('#floors_table tbody');
                const $newTable = $newContent.find('#floors_table');
                
                if ($newTableBody.length && $newTable.length) {
                    $('#floors_table tbody').html($newTableBody.html());
                    initFloorsTable();
                } else {
                    window.location.reload();
                }
            },
            error: function() {
                window.location.reload();
            }
        });
    }

    function openAddFloorModal() {
        $('#id_business_floors').val('');
        $('#floor_name').val('');
        $('#floor_workstations').val('0');
        $('#floormanager_id').val('').trigger('change');
        $('#floor_status').val('Active');

        $('#floorModalLabel').text('Add Floor');

        let el = document.getElementById('floorModal');
        let modal = bootstrap.Modal.getInstance(el);
        if (!modal) modal = new bootstrap.Modal(el);
        modal.show();
    }

    function openEditFloorModal(id) {
        let button = $('.edit-floor-btn[data-id="' + id + '"]');

        $('#id_business_floors').val(button.data('id'));
        $('#floor_name').val(button.data('floor_name'));
        $('#floor_workstations').val(button.data('floor_workstations'));
        $('#floormanager_id').val(button.data('floormanager_id')).trigger('change');
        $('#floor_status').val(button.data('floor_status') || 'Active');

        $('#floorModalLabel').text('Edit Floor');

        let el = document.getElementById('floorModal');
        let modal = bootstrap.Modal.getInstance(el);
        if (!modal) modal = new bootstrap.Modal(el);
        modal.show();
    }

    function save_floor() {
        if ($('#floor_name').val().trim() === '') {
            toastr.warning('Floor name is required.');
            return;
        }

        $('#saveFloorBtn').prop('disabled', true);

        $.ajax({
            url: '{{ route("service_floors.save_or_update") }}',
            method: 'POST',
            data: $('#floorModalForm').serialize(),
            success: function(res) {
                $('#saveFloorBtn').prop('disabled', false);
                if (res.success) {
                    toastr.success(res.message);
                    $('#floorModal').modal('hide');
                    reloadFloorsTable();
                } else {
                    toastr.error(res.message || 'Failed to save floor');
                }
            },
            error: function(xhr) {
                $('#saveFloorBtn').prop('disabled', false);
                toastr.error('Failed to save floor: ' + xhr.responseText);
            }
        });
    }
</script>
@endsection
