@extends('layouts.horizontal', ['title' => 'Staff Loan Applications', 'topbarTitle' => 'Staff Loan Applications'])

@section('css')
@vite(['node_modules/datatables.net-bs5/css/dataTables.bootstrap5.min.css',
'node_modules/datatables.net-responsive-bs5/css/responsive.bootstrap5.min.css',
'node_modules/datatables.net-fixedcolumns-bs5/css/fixedColumns.bootstrap5.min.css',
'node_modules/datatables.net-fixedheader-bs5/css/fixedHeader.bootstrap5.min.css',
'node_modules/datatables.net-buttons-bs5/css/buttons.bootstrap5.min.css',
'node_modules/datatables.net-select-bs5/css/select.bootstrap5.min.css',
'node_modules/dropzone/dist/dropzone.css',
'node_modules/select2/dist/css/select2.min.css',
'node_modules/flatpickr/dist/flatpickr.min.css'])
<style>
    .filter-row { background-color: #f9f9f9; }
    .form-filter { font-size: 12px; padding: 4px 6px; height: 30px; width: 100%; }
    .filter-row th { padding: 4px 6px !important; vertical-align: middle; }
    .dataTables_wrapper .dataTables_filter { float: right; }
    .dataTables_wrapper .dataTables_length { float: left; }
    .dataTables_wrapper .dataTables_paginate { float: right; }
    .filter-buttons { display: flex; gap: 5px; }
    .filter-buttons button { padding: 4px 8px; font-size: 12px; }
    .action-buttons { white-space: nowrap; }
    .amount-cell { text-align: right; }
    .status-badge {
        padding: 4px 8px;
        border-radius: 4px;
        font-size: 12px;
        font-weight: 500;
    }
    .status-pending { background-color: #fff3cd; color: #856404; }
    .status-approved { background-color: #d4edda; color: #155724; }
    .status-rejected { background-color: #f8d7da; color: #721c24; }
    .status-disbursed { background-color: #cce5ff; color: #004085; }

    .dataTables_scrollHead { background: #f8f9fa; }
    
    .btn-icon {
        width: 36px;
        height: 36px;
        display: inline-flex;
        align-items: center;
        justify-content: center;
        padding: 0;
    }
    
    .btn-rounded {
        border-radius: 50%;
    }
</style>
@endsection

@section('content')
<div class="container-fluid">
    <div class="row mb-4">
        <div class="col-12">
            <div class="page-title-box d-flex align-items-center justify-content-between">
                <h2 class="mb-0">Staff Loan Applications</h2>
                <div class="page-title-right d-flex align-items-center gap-3">
                    @php
                        $breadcrumb = getCurrentMenuBreadcrumb();
                    @endphp
                    @if(!empty($breadcrumb))
                        <nav aria-label="breadcrumb">
                            <ol class="breadcrumb mb-0 py-2">
                                @foreach($breadcrumb as $index => $item)
                                    @if($index === count($breadcrumb) - 1)
                                        <li class="breadcrumb-item active" aria-current="page">{{ $item }}</li>
                                    @else
                                        <li class="breadcrumb-item"><a href="javascript: void(0);">{{ $item }}</a></li>
                                    @endif
                                @endforeach
                            </ol>
                        </nav>
                    @endif
                </div>
            </div>
        </div>
    </div>
    <div id="alert-container"></div>
    @if(session('ho') === 'Yes')
        <div class="row mb-3">
            <div class="col-12">
                <div class="card">
                    <div class="card-body">
                        <div class="row align-items-center">
                            <div class="col-md-12">
                                <div class="d-flex align-items-center justify-content-between gap-3">
                                    <div class="flex-grow-1">
                                        <label class="form-label">Select Business</label>
                                        <select class="form-control select2" id="businessSelector">
                                            @foreach($businesses as $business)
                                                <option value="{{ $business->id_business }}" 
                                                    {{ session('selected_business_id') == $business->id_business ? 'selected' : '' }}>
                                                    {{ $business->business_name }}
                                                </option>
                                            @endforeach
                                        </select>
                                    </div>
                                    <div>
                                        <button type="button" class="btn btn-outline-primary" id="addLoanBtn">
                                            <iconify-icon icon="mdi:plus" class="me-1 fs-22"></iconify-icon> Create New Loan
                                        </button>
                                    </div>
                                </div>
                                <input type="hidden" name="filter_business_id" id="filter_business_id" value="{{ session('selected_business_id') ?? '' }}">
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    @else
        <div class="row mb-3">
            <div class="col-12">
                <div class="card">
                    <div class="card-body">
                        <div class="d-flex justify-content-end">
                            <button type="button" class="btn btn-outline-primary" id="addLoanBtn">
                                <iconify-icon icon="mdi:plus" class="me-1 fs-22"></iconify-icon> Create New Loan
                            </button>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    @endif

    <div class="row">
        <div class="col-12">
            <div class="card">
                <div class="card-body">
                    <table id="loan_applications_table" class="table table-bordered dt-responsive fs-13 w-100">
                        <thead>
                            <tr>
                                <th>ID</th>
                                <th>Staff ID</th>
                                <th>Staff Name</th>
                                <th>Application Date</th>
                                <th>Requested Amount</th>
                                <th>Approved Amount</th>
                                <th>Approved By</th>
                                <th>Approved On</th>
                                <th>Installment</th>
                                <th>No. of Months</th>
                                <th>Status</th>
                                <th>Actions</th>
                            </tr>
                            <tr class="filter filter-row">
                                <td>
                                    <input type="text" class="form-control form-filter input-sm" name="id_loan_applications" placeholder="Search ID">
                                </td>
                                <td>
                                    <input type="text" class="form-control form-filter input-sm" name="staff_id" placeholder="Staff ID">
                                </td>
                                <td>
                                    <input type="text" class="form-control form-filter input-sm" name="staff_name" placeholder="Staff Name">
                                </td>
                                <td>
                                    <input type="text" class="form-control form-filter input-sm date-picker" name="created_on" placeholder="Date">
                                </td>
                                <td>
                                    <input type="text" class="form-control form-filter input-sm" name="required_loan_amount" placeholder="Requested">
                                </td>
                                <td>
                                    <input type="text" class="form-control form-filter input-sm" name="approved_amount" placeholder="Approved">
                                </td>
                                <td>
                                    <input type="text" class="form-control form-filter input-sm" name="approved_by" placeholder="Approved By">
                                </td>
                                <td>
                                    <input type="text" class="form-control form-filter input-sm date-picker" name="approved_on" placeholder="Approved Date">
                                </td>
                                <td>
                                    <input type="text" class="form-control form-filter input-sm" name="installment" placeholder="Installment">
                                </td>
                                <td>
                                    <input type="text" class="form-control form-filter input-sm" name="num_of_months" placeholder="Months">
                                </td>
                                <td>
                                    <select name="application_status" class="form-control form-filter input-sm">
                                        <option value="">All</option>
                                        <option value="Pending">Pending</option>
                                        <option value="Approved">Approved</option>
                                        <option value="Rejected">Rejected</option>
                                        <option value="Disbursed">Disbursed</option>
                                    </select>
                                </td>
                                <td>
                                    <div class="filter-buttons">
                                        <button class="btn btn-sm btn-ghost-secondary rounded-circle btn-icon filter-submit" title="Search">
                                            <iconify-icon icon="mdi:magnify" class="fs-22"></iconify-icon>
                                        </button>
                                        <button class="btn btn-sm btn-ghost-secondary rounded-circle btn-icon filter-cancel" title="Reset">
                                            <iconify-icon icon="mdi:refresh" class="fs-22"></iconify-icon>
                                        </button>
                                    </div>
                                </td>
                            </tr>
                        </thead>
                        <tbody></tbody>
                    </table>
                </div>
            </div>
        </div>
    </div>
</div>

<div class="modal fade" id="addLoanModal" tabindex="-1" aria-labelledby="addLoanModalLabel" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="addLoanModalLabel">Create New Loan Application</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
                <form id="addLoanForm">
                    @csrf
                    <input type="hidden" name="business_id" id="loan_business_id" value="{{ session('selected_business_id') ?? '' }}">
                    
                    <div class="row">
                        @if(session('ho') === 'Yes')
                            <div class="col-12 mb-3">
                                <label for="modal_business_id" class="form-label">Business <span class="text-danger">*</span></label>
                                <select class="form-control select2" id="modal_business_id" name="modal_business_id" required>
                                    <option value="">Select Business</option>
                                    @foreach($businesses as $business)
                                        <option value="{{ $business->id_business }}"
                                            {{ (session('selected_business_id') ?? '') == $business->id_business ? 'selected' : '' }}>
                                            {{ $business->business_name }}
                                        </option>
                                    @endforeach
                                </select>
                            </div>
                        @endif
                        <div class="col-12 mb-3">
                            <label for="staff_id" class="form-label">Staff Name <span class="text-danger">*</span></label>
                            <select class="form-control select2" id="staff_id" name="staff_id" required>
                                <option value="">Select Staff</option>
                                @foreach($staffList as $staff)
                                    <option value="{{ $staff->id_staff }}">{{ $staff->staff_fullname }}</option>
                                @endforeach
                            </select>
                            <input type="hidden" name="staff_name" id="staff_name">
                        </div>
                        
                        <div class="col-12 mb-3">
                            <label for="required_loan_amount" class="form-label">Loan Amount <span class="text-danger">*</span></label>
                            <input type="number" step="0.01" min="1" class="form-control" id="required_loan_amount" name="required_loan_amount" required>
                        </div>
                        
                        <div class="col-12 mb-3">
                            <label for="num_of_months" class="form-label">Number of Months <span class="text-danger">*</span></label>
                            <input type="number" min="1" max="36" class="form-control" id="num_of_months" name="num_of_months" required>
                        </div>
                        
                        <div class="col-12 mb-3">
                            <label for="comments" class="form-label">Comments</label>
                            <textarea class="form-control" id="comments" name="comments" rows="3"></textarea>
                        </div>
                    </div>
                </form>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">
                    <iconify-icon icon="mdi:close" class="me-1 fs-22"></iconify-icon> Cancel
                </button>
                <button type="button" class="btn btn-primary" id="saveLoanBtn">
                    <iconify-icon icon="mdi:content-save" class="me-1 fs-22"></iconify-icon> Save Application
                </button>
            </div>
        </div>
    </div>
</div>

<div class="modal fade" id="editLoanModal" tabindex="-1" aria-labelledby="editLoanModalLabel" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="editLoanModalLabel">Update Loan Application</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
                <form id="editLoanForm">
                    @csrf
                    <input type="hidden" name="id_loan_applications" id="edit_id_loan_applications">
                    <input type="hidden" name="staff_id" id="edit_staff_id">
                    
                    <div class="row">
                        <div class="col-12 mb-3">
                            <label class="form-label">Staff Name</label>
                            <input type="text" class="form-control" id="edit_staff_name" readonly>
                        </div>
                        
                        <div class="col-12 mb-3">
                            <label class="form-label">Requested Amount</label>
                            <input type="text" class="form-control" id="edit_requested_amount" readonly>
                            <small class="text-muted">This is the amount requested by the staff</small>
                        </div>
                        
                        <div class="col-12 mb-3">
                            <label for="edit_approved_loan_amount" class="form-label">Approved Amount <span class="text-danger">*</span></label>
                            <input type="number" step="0.01" min="1" class="form-control" id="edit_approved_loan_amount" name="approved_loan_amount" required>
                        </div>
                        
                        <div class="col-12 mb-3">
                            <label for="edit_num_of_months" class="form-label">Number of Months <span class="text-danger">*</span></label>
                            <input type="number" min="1" max="36" class="form-control" id="edit_num_of_months" name="num_of_months" required>
                        </div>
                        
                        <div class="col-12 mb-3">
                            <label for="edit_comment" class="form-label">Comments</label>
                            <textarea class="form-control" id="edit_comment" name="comment" rows="3"></textarea>
                        </div>
                        
                        <div class="col-12 mb-3">
                            <label for="edit_application_status" class="form-label">Status <span class="text-danger">*</span></label>
                            <select class="form-control" id="edit_application_status" name="application_status" required>
                                <option value="Pending">Pending</option>
                                <option value="Approved">Approved</option>
                                <option value="Rejected">Rejected</option>
                                <option value="Disbursed">Disbursed</option>
                            </select>
                        </div>
                    </div>
                </form>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">
                    <iconify-icon icon="mdi:close" class="me-1 fs-22"></iconify-icon> Cancel
                </button>
                <button type="button" class="btn btn-primary" id="updateLoanBtn">
                    <iconify-icon icon="mdi:content-save" class="me-1 fs-22"></iconify-icon> Update Application
                </button>
            </div>
        </div>
    </div>
</div>

<div class="modal fade" id="viewLoanModal" tabindex="-1" aria-labelledby="viewLoanModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="viewLoanModalLabel">Loan Application Details</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
                <div class="row">
                    <div class="col-md-6">
                        <p><strong>Application ID:</strong> <span id="view_id"></span></p>
                        <p><strong>Staff Name:</strong> <span id="view_staff_name"></span></p>
                        <p><strong>Staff ID:</strong> <span id="view_staff_id"></span></p>
                        <p><strong>Requested Amount:</strong> <span id="view_requested_amount"></span></p>
                        <p><strong>Approved Amount:</strong> <span id="view_approved_amount"></span></p>
                    </div>
                    <div class="col-md-6">
                        <p><strong>Status:</strong> <span id="view_status" class="status-badge"></span></p>
                        <p><strong>Created By:</strong> <span id="view_created_by"></span></p>
                        <p><strong>Created On:</strong> <span id="view_created_on"></span></p>
                        <p><strong>Approved By:</strong> <span id="view_approved_by"></span></p>
                        <p><strong>Approved On:</strong> <span id="view_approved_on"></span></p>
                    </div>
                </div>
                <div class="row mt-3">
                    <div class="col-md-6">
                        <p><strong>Number of Months:</strong> <span id="view_num_of_months"></span></p>
                        <p><strong>Monthly Installment:</strong> <span id="view_installment"></span></p>
                        <p><strong>Voucher ID:</strong> <span id="view_voucher_id"></span></p>
                    </div>
                    <div class="col-md-6">
                        <p><strong>Comments:</strong> <span id="view_comments"></span></p>
                        <div id="voucher-link-container"></div>
                    </div>
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">
                    <iconify-icon icon="mdi:close" class="me-1 fs-22"></iconify-icon> Close
                </button>
            </div>
        </div>
    </div>
</div>
@endsection

@section('scripts')
@vite(['resources/js/app.js'])
<script>
    document.addEventListener('DOMContentLoaded', function() {
        initializeDatePickers();
        initializeLoanApplicationsTable();
        setupEventHandlers();
        initializeSelect2();
        setupBusinessChangeHandlers();
    });

    let loanApplicationsTable = null;
    let addLoanModal = null;
    let editLoanModal = null;
    let viewLoanModal = null;
    
    function initializeDatePickers() {
        if (typeof flatpickr !== 'undefined') {
            flatpickr('.date-picker', {
                dateFormat: 'Y-m-d',
                allowInput: true
            });
        }
    }
    
    function initializeSelect2() {
        $('#businessSelector').select2({
            width: '100%'
        });
        
        $('#modal_business_id').select2({
            dropdownParent: $('#addLoanModal'),
            width: '100%'
        });
        
        $('#staff_id').select2({
            dropdownParent: $('#addLoanModal'),
            width: '100%'
        });
        
        $('#staff_id').on('change', function() {
            var staffName = $(this).find('option:selected').text();
            $('#staff_name').val(staffName);
        });
    }

    function setupBusinessChangeHandlers() {
        $('#businessSelector').on('change', function() {
            const businessId = $(this).val();
            $('#filter_business_id').val(businessId);
            
            $('#modal_business_id').val(businessId).trigger('change');
            
            $('#loan_business_id').val(businessId);
            
            if (businessId) {
                loadStaffForBusiness(businessId);
            } else {
                clearStaffDropdown();
            }
            
            if (loanApplicationsTable) {
                loanApplicationsTable.ajax.reload();
            }
        });

        $('#modal_business_id').on('change', function() {
            const businessId = $(this).val();
            console.log('Modal business changed to:', businessId);
            $('#loan_business_id').val(businessId);
            if (businessId) {
                loadStaffForBusiness(businessId);
            } else {
                clearStaffDropdown();
            }
        });
    }

    function loadStaffForBusiness(businessId) {
        if (!businessId) return;
        
        console.log('Loading staff for business:', businessId);
        
        $.ajax({
            url: '{{ route("hrm.get_staff_by_business") }}',
            method: 'GET',
            data: {
                business_id: businessId
            },
            success: function(response) {
                if (response.success) {
                    const staffSelect = $('#staff_id');
                    staffSelect.empty();
                    staffSelect.append('<option value="">Select Staff</option>');
                    
                    response.staff.forEach(function(staff) {
                        staffSelect.append(
                            $('<option></option>')
                                .val(staff.id_staff)
                                .text(staff.staff_fullname)
                        );
                    });
                    staffSelect.select2({
                        dropdownParent: $('#addLoanModal'),
                        width: '100%'
                    });
                    
                    console.log('Staff loaded successfully for business:', businessId);
                } else {
                    showAlert('error', response.message || 'Failed to load staff.');
                }
            },
            error: function(xhr, status, error) {
                console.error('Error loading staff:', error);
                showAlert('error', 'Error loading staff list.');
            }
        });
    }

    function clearStaffDropdown() {
        const staffSelect = $('#staff_id');
        staffSelect.empty();
        staffSelect.append('<option value="">Select Staff</option>');
        staffSelect.select2({
            dropdownParent: $('#addLoanModal'),
            width: '100%'
        });
    }

    function initializeLoanApplicationsTable() {
        if (typeof $.fn.DataTable === 'undefined') {
            console.error('DataTables is not loaded');
            return;
        }
        
        $('select[name="application_status"]').val('Pending');
        
        loanApplicationsTable = $('#loan_applications_table').DataTable({
            scrollX: true, 
            orderCellsTop: true,           
            scrollCollapse: true,
            fixedHeader: {
                header: true,
                footer: false
            },
            processing: true,
            serverSide: true,
            stateSave: true,
            stateDuration: -1,
            stateLoadParams: function(settings, data) {
                if (!data.columns[10] || !data.columns[10].search || data.columns[10].search.search === '') {
                    $('select[name="application_status"]').val('Pending');
                }
            },
            responsive: false,
            ajax: {
                url: "{{ route('hrm.get_loan_applications') }}",
                type: 'POST',
                contentType: 'application/json',
                headers: {
                    'X-CSRF-TOKEN': '{{ csrf_token() }}'
                },
                dataType: 'json',
                data: function (d) {
                    d.id_loan_applications = $('input[name="id_loan_applications"]').val();
                    d.staff_id = $('input[name="staff_id"]').val();
                    d.staff_name = $('input[name="staff_name"]').val();
                    d.created_on = $('input[name="created_on"]').val();
                    d.required_loan_amount = $('input[name="required_loan_amount"]').val();
                    d.approved_amount = $('input[name="approved_amount"]').val();
                    d.approved_by = $('input[name="approved_by"]').val();
                    d.approved_on = $('input[name="approved_on"]').val();
                    d.installment = $('input[name="installment"]').val();
                    d.num_of_months = $('input[name="num_of_months"]').val();
                    d.application_status = $('select[name="application_status"]').val();

                    @if(session('ho') === 'Yes')
                        d.business_id = $('#filter_business_id').val();
                    @endif
                    
                    console.log('DataTable request business_id:', d.business_id);
                    return JSON.stringify(d);
                },
                error: function (xhr, error, thrown) {
                    console.error('DataTables AJAX error:', error, thrown, xhr.responseText);
                    showAlert('error', 'Failed to load loan applications. Please try again.');
                }
            },
            "language": {
                "paginate": {
                    "previous": "<i class='ti ti-chevron-left'></i>",
                    "next": "<i class='ti ti-chevron-right'></i>"
                }
            },
            lengthChange: true,
            pageLength: 10,
            searching: false,
            info: true,
            paging: true,
            buttons: [
                { 
                    extend: "copy",  
                    className: "btn btn-sm btn-ghost-secondary rounded-circle btn-icon",
                    text: '<iconify-icon icon="mdi:content-copy" class="fs-22"></iconify-icon>'
                },
                { 
                    extend: "csv",   
                    className: "btn btn-sm btn-ghost-secondary rounded-circle btn-icon",
                    text: '<iconify-icon icon="mdi:file-delimited" class="fs-22"></iconify-icon>'
                },
                { 
                    extend: "print", 
                    className: "btn btn-sm btn-ghost-secondary rounded-circle btn-icon",
                    text: '<iconify-icon icon="mdi:printer" class="fs-22"></iconify-icon>'
                },
                { 
                    extend: "pdf", 
                    className: "btn btn-sm btn-ghost-secondary rounded-circle btn-icon",
                    text: '<iconify-icon icon="mdi:file-pdf" class="fs-22"></iconify-icon>'
                }
            ],
            columns: [
                { data: 'id_loan_applications', name: 'id_loan_applications', orderable: true, searchable: true },
                { data: 'staff_id', name: 'staff_id', orderable: true, searchable: true },
                { data: 'staff_name', name: 'staff_name', orderable: true, searchable: true },
                { data: 'created_on', name: 'created_on', orderable: true, searchable: true },
                { data: 'required_loan_amount', name: 'required_loan_amount', orderable: true, searchable: true, className: 'amount-cell' },
                { data: 'approved_amount', name: 'approved_amount', orderable: true, searchable: true, className: 'amount-cell' },
                { data: 'approved_by', name: 'approved_by', orderable: true, searchable: true },
                { data: 'approved_on', name: 'approved_on', orderable: true, searchable: true },
                { data: 'installment', name: 'installment', orderable: true, searchable: true, className: 'amount-cell' },
                { data: 'num_of_months', name: 'num_of_months', orderable: true, searchable: true },
                { 
                    data: 'application_status', 
                    name: 'application_status', 
                    orderable: true, 
                    searchable: true,
                    render: function(data, type, row) {
                        let className = '';
                        switch(data) {
                            case 'Pending': className = 'status-pending'; break;
                            case 'Approved': className = 'status-approved'; break;
                            case 'Rejected': className = 'status-rejected'; break;
                            case 'Disbursed': className = 'status-disbursed'; break;
                        }
                        return `<span class="status-badge ${className}">${data}</span>`;
                    }
                },
                { 
                    data: null, 
                    name: 'actions', 
                    orderable: false, 
                    searchable: false,
                    className: 'action-buttons',
                    render: function(data, type, row) {
                        let buttons = '';
                        if (row.application_status === 'Pending') {
                            buttons += `
                                <button class="btn btn-sm btn-ghost-secondary rounded-circle btn-icon" 
                                        onclick="openLoanApp(${row.id_loan_applications})"
                                        title="Edit Application">
                                    <iconify-icon icon="line-md:edit-twotone" class="fs-22"></iconify-icon>
                                </button>
                            `;
                        }
                        
                        buttons += `
                            <button class="btn btn-sm btn-ghost-secondary rounded-circle btn-icon" 
                                    onclick="viewLoanDetails(${row.id_loan_applications})"
                                    title="View Details">
                                <iconify-icon icon="mdi:eye-outline" class="fs-22"></iconify-icon>
                            </button>
                        `;
                        
                        return buttons;
                    }
                }
            ],
            initComplete: function(settings, json) {
                var api = this.api();
                var filterRow = $(api.table().header()).find('tr.filter');
                filterRow.show();
            }
        });
    }
    
    function setupEventHandlers() {
        const addLoanModalEl = document.getElementById('addLoanModal');
        const editLoanModalEl = document.getElementById('editLoanModal');
        const viewLoanModalEl = document.getElementById('viewLoanModal');
        
        if (addLoanModalEl) addLoanModal = new bootstrap.Modal(addLoanModalEl);
        if (editLoanModalEl) editLoanModal = new bootstrap.Modal(editLoanModalEl);
        if (viewLoanModalEl) viewLoanModal = new bootstrap.Modal(viewLoanModalEl);
    
        const addLoanBtn = document.getElementById('addLoanBtn');
        if (addLoanBtn) {
            addLoanBtn.addEventListener('click', function() {
                openAddLoanModal();
            });
        }
        
        
        const saveLoanBtn = document.getElementById('saveLoanBtn');
        if (saveLoanBtn) {
            saveLoanBtn.addEventListener('click', function() {
                saveLoanApplication();
            });
        }
        
       
        const updateLoanBtn = document.getElementById('updateLoanBtn');
        if (updateLoanBtn) {
            updateLoanBtn.addEventListener('click', function() {
                updateLoanApplication();
            });
        }
        
        document.addEventListener('click', function(e) {
            if (e.target.closest('.filter-submit')) {
                e.preventDefault();
                refreshData();
            }
            
            if (e.target.closest('.filter-cancel')) {
                e.preventDefault();
                document.querySelectorAll('.form-filter').forEach(input => {
                    input.value = '';
                });
                document.querySelectorAll('select.form-filter').forEach(select => {
                    select.value = '';
                });
                
                $('select[name="application_status"]').val('Pending');
                
                refreshData();
            }
        });
        
        document.addEventListener('keyup', function(e) {
            if (e.target.classList.contains('form-filter') && e.key === 'Enter') {
                refreshData();
            }
        });
    }

    function refreshData() {
        if (loanApplicationsTable) {
            loanApplicationsTable.ajax.reload(null, false);
        }
    }

    function openAddLoanModal() {
        resetAddForm();
        
        @if(session('ho') === 'Yes')
            const selectedBusinessId = $('#businessSelector').val();

            if (selectedBusinessId) {
                $('#modal_business_id').val(selectedBusinessId).trigger('change');
                $('#loan_business_id').val(selectedBusinessId);
                loadStaffForBusiness(selectedBusinessId);
            } else {
                $('#modal_business_id').val('').trigger('change');
                $('#loan_business_id').val('');
                clearStaffDropdown();
            }
        @endif
        
        if (addLoanModal) {
            addLoanModal.show();
        }
    }

    function resetAddForm() {
        const form = document.getElementById('addLoanForm');
        if (form) {
            $('#staff_id').val('').trigger('change');
            $('#required_loan_amount').val('');
            $('#num_of_months').val('');
            $('#comments').val('');
            $('#staff_name').val('');
        }
        $('.is-invalid').removeClass('is-invalid');
        $('.invalid-feedback').remove();
    }

    function saveLoanApplication() {
        const form = document.getElementById('addLoanForm');
        const formData = new FormData(form);
        
        for (let [key, value] of formData.entries()) {
            console.log(key + ': ' + value);
        }
       
        @if(session('ho') === 'Yes')
            const businessId = $('#loan_business_id').val();
            const modalBusinessId = $('#modal_business_id').val();
            
            if (!businessId || businessId === '') {
                Swal.fire({
                    icon: 'warning',
                    title: 'Business Required',
                    text: 'Please select a business before creating a loan application.',
                    customClass: {
                        confirmButton: "btn btn-danger me-2",
                        cancelButton: "btn btn-warning"
                    }
                });
                return;
            }
        @endif
        const staffId = $('#staff_id').val();
        const amount = $('#required_loan_amount').val();
        const months = $('#num_of_months').val();
    
        if (!staffId || !amount || !months) {
            Swal.fire({
                icon: 'warning',
                title: 'Validation Error',
                text: 'Please fill in all required fields.',
                customClass: {
                    confirmButton: "btn btn-danger me-2",
                    cancelButton: "btn btn-warning"
                }
            });
            return;
        }
        
        Swal.fire({
            title: 'Saving...',
            allowOutsideClick: false,
            customClass: {
                confirmButton: "btn btn-danger me-2",
                cancelButton: "btn btn-warning"
            },
            didOpen: () => Swal.showLoading()
        });
        
        fetch("{{ route('hrm.insert_loan_application') }}", {
            method: 'POST',
            body: formData,
            headers: {
                'X-CSRF-TOKEN': '{{ csrf_token() }}'
            }
        })
        .then(response => response.json())
        .then(response => {
            Swal.close();
            
            if (response.success) {
                if (addLoanModal) {
                    addLoanModal.hide();
                }
                resetAddForm();
                Swal.fire({
                    icon: 'success',
                    title: 'Success!',
                    text: response.message,
                    timer: 2000,
                    showConfirmButton: false,
                    customClass: {
                        confirmButton: "btn btn-danger me-2",
                        cancelButton: "btn btn-warning"
                    }
                }).then(() => {
                    refreshData();
                });
                
            } else {
                Swal.fire({
                    icon: 'error',
                    title: 'Error',
                    text: response.message,
                    customClass: {
                        confirmButton: "btn btn-danger me-2",
                        cancelButton: "btn btn-warning"
                    }
                });
            }
        })
        .catch(error => {
            Swal.close();
            console.error('Error:', error);
            Swal.fire({
                icon: 'error',
                title: 'Error',
                text: 'Error saving loan application. Please try again.',
                customClass: {
                    confirmButton: "btn btn-danger me-2",
                    cancelButton: "btn btn-warning"
                }
            });
        });
    }

    function openLoanApp(id) {
        Swal.fire({
            title: 'Loading...',
            allowOutsideClick: false,
            customClass: {
                confirmButton: "btn btn-danger me-2",
                cancelButton: "btn btn-warning"
            },
            didOpen: () => Swal.showLoading()
        });
        
        fetch("{{ route('hrm.get_loan_application', '') }}/" + id, {
            method: 'GET',
            headers: {
                'Accept': 'application/json',
            }
        })
        .then(response => response.json())
        .then(response => {
            Swal.close();
            
            if (response.success) {
                document.getElementById('edit_id_loan_applications').value = response.data.id_loan_applications;
                document.getElementById('edit_staff_id').value = response.data.staff_id;
                document.getElementById('edit_staff_name').value = response.data.staff_name;
                document.getElementById('edit_requested_amount').value = response.data.required_loan_amount;
                document.getElementById('edit_approved_loan_amount').value = response.data.approved_amount || response.data.required_loan_amount;
                document.getElementById('edit_num_of_months').value = response.data.num_of_months;
                document.getElementById('edit_comment').value = response.data.comments || '';
                document.getElementById('edit_application_status').value = response.data.application_status;
                
            
                if (editLoanModal) {
                    editLoanModal.show();
                }
            } else {
                Swal.fire({
                    icon: 'error',
                    title: 'Error',
                    text: response.message || 'Failed to load loan application.',
                    customClass: {
                        confirmButton: "btn btn-danger me-2",
                        cancelButton: "btn btn-warning"
                    }
                });
            }
        })
        .catch(error => {
            Swal.close();
            console.error('Error:', error);
            Swal.fire({
                icon: 'error',
                title: 'Error',
                text: 'Error loading loan application. Please try again.',
                customClass: {
                    confirmButton: "btn btn-danger me-2",
                    cancelButton: "btn btn-warning"
                }
            });
        });
    }

    function viewLoanDetails(id) {
        Swal.fire({
            title: 'Loading...',
            allowOutsideClick: false,
            customClass: {
                confirmButton: "btn btn-danger me-2",
                cancelButton: "btn btn-warning"
            },
            didOpen: () => Swal.showLoading()
        });
        
        fetch("{{ route('hrm.get_loan_application', '') }}/" + id, {
            method: 'GET',
            headers: {
                'Accept': 'application/json',
            }
        })
        .then(response => response.json())
        .then(response => {
            Swal.close();
            
            if (response.success) {
                const data = response.data;
                
                document.getElementById('view_id').textContent = data.id_loan_applications;
                document.getElementById('view_staff_name').textContent = data.staff_name;
                document.getElementById('view_staff_id').textContent = data.staff_id;
                document.getElementById('view_requested_amount').textContent = 'Rs. ' + (parseFloat(data.required_loan_amount) || 0).toFixed(2);
                document.getElementById('view_approved_amount').textContent = 'Rs. ' + (parseFloat(data.approved_amount) || 0).toFixed(2);
                document.getElementById('view_created_by').textContent = data.created_by || '-';
                document.getElementById('view_created_on').textContent = data.created_on ? new Date(data.created_on).toLocaleDateString('en-GB') : '-';
                document.getElementById('view_approved_by').textContent = data.approved_by || '-';
                document.getElementById('view_approved_on').textContent = data.approved_on ? new Date(data.approved_on).toLocaleDateString('en-GB') : '-';
                document.getElementById('view_num_of_months').textContent = data.num_of_months;
                document.getElementById('view_installment').textContent = 'Rs. ' + (parseFloat(data.installment) || 0).toFixed(2);
                document.getElementById('view_voucher_id').textContent = data.voucher_id || '-';
                document.getElementById('view_comments').textContent = data.comments || '-';
                
                const statusEl = document.getElementById('view_status');
                statusEl.textContent = data.application_status;
                statusEl.className = 'status-badge ';
                switch(data.application_status) {
                    case 'Pending': statusEl.classList.add('status-pending'); break;
                    case 'Approved': statusEl.classList.add('status-approved'); break;
                    case 'Rejected': statusEl.classList.add('status-rejected'); break;
                    case 'Disbursed': statusEl.classList.add('status-disbursed'); break;
                }
            
                const voucherContainer = document.getElementById('voucher-link-container');
                if (voucherContainer && data.voucher_id) {
                    voucherContainer.innerHTML = `
                        <a href="/accounts/account_voucher_print?id=${data.voucher_id}" target="_blank" class="btn btn-sm btn-success">
                            <iconify-icon icon="mdi:receipt-text" class="me-1 fs-22"></iconify-icon> View Voucher
                        </a>
                    `;
                } else if (voucherContainer) {
                    voucherContainer.innerHTML = '';
                }

                if (viewLoanModal) {
                    viewLoanModal.show();
                }
            } else {
                Swal.fire({
                    icon: 'error',
                    title: 'Error',
                    text: response.message || 'Failed to load loan details.',
                    customClass: {
                        confirmButton: "btn btn-danger me-2",
                        cancelButton: "btn btn-warning"
                    }
                });
            }
        })
        .catch(error => {
            Swal.close();
            console.error('Error:', error);
            Swal.fire({
                icon: 'error',
                title: 'Error',
                text: 'Error loading loan details. Please try again.',
                customClass: {
                    confirmButton: "btn btn-danger me-2",
                    cancelButton: "btn btn-warning"
                }
            });
        });
    }

    function updateLoanApplication() {
        const form = document.getElementById('editLoanForm');
        const formData = new FormData(form);
        
        const amount = document.getElementById('edit_approved_loan_amount');
        const months = document.getElementById('edit_num_of_months');
        const status = document.getElementById('edit_application_status');
        
        if (!amount.value || !months.value || !status.value) {
            Swal.fire({
                icon: 'warning',
                title: 'Validation Error',
                text: 'Please fill in all required fields.',
                customClass: {
                    confirmButton: "btn btn-danger me-2",
                    cancelButton: "btn btn-warning"
                }
            });
            return;
        }
        
        Swal.fire({
            title: 'Updating...',
            allowOutsideClick: false,
            customClass: {
                confirmButton: "btn btn-danger me-2",
                cancelButton: "btn btn-warning"
            },
            didOpen: () => Swal.showLoading()
        });
        
        fetch("{{ route('hrm.update_loan_application') }}", {
            method: 'POST',
            body: formData,
            headers: {
                'X-CSRF-TOKEN': '{{ csrf_token() }}'
            }
        })
        .then(response => response.json())
        .then(response => {
            Swal.close();
            
            if (response.success) {
                if (editLoanModal) {
                    editLoanModal.hide();
                }
                
                Swal.fire({
                    icon: 'success',
                    title: 'Success!',
                    text: response.message,
                    timer: 2000,
                    showConfirmButton: false,
                    customClass: {
                        confirmButton: "btn btn-danger me-2",
                        cancelButton: "btn btn-warning"
                    }
                }).then(() => {
                    refreshData();
                });
                
            } else {
                Swal.fire({
                    icon: 'error',
                    title: 'Error',
                    text: response.message,
                    customClass: {
                        confirmButton: "btn btn-danger me-2",
                        cancelButton: "btn btn-warning"
                    }
                });
            }
        })
        .catch(error => {
            Swal.close();
            console.error('Error:', error);
            Swal.fire({
                icon: 'error',
                title: 'Error',
                text: 'Error updating loan application. Please try again.',
                customClass: {
                    confirmButton: "btn btn-danger me-2",
                    cancelButton: "btn btn-warning"
                }
            });
        });
    }

    function showAlert(type, message) {
        const alertHtml = `
            <div class="row">
                <div class="col-12">
                    <div class="alert alert-${type} alert-dismissible fade show" role="alert">
                        <strong>${type === 'success' ? 'Success!' : 'Error!'}</strong> ${message}
                        <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                    </div>
                </div>
            </div>
        `;
        
        const container = document.getElementById('alert-container');
        if (container) {
            container.innerHTML = alertHtml;
            
            setTimeout(() => {
                container.innerHTML = '';
            }, 5000);
        }
    }

    document.addEventListener('DOMContentLoaded', function() {
        const addLoanModalEl = document.getElementById('addLoanModal');
        const editLoanModalEl = document.getElementById('editLoanModal');
        const viewLoanModalEl = document.getElementById('viewLoanModal');
        
        if (addLoanModalEl) {
            addLoanModalEl.addEventListener('hidden.bs.modal', function () {
                resetAddForm();
            });
        }
        if (viewLoanModalEl) {
            viewLoanModalEl.addEventListener('hidden.bs.modal', function () {
                const voucherContainer = document.getElementById('voucher-link-container');
                if (voucherContainer) {
                    voucherContainer.innerHTML = '';
                }
            });
        }
    });
    window.openLoanApp = openLoanApp;
    window.viewLoanDetails = viewLoanDetails;
    window.loadStaffForBusiness = loadStaffForBusiness;
    window.clearStaffDropdown = clearStaffDropdown;
</script>
@endsection