@extends('layouts.horizontal', ['title' => 'Staff Allowances', 'topbarTitle' => 'Staff Allowances'])

@section('css')
@vite(['node_modules/datatables.net-bs5/css/dataTables.bootstrap5.min.css',
'node_modules/datatables.net-responsive-bs5/css/responsive.bootstrap5.min.css',
'node_modules/datatables.net-fixedcolumns-bs5/css/fixedColumns.bootstrap5.min.css',
'node_modules/datatables.net-fixedheader-bs5/css/fixedHeader.bootstrap5.min.css',
'node_modules/datatables.net-buttons-bs5/css/buttons.bootstrap5.min.css',
'node_modules/datatables.net-select-bs5/css/select.bootstrap5.min.css',
'node_modules/dropzone/dist/dropzone.css',
'node_modules/select2/dist/css/select2.min.css',
'node_modules/flatpickr/dist/flatpickr.min.css'])
<style>
    .filter-row { background-color: #f9f9f9; }
    .form-filter { font-size: 12px; padding: 4px 6px; height: 30px; width: 100%; }
    .filter-row th { padding: 4px 6px !important; vertical-align: middle; }
    .dataTables_wrapper .dataTables_filter { float: right; }
    .dataTables_wrapper .dataTables_length { float: left; }
    .dataTables_wrapper .dataTables_paginate { float: right; }
    .filter-buttons { display: flex; gap: 5px; }
    .filter-buttons button { padding: 4px 8px; font-size: 12px; }
    .action-buttons { white-space: nowrap; }
    .entity-select { width: 100%; }
    .staff-count { 
        display: inline-block;
        background-color: #e9ecef;
        color: #495057;
        padding: 2px 8px;
        border-radius: 12px;
        font-size: 12px;
        font-weight: 600;
    }
    .business-selector-section {
        margin-bottom: 20px;
    }
    
</style>
@endsection

@section('content')
<div class="container-fluid">
    <div class="row mb-4">
        <div class="col-12">
            <div class="page-title-box d-flex align-items-center justify-content-between">
                <h2 class="mb-0">Staff Allowances</h2>
                <div class="page-title-right d-flex align-items-center gap-3">
                    @php
                        $breadcrumb = getCurrentMenuBreadcrumb();
                    @endphp
                    @if(!empty($breadcrumb))
                        <nav aria-label="breadcrumb">
                            <ol class="breadcrumb mb-0 py-2">
                                @foreach($breadcrumb as $index => $item)
                                    @if($index === count($breadcrumb) - 1)
                                        <li class="breadcrumb-item active" aria-current="page">{{ $item }}</li>
                                    @else
                                        <li class="breadcrumb-item"><a href="javascript: void(0);">{{ $item }}</a></li>
                                    @endif
                                @endforeach
                            </ol>
                        </nav>
                    @endif
                </div>
            </div>
        </div>
    </div>
    <!-- end page title -->

    <!-- Success/Error Messages -->
    <div id="alert-container"></div>
 
    <div class="row">
        <div class="col-12">
            <div class="card">
                <div class="card-body">
                    <table id="allowances_table" class="table table-bordered dt-responsive fs-13 w-100">
                        <thead>
                            <tr>
                                <th>ID</th>
                                <th>Allowance Name</th>
                                <th>Entity Name</th>
                                <th>Business</th>
                                <th>Created By</th>
                                <th>Created At</th>
                                <th>Actions</th>
                            </tr>
                            <!-- Filter Row -->
                            <tr class="filter filter-row">
                                <td>
                                    <input type="text" class="form-control form-filter input-sm" name="id_allowances" placeholder="Search ID">
                                </td>
                                <td>
                                    <input type="text" class="form-control form-filter input-sm" name="allowance_name" placeholder="Search Allowance Name">
                                </td>
                                <td>
                                    <input type="text" class="form-control form-filter input-sm" name="entity_name" placeholder="Search Entity Name">
                                </td>
                                <td>
                                    <input type="text" class="form-control form-filter input-sm" name="business_name" placeholder="Search Business">
                                </td>
                                <td>
                                    <input type="text" class="form-control form-filter input-sm" name="created_by" placeholder="Search Created By">
                                </td>
                                <td>
                                    <input type="text" class="form-control form-filter input-sm date-picker" name="created_at" placeholder="Search Date">
                                </td>
                                <td>
                                    <div class="filter-buttons">
                                        <button class="btn btn-sm btn-ghost-secondary rounded-circle btn-icon filter-submit">
                                            <iconify-icon icon="mdi:magnify" class="fs-22"></iconify-icon>
                                        </button>
                                        <button class="btn btn-sm btn-ghost-secondary rounded-circle btn-icon filter-cancel">
                                            <iconify-icon icon="mdi:refresh" class="fs-22"></iconify-icon>
                                        </button>
                                    </div>
                                </td>
                            </tr>
                        </thead>
                        <tbody></tbody>
                    </table>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Add/Edit Allowance Modal -->
<div class="modal fade" id="allowanceModal" tabindex="-1" aria-labelledby="allowanceModalLabel" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="allowanceModalLabel">Add New Allowance</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
                <form id="allowanceForm">
                    @csrf
                    <input type="hidden" name="id_allowances" id="id_allowances">
                    
                    <!-- Business Selection for HO Users -->
                 
                    <input type="hidden" name="business_id" id="business_id" value="{{ session('business_id') }}">
                    
                    <div class="row">
                        <div class="col-12 mb-3">
                            <label for="allowance_name" class="form-label">Allowance Name <span class="text-danger">*</span></label>
                            <input type="text" class="form-control" id="allowance_name" name="allowance_name" required>
                            <div class="invalid-feedback" id="allowance_name_error"></div>
                        </div>
                        
                        <div class="col-12 mb-3">
                            <label for="entity_name" class="form-label">Entity Name <span class="text-danger">*</span></label>
                            <select class="form-control entity-select" id="entity_name" name="entity_name" required>
                                <option value="">Select Entity Name</option>
                                @foreach($mappedEntities as $entity)
                                    <option value="{{ $entity }}">{{ $entity }}</option>
                                @endforeach
                            </select>
                            <small class="text-muted">Must exist in Account Event Mapping with Event ID 26</small>
                            <div class="invalid-feedback" id="entity_name_error"></div>
                        </div>
                    </div>
                </form>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                <button type="button" class="btn btn-primary" id="saveAllowanceBtn">Save Allowance</button>
            </div>
        </div>
    </div>
</div>

<!-- View Allowance Modal -->
<div class="modal fade" id="viewAllowanceModal" tabindex="-1" aria-labelledby="viewAllowanceModalLabel" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="viewAllowanceModalLabel">Allowance Details</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
                <div class="row">
                    <div class="col-md-6">
                        <p><strong>Allowance ID:</strong> <span id="view_id"></span></p>
                        <p><strong>Allowance Name:</strong> <span id="view_allowance_name"></span></p>
                        <p><strong>Entity Name:</strong> <span id="view_entity_name"></span></p>
                        <p><strong>Business:</strong> <span id="view_business"></span></p>
                    </div>
                    <div class="col-md-6">
                        <p><strong>Created By:</strong> <span id="view_created_by"></span></p>
                        <p><strong>Created Date:</strong> <span id="view_created_at"></span></p>
                    </div>
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">
                    <iconify-icon icon="mdi:close" class="me-1 fs-22"></iconify-icon> Close
                </button>
            </div>
        </div>
    </div>
</div>

@endsection

@section('scripts')
@vite(['resources/js/app.js'])
<script>
    document.addEventListener('DOMContentLoaded', function() {
        initializeDatePickers();
        initializeSelect2();
        initializeAllowancesTable();
        setupEventHandlers();
    });
    
    let allowancesTable = null;
    let allowanceModal = null;
    let viewAllowanceModal = null;
    
    function initializeDatePickers() {
        if (typeof flatpickr !== 'undefined') {
            flatpickr('.date-picker', {
                dateFormat: 'Y-m-d',
                allowInput: true
            });
        }
    }
    
    function initializeSelect2() {
        if (typeof $().select2 === 'function') {
            
            // Initialize entity select in modal
            $('#entity_name').select2({
                dropdownParent: $('#allowanceModal'),
                width: '100%'
            });
        }
    }
    
    function initializeAllowancesTable() {
        if (typeof $.fn.DataTable === 'undefined') {
            console.error('DataTables is not loaded');
            return;
        }
        
        allowancesTable = $('#allowances_table').DataTable({
            scrollX: true, 
            orderCellsTop: true,           
            scrollCollapse: true,
            fixedHeader: {
                header: true,
                footer: false
            },
            processing: true,
            serverSide: true,
            stateSave: true,
            stateDuration: -1,
            responsive: false,
            ajax: {
                url: "{{ route('staff.allowances.data') }}",
                type: 'POST',
                contentType: 'application/json',
                headers: {
                    'X-CSRF-TOKEN': '{{ csrf_token() }}'
                },
                dataType: 'json',
                data: function (d) {
                    d.id_allowances = $('input[name="id_allowances"]').val();
                    d.allowance_name = $('input[name="allowance_name"]').val();
                    d.entity_name = $('input[name="entity_name"]').val();
                    d.business_name = $('input[name="business_name"]').val();
                    d.created_by = $('input[name="created_by"]').val();
                    d.created_at = $('input[name="created_at"]').val();
                    
                    return JSON.stringify(d);
                },
                error: function (xhr, error, thrown) {
                    console.error('DataTables AJAX error:', error, thrown, xhr.responseText);
                    showAlert('error', 'Failed to load data. Please try again.');
                }
            },
            "language": {
                "paginate": {
                    "previous": "<i class='ti ti-chevron-left'></i>",
                    "next": "<i class='ti ti-chevron-right'></i>"
                }
            },
            lengthChange: true,
            pageLength: 10,
            searching: false,
            info: true,
            paging: true,
            buttons: [
                { extend: 'copy', className: 'btn btn-sm btn-soft-primary' },
                { extend: 'csv', className: 'btn btn-sm btn-soft-primary' },
                { extend: 'print', className: 'btn btn-sm btn-soft-primary' }
            ],
            layout: {
                topStart: 'buttons',
                topEnd: function() {
                    return '<div class="d-flex justify-content-end"><button type="button" class="btn btn-outline-primary" id="addAllowanceBtn"><iconify-icon icon="mdi:plus" class="me-1"></iconify-icon> Add New Allowance</button></div>';
                },
                bottomStart: {
                    pageLength: {
                        menu: [10, 25, 50, 100]
                    },
                    info: {}
                },
                bottomEnd: 'paging'
            },
            columns: [
                { data: 'id_allowances', name: 'id_allowances', orderable: true, searchable: true },
                { data: 'allowance_name', name: 'allowance_name', orderable: true, searchable: true },
                { data: 'entity_name', name: 'entity_name', orderable: true, searchable: true },
                { data: 'business_name', name: 'business_name', orderable: true, searchable: true },
                { 
                    data: 'created_by', 
                    name: 'created_by', 
                    orderable: true, 
                    searchable: true,
                    render: function(data, type, row) {
                        return data || '-';
                    }
                },
                { 
                    data: 'created_at', 
                    name: 'created_at', 
                    orderable: true, 
                    searchable: true,
                    render: function(data, type, row) {
                        return data ? new Date(data).toLocaleDateString('en-GB') : '-';
                    }
                },
                { 
                    data: null, 
                    name: 'actions', 
                    orderable: false, 
                    searchable: false,
                    className: 'action-buttons',
                    render: function(data, type, row) {
                        let buttons = '';
                        
                        buttons += `
                            <button class="btn btn-sm btn-ghost-secondary rounded-circle btn-icon" 
                                    onclick="viewAllowanceDetails(${row.id_allowances})"
                                    title="View Details">
                                <iconify-icon icon="mdi:eye-outline" class="fs-22"></iconify-icon>
                            </button>
                            <button class="btn btn-sm btn-ghost-secondary rounded-circle btn-icon edit-allowance-btn" 
                                    data-id="${row.id_allowances}" 
                                    title="Edit Allowance">
                                <iconify-icon icon="line-md:edit-twotone" class="fs-22"></iconify-icon>
                            </button>
                        `;
                        
                        return buttons;
                    }
                }
            ],
            initComplete: function(settings, json) {
                var api = this.api();
                var filterRow = $(api.table().header()).find('tr.filter');
                filterRow.show();
            }
        });
    }
    
    function setupEventHandlers() {
        // Initialize Bootstrap modals
        const allowanceModalEl = document.getElementById('allowanceModal');
        const viewAllowanceModalEl = document.getElementById('viewAllowanceModal');
        
        if (allowanceModalEl) allowanceModal = new bootstrap.Modal(allowanceModalEl);
        if (viewAllowanceModalEl) viewAllowanceModal = new bootstrap.Modal(viewAllowanceModalEl);
        
        // Add allowance button
        const addAllowanceBtn = document.getElementById('addAllowanceBtn');
        if (addAllowanceBtn) {
            addAllowanceBtn.addEventListener('click', function() {
                openAllowanceModal();
            });
        }
        
        // Save allowance button
        const saveAllowanceBtn = document.getElementById('saveAllowanceBtn');
        if (saveAllowanceBtn) {
            saveAllowanceBtn.addEventListener('click', function() {
                saveAllowance();
            });
        }
        
        // Edit buttons (delegated events since they're dynamically created)
        document.addEventListener('click', function(e) {
            // Edit button
            if (e.target.closest('.edit-allowance-btn')) {
                const btn = e.target.closest('.edit-allowance-btn');
                const id = btn.getAttribute('data-id');
                editAllowance(id);
            }
            
            // Filter submit button
            if (e.target.closest('.filter-submit')) {
                e.preventDefault();
                refreshData();
            }
            
            // Filter cancel button
            if (e.target.closest('.filter-cancel')) {
                e.preventDefault();
                document.querySelectorAll('.form-filter').forEach(input => {
                    input.value = '';
                });
                refreshData();
            }
        });
        
        document.addEventListener('keyup', function(e) {
            if (e.target.classList.contains('form-filter') && e.key === 'Enter') {
                refreshData();
            }
        });
    }

    function refreshData() {
        if (allowancesTable) {
            allowancesTable.ajax.reload(null, false);
        }
    }

    function openAllowanceModal() {
        resetModalForm();
        clearValidationErrors();
        
        // Update modal title
        const modalTitle = document.getElementById('allowanceModalLabel');
        if (modalTitle) {
            modalTitle.textContent = 'Add New Allowance';
        }
        
       
        
        // Show modal
        if (allowanceModal) {
            allowanceModal.show();
        }
    }

    function resetModalForm() {
        // Reset form
        const form = document.getElementById('allowanceForm');
        if (form) {
            form.reset();
        }
        
        // Reset hidden field
        const idInput = document.getElementById('id_allowances');
        if (idInput) {
            idInput.value = '';
        }
        
        // Reset Select2
        const entitySelect = $('#entity_name');
        if (entitySelect.length) {
            entitySelect.val('').trigger('change');
        }
        
        @if(session('ho') === 'Yes')
            const businessSelect = $('#business_id');
            if (businessSelect.length) {
                businessSelect.val('').trigger('change');
            }
        @endif
    }

    function editAllowance(id) {
        // Show loading
        Swal.fire({
            title: 'Loading...',
            text: 'Please wait',
            allowOutsideClick: false,
            customClass: {
                confirmButton: "btn btn-danger me-2", 
                cancelButton: "btn btn-warning"
            },
            didOpen: () => {
                Swal.showLoading();
            }
        });
        
        fetch("{{ url('staff/get-allowance') }}/" + id, {
            method: 'GET',
            headers: {
                'Accept': 'application/json',
            }
        })
        .then(response => response.json())
        .then(response => {
            Swal.close();
            
            if (response.success) {
                resetModalForm();
                
                // Set form values
                const idInput = document.getElementById('id_allowances');
                const nameInput = document.getElementById('allowance_name');
                
                if (idInput) idInput.value = response.data.id_allowances;
                if (nameInput) nameInput.value = response.data.allowance_name;
                
                // Update modal title
                const modalTitle = document.getElementById('allowanceModalLabel');
                if (modalTitle) {
                    modalTitle.textContent = 'Edit Allowance';
                }
                
                clearValidationErrors();
                
                // Show modal
                if (allowanceModal) {
                    allowanceModal.show();
                }
                
                // Set values after a short delay to ensure modal is visible
                setTimeout(() => {
                    const entitySelect = $('#entity_name');
                    if (entitySelect.length && response.data.entity_name) {
                        entitySelect.val(response.data.entity_name).trigger('change');
                    }
                }, 100);
                
            } else {
                Swal.fire({
                    icon: 'error',
                    title: 'Access Denied',
                    text: response.message || 'You do not have permission to edit this allowance.',
                    customClass: {
                        confirmButton: "btn btn-danger me-2", 
                        cancelButton: "btn btn-warning"
                    }
                });
            }
        })
        .catch(error => {
            Swal.close();
            console.error('Error:', error);
            Swal.fire({
                icon: 'error',
                title: 'Error',
                text: 'Error loading allowance data. Please try again.',
                customClass: {
                    confirmButton: "btn btn-danger me-2", 
                    cancelButton: "btn btn-warning"
                },
            });
        });
    }
    
    function viewAllowanceDetails(id) {
        Swal.fire({
            title: 'Loading...',
            allowOutsideClick: false,
            customClass: {
                confirmButton: "btn btn-danger me-2",
                cancelButton: "btn btn-warning"
            },
            didOpen: () => Swal.showLoading()
        });
        
        fetch("{{ url('staff/get-allowance') }}/" + id, {
            method: 'GET',
            headers: {
                'Accept': 'application/json',
            }
        })
        .then(response => response.json())
        .then(response => {
            Swal.close();
            
            if (response.success) {
                const data = response.data;
                
                document.getElementById('view_id').textContent = data.id_allowances;
                document.getElementById('view_allowance_name').textContent = data.allowance_name;
                document.getElementById('view_entity_name').textContent = data.entity_name;
                document.getElementById('view_business').textContent = data.business_name || 'N/A';
                document.getElementById('view_created_by').textContent = data.created_by || '-';
                document.getElementById('view_created_at').textContent = data.created_at ? new Date(data.created_at).toLocaleDateString('en-GB') : '-';

                if (viewAllowanceModal) {
                    viewAllowanceModal.show();
                }
            } else {
                Swal.fire({
                    icon: 'error',
                    title: 'Error',
                    text: response.message || 'Failed to load allowance details.',
                    customClass: {
                        confirmButton: "btn btn-danger me-2",
                        cancelButton: "btn btn-warning"
                    }
                });
            }
        })
        .catch(error => {
            Swal.close();
            console.error('Error:', error);
            Swal.fire({
                icon: 'error',
                title: 'Error',
                text: 'Error loading allowance details. Please try again.',
                customClass: {
                    confirmButton: "btn btn-danger me-2",
                    cancelButton: "btn btn-warning"
                }
            });
        });
    }

    function saveAllowance() {
        clearValidationErrors();
        
        const form = document.getElementById('allowanceForm');
        if (!form) return;
        
        const formData = new FormData(form);
        
        // Validate required fields
        const allowanceName = document.getElementById('allowance_name');
        const entityName = document.getElementById('entity_name');
        
        let isValid = true;
        
        if (!allowanceName || !allowanceName.value.trim()) {
            if (allowanceName) {
                allowanceName.classList.add('is-invalid');
                const errorEl = document.getElementById('allowance_name_error');
                if (errorEl) errorEl.textContent = 'Allowance name is required.';
            }
            isValid = false;
        }
        
        if (!entityName || !entityName.value.trim()) {
            if (entityName) {
                entityName.classList.add('is-invalid');
                const errorEl = document.getElementById('entity_name_error');
                if (errorEl) errorEl.textContent = 'Entity name is required.';
            }
            isValid = false;
        }
        
      
        
        if (!isValid) {
            Swal.fire({
                icon: 'warning',
                title: 'Validation Error',
                text: 'Please fill in all required fields.',
                customClass: {
                    confirmButton: "btn btn-danger me-2", 
                    cancelButton: "btn btn-warning"
                },
            });
            return;
        }
        
        // Add CSRF token
        const csrfToken = document.querySelector('meta[name="csrf-token"]');
        if (csrfToken) {
            formData.append('_token', csrfToken.getAttribute('content'));
        }
        
        // Show loading
        Swal.fire({
            title: 'Saving...',
            text: 'Please wait',
            allowOutsideClick: false,
            customClass: {
                confirmButton: "btn btn-danger me-2", 
                cancelButton: "btn btn-warning"
            },
            didOpen: () => {
                Swal.showLoading();
            }
        });
        
        fetch("{{ route('staff.save_staff_allowance') }}", {
            method: 'POST',
            body: formData,
            headers: {
                'X-CSRF-TOKEN': csrfToken ? csrfToken.getAttribute('content') : '{{ csrf_token() }}'
            }
        })
        .then(response => response.json())
        .then(response => {
            Swal.close();
            
            if (response.success) {
                // Close modal and reset form
                if (allowanceModal) {
                    allowanceModal.hide();
                }
                resetModalForm();
                
                // Show success message
                Swal.fire({
                    icon: 'success',
                    title: 'Success!',
                    text: response.message,
                    timer: 2000,
                    showConfirmButton: false,
                    customClass: {
                        confirmButton: "btn btn-danger me-2", 
                        cancelButton: "btn btn-warning"
                    },
                }).then(() => {
                    refreshData();
                });
                
            } else {
                // Show error in Swal and keep modal open
                Swal.fire({
                    icon: 'error',
                    title: 'Error',
                    text: response.message,
                    customClass: {
                        confirmButton: "btn btn-danger me-2", 
                        cancelButton: "btn btn-warning"
                    },
                });
            }
        })
        .catch(error => {
            Swal.close();
            console.error('Save error:', error);
            
            // Show error in Swal and reset modal
            Swal.fire({
                icon: 'error',
                title: 'Error',
                text: 'Error saving allowance. Please try again.',
                customClass: {
                    confirmButton: "btn btn-danger me-2", 
                    cancelButton: "btn btn-warning"
                },
            }).then(() => {
                if (allowanceModal) {
                    allowanceModal.hide();
                }
                resetModalForm();
            });
        });
    }

    function clearValidationErrors() {
        // Remove invalid class from all form controls
        document.querySelectorAll('.form-control.is-invalid').forEach(el => {
            el.classList.remove('is-invalid');
        });
        
        // Clear error messages
        document.querySelectorAll('.invalid-feedback').forEach(el => {
            el.textContent = '';
        });
    }

    function showAlert(type, message) {
        const alertHtml = `
            <div class="row">
                <div class="col-12">
                    <div class="alert alert-${type} alert-dismissible fade show" role="alert">
                        <strong>${type === 'success' ? 'Success!' : 'Error!'}</strong> ${message}
                        <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                    </div>
                </div>
            </div>
        `;
        
        const container = document.getElementById('alert-container');
        if (container) {
            container.innerHTML = alertHtml;
            
            // Auto remove after 5 seconds
            setTimeout(() => {
                container.innerHTML = '';
            }, 5000);
        }
    }

    // Clean up when modal is hidden
    document.addEventListener('DOMContentLoaded', function() {
        const allowanceModalEl = document.getElementById('allowanceModal');
        if (allowanceModalEl) {
            allowanceModalEl.addEventListener('hidden.bs.modal', function () {
                resetModalForm();
                clearValidationErrors();
            });
        }
    });
    
    window.viewAllowanceDetails = viewAllowanceDetails;
</script>
@endsection