@extends('layouts.horizontal', ['title' => 'Attendance Requests', 'topbarTitle' => 'Attendance Requests View'])

@section('css')
@vite(['node_modules/datatables.net-bs5/css/dataTables.bootstrap5.min.css',
'node_modules/datatables.net-responsive-bs5/css/responsive.bootstrap5.min.css',
'node_modules/datatables.net-fixedcolumns-bs5/css/fixedColumns.bootstrap5.min.css',
'node_modules/datatables.net-fixedheader-bs5/css/fixedHeader.bootstrap5.min.css',
'node_modules/datatables.net-buttons-bs5/css/buttons.bootstrap5.min.css',
'node_modules/datatables.net-select-bs5/css/select.bootstrap5.min.css',
'node_modules/dropzone/dist/dropzone.css',
'node_modules/select2/dist/css/select2.min.css',
'node_modules/flatpickr/dist/flatpickr.min.css'])
@endsection

@section('content')
<style>
    tr.strike-through > td {text-decoration:line-through !important; color:red;} 
    .filter-row { background-color: #f9f9f9; }
    .form-filter { font-size: 12px; padding: 4px 6px; height: 30px; width: 100%; }
    .filter-row th { padding: 4px 6px !important; vertical-align: middle; }
    .filter-buttons { display: flex; gap: 5px; }
    .filter-buttons button { padding: 4px 8px; font-size: 12px; }
    .action-buttons { white-space: nowrap; }
    .btn-icon {
        width: 36px;
        height: 36px;
        display: inline-flex;
        align-items: center;
        justify-content: center;
        padding: 0;
    }
    .btn-rounded {
        border-radius: 50%;
    }
    .status-badge {
        padding: 4px 8px;
        border-radius: 4px;
        font-size: 12px;
        font-weight: 500;
    }
    .status-pending { background-color: #fff3cd; color: #856404; }
    .status-approved { background-color: #d4edda; color: #155724; }
    .status-rejected { background-color: #f8d7da; color: #721c24; }
    .table-card {
        background: white;
        border-radius: 8px;
        overflow: hidden;
        box-shadow: 0 2px 4px rgba(0,0,0,0.05);
    }
    .table-card .table {
        margin-bottom: 0;
    }
    .table-card .table thead th {
        background-color: #f1f5f9;
        border-bottom: 2px solid #dee2e6;
    }
    /* Business selector width fix */
    #businessSelector {
        width: fit-content !important;
        min-width: 200px;
        max-width: 400px;
    }
    #businessSelector + .select2-container {
        width: fit-content !important;
        min-width: 200px;
        max-width: 400px;
    }
</style>

<div class="container-fluid">
    <div class="row mb-4">
        <div class="col-12">
            <div class="page-title-box d-flex align-items-center justify-content-between">
                <h2 class="mb-0">Attendance Requests</h2>
                <div class="page-title-right d-flex align-items-center gap-3">
                    @php
                        $breadcrumb = getCurrentMenuBreadcrumb();
                    @endphp
                    @if(!empty($breadcrumb))
                        <nav aria-label="breadcrumb">
                            <ol class="breadcrumb mb-0 py-2">
                                @foreach($breadcrumb as $index => $item)
                                    @if($index === count($breadcrumb) - 1)
                                        <li class="breadcrumb-item active" aria-current="page">{{ $item }}</li>
                                    @else
                                        <li class="breadcrumb-item"><a href="javascript: void(0);">{{ $item }}</a></li>
                                    @endif
                                @endforeach
                            </ol>
                        </nav>
                    @endif
                </div>
            </div>
        </div>
    </div>

    @if(session('Success'))
    <div class="row">
        <div class="col-12">
            <div class="alert alert-success alert-dismissible fade show" role="alert">
                <strong>Success!</strong> {{ session('Success') }}
                <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
            </div>
        </div>
    </div>
    @endif

    @if(session('Error'))
    <div class="row">
        <div class="col-12">
            <div class="alert alert-danger alert-dismissible fade show" role="alert">
                <strong>Error!</strong> {{ session('Error') }}
                <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
            </div>
        </div>
    </div>
    @endif

   <!-- Business Selector Section - Only show for HO users -->
    @if(session('ho') === 'Yes')
        <div class="row mb-3">
            <div class="col-12">
                <div class="card">
                    <div class="card-body">
                        <div class="row align-items-center">
                                <div class="col-md-12">
                                <label class="form-label">Select Business</label><br>
                                <select class="form-control" id="businessSelector">
                                    <option value="">All Businesses</option>
                                    @foreach($businesses as $biz)
                                        <option value="{{ $biz->id_business }}" 
                                            {{ (request()->get('business_id') == $biz->id_business) ? 'selected' : '' }}>
                                            {{ $biz->business_name }}
                                        </option>
                                    @endforeach
                                </select>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    @endif

    <div class="row">
        <div class="col-12">
            <div class="table-card">
                <div class="table-responsive">
                    <table id="attendance_table" class="table table-hover">
                        <thead>
                            <tr>
                                <th>Request ID</th>
                                <th>Staff ID</th>
                                <th>Staff Name</th>
                                <th>Branch</th>
                                <th>Request Date</th>
                                <th class="text-end">Time In</th>
                                <th class="text-end">Time Out</th>
                                <th>Location</th>
                                <th>Status</th>
                                <th>Approved By</th>
                                <th>Actions</th>
                            </tr>
                            <tr class="filter filter-row">
                                <td>
                                    <input type="text" class="form-control form-filter input-sm" name="id_attendance_request" placeholder="Search ID">
                                </td>
                                <td>
                                    <input type="text" class="form-control form-filter input-sm" name="staff_id" placeholder="Search Staff ID">
                                </td>
                                <td>
                                    <input type="text" class="form-control form-filter input-sm" name="staff_name" placeholder="Search Name">
                                </td>
                                <td>
                                    <input type="text" class="form-control form-filter input-sm" name="business" placeholder="Search Branch">
                                </td>
                                <td>
                                    <input type="text" class="form-control form-filter input-sm date-picker" name="request_date" placeholder="Search Date">
                                </td>
                                <td>
                                    <input type="text" class="form-control form-filter input-sm" name="request_time_in" placeholder="Search Time In">
                                </td>
                                <td>
                                    <input type="text" class="form-control form-filter input-sm" name="request_time_out" placeholder="Search Time Out">
                                </td>
                                <td>
                                    <input type="text" class="form-control form-filter input-sm" name="address" placeholder="Search Location">
                                </td>
                                <td>
                                    <select name="request_status" class="form-control form-filter input-sm">
                                        <option value="All">All</option>
                                        <option value="Pending">Pending</option>
                                        <option value="Approved">Approved</option>
                                        <option value="Rejected">Rejected</option>
                                    </select>
                                </td>
                                <td>
                                    <input type="text" class="form-control form-filter input-sm" name="approved_by" placeholder="Search Approved By">
                                </td>
                                <td>
                                    <div class="filter-buttons">
                                        <button class="btn btn-sm btn-ghost-secondary rounded-circle btn-icon filter-submit">
                                            <iconify-icon icon="mdi:magnify" class="fs-22"></iconify-icon>
                                        </button>
                                        <button class="btn btn-sm btn-ghost-secondary rounded-circle btn-icon filter-cancel">
                                            <iconify-icon icon="mdi:refresh" class="fs-22"></iconify-icon>
                                        </button>
                                    </div>
                                </td>
                            </tr>
                        </thead>
                        <tbody></tbody>
                    </table>
                </div>
            </div>
        </div>
    </div>
</div>
@endsection

@section('scripts')
<script>
    document.addEventListener('DOMContentLoaded', function() {
        initializeDatePickers();
        initializeAttendanceTable();
        setupEventHandlers();
        setupBusinessChangeHandlers();
        initializeSelect2();
    });
    
    function initializeDatePickers() {
        flatpickr('.date-picker', {
            dateFormat: 'Y-m-d',
            allowInput: true
        });
    }
    
    function initializeSelect2() {
        $('#businessSelector').select2({
            width: 'fit-content',
            placeholder: 'Select Business'
        });
    }

    function setupBusinessChangeHandlers() {
        // Create hidden input if it doesn't exist
        if ($('#filter_business_id').length === 0) {
            $('body').append('<input type="hidden" id="filter_business_id" value="">');
        }
        
        $('#businessSelector').on('change', function() {
            const businessId = $(this).val();
            $('#filter_business_id').val(businessId);
            
            if (window.attendanceTable) {
                window.attendanceTable.ajax.reload();
            }
        });
    }
    
    function initializeAttendanceTable() {
        var table = $('#attendance_table').DataTable({
            scrollX: true,            
            scrollCollapse: true,
            orderCellsTop: true,
            fixedHeader: {
                header: true,
                footer: false
            },
            processing: true,
            serverSide: true,
            stateSave: true,
            stateDuration: -1,
            // Handle state loading for status filter
            stateLoadParams: function(settings, data) {
                // Check if there's saved search data
                if (data.search && data.search.columns) {
                    // Find status column index (it's the 8th column - adjust if needed)
                    const statusColIndex = 8;
                    if (data.search.columns[statusColIndex] && data.search.columns[statusColIndex].search) {
                        const savedStatus = data.search.columns[statusColIndex].search.search;
                        // Update the status dropdown with saved value
                        if (savedStatus) {
                            $('select[name="request_status"]').val(savedStatus);
                        }
                    }
                }
            },
            responsive: false,
            ajax: {
                url: "{{ route('attendance.requests.data') }}",
                type: 'POST',
                contentType: 'application/json',
                headers: {
                    'X-CSRF-TOKEN': '{{ csrf_token() }}'
                },
                dataType: 'json',
                data: function (d) {
                    d.id_attendance_request = $('input[name="id_attendance_request"]').val();
                    d.staff_id = $('input[name="staff_id"]').val();
                    d.staff_name = $('input[name="staff_name"]').val();
                    d.business = $('input[name="business"]').val();
                    d.request_date = $('input[name="request_date"]').val();
                    d.request_time_in = $('input[name="request_time_in"]').val();
                    d.request_time_out = $('input[name="request_time_out"]').val();
                    d.address = $('input[name="address"]').val();
                    
                    // Get status value - handle empty (All), Pending, Approved, Rejected
                    const statusVal = $('select[name="request_status"]').val();
                    d.request_status = statusVal;
                    
                    d.approved_by = $('input[name="approved_by"]').val();
                    
                    // Add business filter for HO users
                    @if(session('ho') === 'Yes' && session('ho_accounts') === 'Yes')
                        d.business_id = $('#filter_business_id').val();
                    @endif
                    
                    return JSON.stringify(d);
                },
                error: function (xhr, error, thrown) {
                    console.error('DataTables AJAX error:', error, thrown, xhr.responseText);
                    Swal.fire({
                        icon: 'error',
                        title: 'Error',
                        text: 'Failed to load data. Please try again.',
                        customClass: {
                            confirmButton: "btn btn-danger me-2", 
                            cancelButton: "btn btn-warning"
                        },
                    });
                }
            },
            "language": {
                "paginate": {
                    "previous": "<i class='ti ti-chevron-left'></i>",
                    "next": "<i class='ti ti-chevron-right'></i>"
                }
            },
            lengthChange: true,
            pageLength: 10,
            lengthMenu: [[10, 25, 50, 100], [10, 25, 50, 100]],
            searching: false,
            info: true,
            paging: true,
            layout: {
                topStart: null,
                topEnd: null,
                bottomStart: {
                    pageLength: { menu: [[10, 25, 50, 100], [10, 25, 50, 100]] },
                    info: {}
                },
                bottomEnd: "paging"
            },
            buttons: [
                { 
                    extend: "copy",  
                    className: "btn btn-sm btn-ghost-secondary rounded-circle btn-icon",
                    text: '<iconify-icon icon="mdi:content-copy" class="fs-22"></iconify-icon>'
                },
                { 
                    extend: "csv",   
                    className: "btn btn-sm btn-ghost-secondary rounded-circle btn-icon",
                    text: '<iconify-icon icon="mdi:file-delimited" class="fs-22"></iconify-icon>'
                },
                { 
                    extend: "print", 
                    className: "btn btn-sm btn-ghost-secondary rounded-circle btn-icon",
                    text: '<iconify-icon icon="mdi:printer" class="fs-22"></iconify-icon>'
                },
                { 
                    extend: "pdf", 
                    className: "btn btn-sm btn-ghost-secondary rounded-circle btn-icon",
                    text: '<iconify-icon icon="mdi:file-pdf" class="fs-22"></iconify-icon>'
                }
            ],
            columns: [
                { data: 'id_attendance_request', name: 'id_attendance_request', orderable: true, searchable: true },
                { data: 'staff_id', name: 'staff_id', orderable: true, searchable: true },
                { data: 'staff_name', name: 'staff_name', orderable: true, searchable: true },
                { data: 'business', name: 'business', orderable: true, searchable: true },
                { data: 'request_date', name: 'request_date', orderable: true, searchable: true },
                { data: 'request_time_in', name: 'request_time_in', className: 'text-end', orderable: true, searchable: true },
                { data: 'request_time_out', name: 'request_time_out', className: 'text-end', orderable: true, searchable: true },
                { data: 'address', name: 'address', orderable: true, searchable: true },
                { 
                    data: 'request_status', 
                    name: 'request_status', 
                    orderable: true, 
                    searchable: true,
                    render: function(data, type, row) {
                        let className = '';
                        let statusText = '';
                        
                        if(data.toLowerCase() === 'approved') {
                            className = 'status-approved';
                            statusText = 'Approved';
                        } else if(data.toLowerCase() === 'rejected') {
                            className = 'status-rejected';
                            statusText = 'Rejected';
                        } else {
                            className = 'status-pending';
                            statusText = 'Pending';
                        }
                        
                        return `<span class="status-badge ${className}">${statusText}</span>`;
                    }
                },
                { data: 'approved_by', name: 'approved_by', orderable: true, searchable: true },
                { 
                    data: null, 
                    name: 'actions', 
                    orderable: false, 
                    searchable: false,
                    className: 'action-buttons',
                    render: function(data, type, row) {
                        var status = row.request_status.toLowerCase();
                        var buttons = '';
                        
                        if(status === 'pending') {
                            buttons += `
                                <div class="filter-buttons d-flex gap-1 justify-content-center">
                                    <button 
                                        onclick="updateStatus(${row.id_attendance_request}, 'approved')" 
                                        class="btn btn-sm btn-ghost-secondary rounded-circle btn-icon"
                                        title="Approve">
                                        <iconify-icon icon="mdi:check" class="fs-22 text-success"></iconify-icon>
                                    </button>
                                    <button 
                                        onclick="updateStatus(${row.id_attendance_request}, 'rejected')" 
                                        class="btn btn-sm btn-ghost-secondary rounded-circle btn-icon"
                                        title="Reject">
                                        <iconify-icon icon="mdi:close" class="fs-22 text-danger"></iconify-icon>
                                    </button>
                                </div>
                            `;
                        } else {
                            buttons += `<span class="text-muted">No actions available</span>`;
                        }
                        
                        return buttons;
                    }
                }
            ],
            "rowCallback": function(row, data, displayNum, displayIndex, dataIndex) {
                if (data.request_status.toLowerCase() === 'rejected') {
                    $(row).addClass('strike-through');
                }
            },
            initComplete: function(settings, json) {
                var api = this.api();
                var filterRow = $(api.table().header()).find('tr.filter');
                filterRow.show();
                
                // Set default status to "Pending" on first load
                const state = api.state.loaded();
                if (!state || !state.columns || !state.columns[8] || !state.columns[8].search || state.columns[8].search.search === '') {
                    // First load or no saved status, set to "Pending"
                    $('select[name="request_status"]').val('Pending');
                }
            }
        });
        window.attendanceTable = table;
    }
    
    function setupEventHandlers() {
        // Search button
        $(document).on('click', '.filter-submit', function(e) {
            e.preventDefault();
            refresh_data();
        });

        // Reset button - clear all filters
        $(document).on('click', '.filter-cancel', function(e) {
            e.preventDefault();
            document.querySelectorAll('.form-filter').forEach(input => {
                input.value = '';
            });
            document.querySelectorAll('select.form-filter').forEach(select => {
                select.value = '';
            });
            
            // Reset status to "Pending" when clearing filters
            $('select[name="request_status"]').val('Pending');
            
            refresh_data();
        });

        // Auto-search when status changes
        $(document).on('change', 'select[name="request_status"]', function() {
            refresh_data();
        });

        // Enter key in filter inputs
        $(document).on('keyup', '.form-filter', function(e) {
            if (e.keyCode === 13) {
                refresh_data();
            }
        });
    }

    function refresh_data() {
        if (window.attendanceTable) {
            window.attendanceTable.ajax.reload(null, false);
        }
    }

    function updateStatus(id, status) {
        var statusText = status === 'approved' ? 'approve' : 'reject';
        var statusDisplay = status === 'approved' ? 'Approve' : 'Reject';
        
        Swal.fire({
            title: `Are you sure?`,
            text: `Do you want to ${statusText} this attendance request?`,
            icon: 'warning',
            showCancelButton: true,
            customClass: {
                confirmButton: "btn btn-danger me-2", 
                cancelButton: "btn btn-warning"
            },
            confirmButtonText: `Yes, ${statusText} it!`,
            cancelButtonText: 'Cancel'
        }).then((result) => {
            if (result.isConfirmed) {
                Swal.fire({
                    title: 'Processing...',
                    text: 'Please wait',
                    allowOutsideClick: false,
                    customClass: {
                        confirmButton: "btn btn-danger me-2", 
                        cancelButton: "btn btn-warning"
                    },
                    didOpen: () => {
                        Swal.showLoading();
                    }
                });
                
                $.ajax({
                    url: "{{ route('attendance.update.status', ['id' => 'PLACEHOLDER']) }}".replace('PLACEHOLDER', id),
                    type: 'POST',
                    headers: {
                        'X-CSRF-TOKEN': '{{ csrf_token() }}'
                    },
                    data: {
                        request_status: status
                    },
                    success: function(response) {
                        Swal.close();
                        if(response.success) {
                            Swal.fire({
                                icon: 'success',
                                title: 'Success',
                                text: response.message || `Request ${status} successfully`,
                                timer: 2000,
                                showConfirmButton: false,
                                customClass: {
                                    confirmButton: "btn btn-danger me-2", 
                                    cancelButton: "btn btn-warning"
                                },
                            }).then(() => {
                                refresh_data();
                            });
                        } else {
                            Swal.fire({
                                icon: 'error',
                                title: 'Error',
                                text: response.message || 'Error updating status',
                                customClass: {
                                    confirmButton: "btn btn-danger me-2", 
                                    cancelButton: "btn btn-warning"
                                },
                            });
                        }
                    },
                    error: function(xhr) {
                        Swal.close();
                        var errorMessage = 'Error updating status';
                        if (xhr.responseJSON && xhr.responseJSON.message) {
                            errorMessage = xhr.responseJSON.message;
                        } else if (xhr.statusText) {
                            errorMessage = xhr.statusText;
                        }
                        
                        Swal.fire({
                            icon: 'error',
                            title: 'Error',
                            text: errorMessage,
                            customClass: {
                                confirmButton: "btn btn-danger me-2", 
                                cancelButton: "btn btn-warning"
                            },
                        });
                    }
                });
            }
        });
    }
</script>
@endsection