@extends('layouts.horizontal', ['title' => 'Deduction Policies', 'topbarTitle' => 'Deduction Policies'])

@section('css')
@vite(['node_modules/datatables.net-bs5/css/dataTables.bootstrap5.min.css',
'node_modules/datatables.net-responsive-bs5/css/responsive.bootstrap5.min.css',
'node_modules/datatables.net-fixedcolumns-bs5/css/fixedColumns.bootstrap5.min.css',
'node_modules/datatables.net-fixedheader-bs5/css/fixedHeader.bootstrap5.min.css',
'node_modules/datatables.net-buttons-bs5/css/buttons.bootstrap5.min.css',
'node_modules/datatables.net-select-bs5/css/select.bootstrap5.min.css',
'node_modules/dropzone/dist/dropzone.css',
'node_modules/select2/dist/css/select2.min.css',
'node_modules/flatpickr/dist/flatpickr.min.css'])
<style>
    .filter-row { background-color: #f9f9f9; }
    .form-filter { font-size: 12px; padding: 4px 6px; height: 30px; width: 100%; }
    .filter-row th { padding: 4px 6px !important; vertical-align: middle; }
    .dataTables_wrapper .dataTables_filter { float: right; }
    .dataTables_wrapper .dataTables_length { float: left; }
    .dataTables_wrapper .dataTables_paginate { float: right; }
    .filter-buttons { display: flex; gap: 5px; }
    .filter-buttons button { padding: 4px 8px; font-size: 12px; }
    .action-buttons { white-space: nowrap; }
    .staff-count { 
        display: inline-block;
        background-color: #e9ecef;
        color: #495057;
        padding: 2px 8px;
        border-radius: 12px;
        font-size: 12px;
        font-weight: 600;
    }
    
    .dataTables_scrollHead { background: #f8f9fa; }
    
    .btn-icon {
        width: 36px;
        height: 36px;
        display: inline-flex;
        align-items: center;
        justify-content: center;
        padding: 0;
    }
    
    .btn-rounded {
        border-radius: 50%;
    }
    
    .form-label {
        font-weight: 500;
        color: #495057;
    }
    
    .required:after {
        content: " *";
        color: #dc3545;
    }
</style>
@endsection

@section('content')
<div class="container-fluid">
    <div class="row mb-4">
        <div class="col-12">
            <div class="page-title-box d-flex align-items-center justify-content-between">
                <h2 class="mb-0">Deduction Policies</h2>
                <div class="page-title-right d-flex align-items-center gap-3">
                    @php
                        $breadcrumb = getCurrentMenuBreadcrumb();
                    @endphp
                    @if(!empty($breadcrumb))
                        <nav aria-label="breadcrumb">
                            <ol class="breadcrumb mb-0 py-2">
                                @foreach($breadcrumb as $index => $item)
                                    @if($index === count($breadcrumb) - 1)
                                        <li class="breadcrumb-item active" aria-current="page">{{ $item }}</li>
                                    @else
                                        <li class="breadcrumb-item"><a href="javascript: void(0);">{{ $item }}</a></li>
                                    @endif
                                @endforeach
                            </ol>
                        </nav>
                    @endif
                </div>
            </div>
        </div>
    </div>

    <div id="alert-container"></div>
    
    <div class="row">
        <div class="col-12">
            <div class="card">
                <div class="card-body">
                    <table id="deduction_policies_table" class="table table-bordered dt-responsive fs-13 w-100">
                        <thead>
                            <tr>
                                <th>ID</th>
                                <th>Policy Name</th>
                                <th>Business</th>
                                <th>Days Deducted per Absent</th>
                                <th>Days Deducted per Late</th>
                                <th>Max Late Allowed</th>
                                <th>Max Absents Allowed</th>
                                <th>Assigned Staff</th>
                                <th>Created Date</th>
                                <th>Actions</th>
                            </tr>
                            <tr class="filter filter-row">
                                <td>
                                    <input type="text" class="form-control form-filter input-sm" name="id_deduction_policy" placeholder="Search ID">
                                </td>
                                <td>
                                    <input type="text" class="form-control form-filter input-sm" name="deduction_policy" placeholder="Policy Name">
                                </td>
                                <td>
                                    <input type="text" class="form-control form-filter input-sm" name="business_name" placeholder="Business">
                                </td>
                                <td>
                                    <input type="text" class="form-control form-filter input-sm" name="deduction_days_absent" placeholder="Days/Absent">
                                </td>
                                <td>
                                    <input type="text" class="form-control form-filter input-sm" name="deduction_days_late" placeholder="Days/Late">
                                </td>
                                <td>
                                    <input type="text" class="form-control form-filter input-sm" name="total_late_allowed" placeholder="Max Late">
                                </td>
                                <td>
                                    <input type="text" class="form-control form-filter input-sm" name="total_absents_allowed" placeholder="Max Absents">
                                </td>
                                <td></td>
                                <td>
                                    <input type="text" class="form-control form-filter input-sm date-picker" name="created_at" placeholder="Created Date">
                                </td>
                                <td>
                                    <div class="filter-buttons">
                                        <button class="btn btn-sm btn-ghost-secondary rounded-circle btn-icon filter-submit" title="Search">
                                            <iconify-icon icon="mdi:magnify" class="fs-22"></iconify-icon>
                                        </button>
                                        <button class="btn btn-sm btn-ghost-secondary rounded-circle btn-icon filter-cancel" title="Reset">
                                            <iconify-icon icon="mdi:refresh" class="fs-22"></iconify-icon>
                                        </button>
                                    </div>
                                </td>
                            </tr>
                        </thead>
                        <tbody></tbody>
                    </table>
                </div>
            </div>
        </div>
    </div>
</div>

<div class="modal fade" id="policyModal" tabindex="-1" aria-labelledby="policyModalLabel" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="policyModalLabel">Create Deduction Policy</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
                <form id="policyForm">
                    @csrf
                    <input type="hidden" name="id_deduction_policy" id="policy_id">
                    <!-- Hidden business_id field - always use session business_id -->
                    <input type="hidden" name="business_id" id="business_id" value="{{ session('business_id') }}">
                    
                    <div class="row">
                        <div class="col-12 mb-3">
                            <label for="deduction_policy" class="form-label required">Policy Name</label>
                            <input type="text" class="form-control" id="deduction_policy" name="deduction_policy" required>
                        </div>
                        
                        <div class="col-md-6 mb-3">
                            <label for="deduction_days_absent" class="form-label required">Days Deducted per Absent</label>
                            <input type="number" step="0.01" min="0" class="form-control" id="deduction_days_absent" name="deduction_days_absent" required>
                            <small class="text-muted">Number of days deducted for each absence</small>
                        </div>
                        
                        <div class="col-md-6 mb-3">
                            <label for="deduction_days_late" class="form-label required">Days Deducted per Late</label>
                            <input type="number" step="0.01" min="0" class="form-control" id="deduction_days_late" name="deduction_days_late" required>
                            <small class="text-muted">Number of days deducted for each late arrival</small>
                        </div>
                        
                        <div class="col-md-6 mb-3">
                            <label for="total_late_allowed" class="form-label required">Max Late Arrivals Allowed</label>
                            <input type="number" min="0" class="form-control" id="total_late_allowed" name="total_late_allowed" required>
                            <small class="text-muted">Number of late arrivals allowed before deduction</small>
                        </div>
                        
                        <div class="col-md-6 mb-3">
                            <label for="total_absents_allowed" class="form-label required">Max Absences Allowed</label>
                            <input type="number" min="0" class="form-control" id="total_absents_allowed" name="total_absents_allowed" required>
                            <small class="text-muted">Number of absences allowed before deduction</small>
                        </div>
                    </div>
                </form>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">
                    <iconify-icon icon="mdi:close" class="me-1 fs-22"></iconify-icon> Cancel
                </button>
                <button type="button" class="btn btn-primary" id="savePolicyBtn">
                    <iconify-icon icon="mdi:content-save" class="me-1 fs-22"></iconify-icon> Save Policy
                </button>
            </div>
        </div>
    </div>
</div>

<div class="modal fade" id="viewPolicyModal" tabindex="-1" aria-labelledby="viewPolicyModalLabel" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="viewPolicyModalLabel">Deduction Policy Details</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
                <div class="row">
                    <div class="col-md-6">
                        <p><strong>Policy ID:</strong> <span id="view_id"></span></p>
                        <p><strong>Policy Name:</strong> <span id="view_policy_name"></span></p>
                        <p><strong>Business:</strong> <span id="view_business"></span></p>
                        <p><strong>Days/Absent:</strong> <span id="view_days_absent"></span></p>
                        <p><strong>Days/Late:</strong> <span id="view_days_late"></span></p>
                    </div>
                    <div class="col-md-6">
                        <p><strong>Max Late Allowed:</strong> <span id="view_max_late"></span></p>
                        <p><strong>Max Absents Allowed:</strong> <span id="view_max_absents"></span></p>
                        <p><strong>Created Date:</strong> <span id="view_created_at"></span></p>
                        <p><strong>Assigned Staff:</strong> <span id="view_staff_count" class="staff-count"></span></p>
                    </div>
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">
                    <iconify-icon icon="mdi:close" class="me-1 fs-22"></iconify-icon> Close
                </button>
            </div>
        </div>
    </div>
</div>
@endsection

@section('scripts')
@vite(['resources/js/app.js'])
<script>
    document.addEventListener('DOMContentLoaded', function() {
        initializeDatePickers();
        initializeDeductionPoliciesTable();
        setupEventHandlers();
    });
    
    let deductionPoliciesTable = null;
    let policyModal = null;
    let viewPolicyModal = null;
    
    function initializeDatePickers() {
        if (typeof flatpickr !== 'undefined') {
            flatpickr('.date-picker', {
                dateFormat: 'Y-m-d',
                allowInput: true
            });
        }
    }
    
    function initializeDeductionPoliciesTable() {
        if (typeof $.fn.DataTable === 'undefined') {
            console.error('DataTables is not loaded');
            return;
        }
        
        deductionPoliciesTable = $('#deduction_policies_table').DataTable({
            scrollX: true, 
            orderCellsTop: true,           
            scrollCollapse: true,
            fixedHeader: {
                header: true,
                footer: false
            },
            processing: true,
            serverSide: true,
            stateSave: true,
            stateDuration: -1,
            responsive: false,
            ajax: {
                url: "{{ route('hrm.get_deduction_policies') }}",
                type: 'POST',
                contentType: 'application/json',
                headers: {
                    'X-CSRF-TOKEN': '{{ csrf_token() }}'
                },
                dataType: 'json',
                data: function (d) {
                    d.id_deduction_policy = $('input[name="id_deduction_policy"]').val();
                    d.deduction_policy = $('input[name="deduction_policy"]').val();
                    d.business_name = $('input[name="business_name"]').val();
                    d.deduction_days_absent = $('input[name="deduction_days_absent"]').val();
                    d.deduction_days_late = $('input[name="deduction_days_late"]').val();
                    d.total_late_allowed = $('input[name="total_late_allowed"]').val();
                    d.total_absents_allowed = $('input[name="total_absents_allowed"]').val();
                    d.created_at = $('input[name="created_at"]').val();
                    
                    return JSON.stringify(d);
                },
                error: function (xhr, error, thrown) {
                    console.error('DataTables AJAX error:', error, thrown, xhr.responseText);
                    showAlert('error', 'Failed to load deduction policies. Please try again.');
                }
            },
            "language": {
                "paginate": {
                    "previous": "<i class='ti ti-chevron-left'></i>",
                    "next": "<i class='ti ti-chevron-right'></i>"
                }
            },
            lengthChange: true,
            pageLength: 10,
            searching: false,
            info: true,
            paging: true,
            buttons: [
                { extend: 'copy', className: 'btn btn-sm btn-soft-primary' },
                { extend: 'csv', className: 'btn btn-sm btn-soft-primary' },
                { extend: 'print', className: 'btn btn-sm btn-soft-primary' }
            ],
            layout: {
                topStart: 'buttons',
                topEnd: function() {
                    return '<div class="d-flex justify-content-end"><button type="button" class="btn btn-outline-primary" id="addPolicyBtn"><iconify-icon icon="mdi:plus" class="me-1 fs-22"></iconify-icon> Create New Policy</button></div>';
                },
                bottomStart: {
                    pageLength: {
                        menu: [10, 25, 50, 100]
                    },
                    info: {}
                },
                bottomEnd: 'paging'
            },
            columns: [
                { data: 'id_deduction_policy', name: 'id_deduction_policy', orderable: true, searchable: true },
                { data: 'deduction_policy', name: 'deduction_policy', orderable: true, searchable: true },
                { data: 'business_name', name: 'business_name', orderable: true, searchable: true },
                { data: 'deduction_days_absent', name: 'deduction_days_absent', orderable: true, searchable: true },
                { data: 'deduction_days_late', name: 'deduction_days_late', orderable: true, searchable: true },
                { data: 'total_late_allowed', name: 'total_late_allowed', orderable: true, searchable: true },
                { data: 'total_absents_allowed', name: 'total_absents_allowed', orderable: true, searchable: true },
                { 
                    data: 'staff_count', 
                    name: 'staff_count', 
                    orderable: true, 
                    searchable: false,
                    render: function(data, type, row) {
                        if (data > 0) {
                            return '<span class="staff-count">' + data + ' Staff</span>';
                        }
                        return '0';
                    }
                },
                { data: 'created_at', name: 'created_at', orderable: true, searchable: true },
                { 
                    data: null, 
                    name: 'actions', 
                    orderable: false, 
                    searchable: false,
                    className: 'action-buttons',
                    render: function(data, type, row) {
                        let buttons = '';
                        
                        buttons += `
                            <button class="btn btn-sm btn-ghost-secondary rounded-circle btn-icon view-policy-btn" 
                                    data-id="${row.id_deduction_policy}"
                                    title="View Details">
                                <iconify-icon icon="mdi:eye-outline" class="fs-22"></iconify-icon>
                            </button>
                        `;

                        if (!row.is_assigned) {
                            buttons += `
                                <button class="btn btn-sm btn-ghost-secondary rounded-circle btn-icon edit-policy-btn" 
                                        data-id="${row.id_deduction_policy}"
                                        title="Edit Policy">
                                    <iconify-icon icon="line-md:edit-twotone" class="fs-22"></iconify-icon>
                                </button>
                            `;
                        }
                        
                        if (!row.is_assigned) {
                            buttons += `
                                <button class="btn btn-sm btn-ghost-danger rounded-circle btn-icon delete-policy-btn" 
                                        data-id="${row.id_deduction_policy}"
                                        title="Delete Policy">
                                    <iconify-icon icon="mdi:trash-can-outline" class="fs-22"></iconify-icon>
                                </button>
                            `;
                        }
                        
                        return buttons;
                    }
                }
            ],
            initComplete: function(settings, json) {
                var api = this.api();
                var filterRow = $(api.table().header()).find('tr.filter');
                filterRow.show();
            }
        });
    }
    
    function setupEventHandlers() {
        const policyModalEl = document.getElementById('policyModal');
        const viewPolicyModalEl = document.getElementById('viewPolicyModal');
        
        if (policyModalEl) policyModal = new bootstrap.Modal(policyModalEl);
        if (viewPolicyModalEl) viewPolicyModal = new bootstrap.Modal(viewPolicyModalEl);
        
        const addPolicyBtn = document.getElementById('addPolicyBtn');
        if (addPolicyBtn) {
            addPolicyBtn.addEventListener('click', function() {
                openAddPolicyModal();
            });
        }
        
        const savePolicyBtn = document.getElementById('savePolicyBtn');
        if (savePolicyBtn) {
            savePolicyBtn.addEventListener('click', function() {
                savePolicy();
            });
        }
        
        // Use event delegation for dynamic buttons
        $(document).on('click', '.view-policy-btn', function() {
            const id = $(this).data('id');
            viewPolicyDetails(id);
        });
        
        $(document).on('click', '.edit-policy-btn', function() {
            const id = $(this).data('id');
            openEditPolicy(id);
        });
        
        $(document).on('click', '.delete-policy-btn', function() {
            const id = $(this).data('id');
            deletePolicy(id);
        });
        
        document.addEventListener('click', function(e) {
            if (e.target.closest('.filter-submit')) {
                e.preventDefault();
                refreshData();
            }
            
            if (e.target.closest('.filter-cancel')) {
                e.preventDefault();
                document.querySelectorAll('.form-filter').forEach(input => {
                    input.value = '';
                });
                refreshData();
            }
        });
        
        document.addEventListener('keyup', function(e) {
            if (e.target.classList.contains('form-filter') && e.key === 'Enter') {
                refreshData();
            }
        });
    }

    function refreshData() {
        if (deductionPoliciesTable) {
            deductionPoliciesTable.ajax.reload(null, false);
        }
    }

    function openAddPolicyModal() {
        resetPolicyForm();
        document.getElementById('policyModalLabel').textContent = 'Create Deduction Policy';
        
        if (policyModal) {
            policyModal.show();
        }
    }

    function openEditPolicy(id) {
        Swal.fire({
            title: 'Loading...',
            allowOutsideClick: false,
            customClass: {
                confirmButton: "btn btn-danger me-2",
                cancelButton: "btn btn-warning"
            },
            didOpen: () => Swal.showLoading()
        });
        
        fetch("{{ route('hrm.get_deduction_policy', '') }}/" + id, {
            method: 'GET',
            headers: {
                'Accept': 'application/json',
            }
        })
        .then(response => response.json())
        .then(response => {
            Swal.close();
            
            if (response.success) {
                const data = response.data;
                
                document.getElementById('policy_id').value = data.id_deduction_policy;
                document.getElementById('deduction_policy').value = data.deduction_policy;
                document.getElementById('deduction_days_absent').value = data.deduction_days_absent;
                document.getElementById('deduction_days_late').value = data.deduction_days_late;
                document.getElementById('total_late_allowed').value = data.total_late_allowed;
                document.getElementById('total_absents_allowed').value = data.total_absents_allowed;
                
                document.getElementById('policyModalLabel').textContent = 'Edit Deduction Policy';
                if (policyModal) {
                    policyModal.show();
                }
            } else {
                Swal.fire({
                    icon: 'error',
                    title: 'Error',
                    text: response.message || 'Failed to load deduction policy.',
                    customClass: {
                        confirmButton: "btn btn-danger me-2",
                        cancelButton: "btn btn-warning"
                    }
                });
            }
        })
        .catch(error => {
            Swal.close();
            console.error('Error:', error);
            Swal.fire({
                icon: 'error',
                title: 'Error',
                text: 'Error loading deduction policy. Please try again.',
                customClass: {
                    confirmButton: "btn btn-danger me-2",
                    cancelButton: "btn btn-warning"
                }
            });
        });
    }

    function viewPolicyDetails(id) {
        Swal.fire({
            title: 'Loading...',
            allowOutsideClick: false,
            customClass: {
                confirmButton: "btn btn-danger me-2",
                cancelButton: "btn btn-warning"
            },
            didOpen: () => Swal.showLoading()
        });
        
        fetch("{{ route('hrm.get_deduction_policy', '') }}/" + id, {
            method: 'GET',
            headers: {
                'Accept': 'application/json',
            }
        })
        .then(response => response.json())
        .then(response => {
            Swal.close();
            
            if (response.success) {
                const data = response.data;
                
                document.getElementById('view_id').textContent = data.id_deduction_policy;
                document.getElementById('view_policy_name').textContent = data.deduction_policy;
                document.getElementById('view_business').textContent = data.business_name || 'N/A';
                document.getElementById('view_days_absent').textContent = data.deduction_days_absent;
                document.getElementById('view_days_late').textContent = data.deduction_days_late;
                document.getElementById('view_max_late').textContent = data.total_late_allowed;
                document.getElementById('view_max_absents').textContent = data.total_absents_allowed;
                document.getElementById('view_created_at').textContent = data.created_at ? new Date(data.created_at).toLocaleDateString('en-GB') : '-';
                document.getElementById('view_staff_count').textContent = response.staff_count || 0;

                if (viewPolicyModal) {
                    viewPolicyModal.show();
                }
            } else {
                Swal.fire({
                    icon: 'error',
                    title: 'Error',
                    text: response.message || 'Failed to load policy details.',
                    customClass: {
                        confirmButton: "btn btn-danger me-2",
                        cancelButton: "btn btn-warning"
                    }
                });
            }
        })
        .catch(error => {
            Swal.close();
            console.error('Error:', error);
            Swal.fire({
                icon: 'error',
                title: 'Error',
                text: 'Error loading policy details. Please try again.',
                customClass: {
                    confirmButton: "btn btn-danger me-2",
                    cancelButton: "btn btn-warning"
                }
            });
        });
    }

    function deletePolicy(id) {
        Swal.fire({
            title: 'Are you sure?',
            text: "You won't be able to revert this!",
            icon: 'warning',
            showCancelButton: true,
            confirmButtonColor: '#3085d6',
            cancelButtonColor: '#d33',
            confirmButtonText: 'Yes, delete it!',
            customClass: {
                confirmButton: "btn btn-danger me-2",
                cancelButton: "btn btn-warning"
            }
        }).then((result) => {
            if (result.isConfirmed) {
                Swal.fire({
                    title: 'Deleting...',
                    allowOutsideClick: false,
                    customClass: {
                        confirmButton: "btn btn-danger me-2",
                        cancelButton: "btn btn-warning"
                    },
                    didOpen: () => Swal.showLoading()
                });
                
                fetch("{{ route('hrm.delete_deduction_policy', '') }}/" + id, {
                    method: 'POST',
                    headers: {
                        'X-CSRF-TOKEN': '{{ csrf_token() }}',
                        'Accept': 'application/json',
                    }
                })
                .then(response => response.json())
                .then(response => {
                    Swal.close();
                    
                    if (response.success) {
                        Swal.fire({
                            icon: 'success',
                            title: 'Deleted!',
                            text: response.message,
                            timer: 2000,
                            showConfirmButton: false,
                            customClass: {
                                confirmButton: "btn btn-danger me-2",
                                cancelButton: "btn btn-warning"
                            }
                        }).then(() => {
                            refreshData();
                        });
                    } else {
                        Swal.fire({
                            icon: 'error',
                            title: 'Error',
                            text: response.message,
                            customClass: {
                                confirmButton: "btn btn-danger me-2",
                                cancelButton: "btn btn-warning"
                            }
                        });
                    }
                })
                .catch(error => {
                    Swal.close();
                    console.error('Error:', error);
                    Swal.fire({
                        icon: 'error',
                        title: 'Error',
                        text: 'Error deleting policy. Please try again.',
                        customClass: {
                            confirmButton: "btn btn-danger me-2",
                            cancelButton: "btn btn-warning"
                        }
                    });
                });
            }
        });
    }

    function savePolicy() {
        const form = document.getElementById('policyForm');
        const formData = new FormData(form);
        
        const policyName = document.getElementById('deduction_policy');
        const daysAbsent = document.getElementById('deduction_days_absent');
        const daysLate = document.getElementById('deduction_days_late');
        const maxLate = document.getElementById('total_late_allowed');
        const maxAbsents = document.getElementById('total_absents_allowed');
        
        if (!policyName.value || !daysAbsent.value || !daysLate.value || !maxLate.value || !maxAbsents.value) {
            Swal.fire({
                icon: 'warning',
                title: 'Validation Error',
                text: 'Please fill in all required fields.',
                customClass: {
                    confirmButton: "btn btn-danger me-2",
                    cancelButton: "btn btn-warning"
                }
            });
            return;
        }
        
        Swal.fire({
            title: 'Saving...',
            allowOutsideClick: false,
            customClass: {
                confirmButton: "btn btn-danger me-2",
                cancelButton: "btn btn-warning"
            },
            didOpen: () => Swal.showLoading()
        });
        
        fetch("{{ route('hrm.save_deduction_policy') }}", {
            method: 'POST',
            body: formData,
            headers: {
                'X-CSRF-TOKEN': '{{ csrf_token() }}'
            }
        })
        .then(response => response.json())
        .then(response => {
            Swal.close();
            
            if (response.success) {
                if (policyModal) {
                    policyModal.hide();
                }
                resetPolicyForm();
                Swal.fire({
                    icon: 'success',
                    title: 'Success!',
                    text: response.message,
                    timer: 2000,
                    showConfirmButton: false,
                    customClass: {
                        confirmButton: "btn btn-danger me-2",
                        cancelButton: "btn btn-warning"
                    }
                }).then(() => {
                    refreshData();
                });
                
            } else {
                Swal.fire({
                    icon: 'error',
                    title: 'Error',
                    text: response.message,
                    customClass: {
                        confirmButton: "btn btn-danger me-2",
                        cancelButton: "btn btn-warning"
                    }
                });
            }
        })
        .catch(error => {
            Swal.close();
            console.error('Error:', error);
            Swal.fire({
                icon: 'error',
                title: 'Error',
                text: 'Error saving deduction policy. Please try again.',
                customClass: {
                    confirmButton: "btn btn-danger me-2",
                    cancelButton: "btn btn-warning"
                }
            });
        });
    }

    function resetPolicyForm() {
        const form = document.getElementById('policyForm');
        if (form) {
            form.reset();
            document.getElementById('policy_id').value = '';
        }
    }

    function showAlert(type, message) {
        const alertHtml = `
            <div class="row">
                <div class="col-12">
                    <div class="alert alert-${type} alert-dismissible fade show" role="alert">
                        <strong>${type === 'success' ? 'Success!' : 'Error!'}</strong> ${message}
                        <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                    </div>
                </div>
            </div>
        `;
        
        const container = document.getElementById('alert-container');
        if (container) {
            container.innerHTML = alertHtml;
            
            setTimeout(() => {
                container.innerHTML = '';
            }, 5000);
        }
    }

    document.addEventListener('DOMContentLoaded', function() {
        const policyModalEl = document.getElementById('policyModal');
        
        if (policyModalEl) {
            policyModalEl.addEventListener('hidden.bs.modal', function () {
                resetPolicyForm();
            });
        }
    });
</script>
@endsection