@extends('layouts.horizontal', ['title' => 'Leave Policies', 'topbarTitle' => 'Leave Policies'])

@section('css')
@vite(['node_modules/datatables.net-bs5/css/dataTables.bootstrap5.min.css',
'node_modules/datatables.net-responsive-bs5/css/responsive.bootstrap5.min.css',
'node_modules/datatables.net-fixedcolumns-bs5/css/fixedColumns.bootstrap5.min.css',
'node_modules/datatables.net-fixedheader-bs5/css/fixedHeader.bootstrap5.min.css',
'node_modules/datatables.net-buttons-bs5/css/buttons.bootstrap5.min.css',
'node_modules/datatables.net-select-bs5/css/select.bootstrap5.min.css',
'node_modules/dropzone/dist/dropzone.css',
'node_modules/select2/dist/css/select2.min.css',
'node_modules/flatpickr/dist/flatpickr.min.css'])
<style>
    .filter-row { background-color: #f9f9f9; }
    .form-filter { font-size: 12px; padding: 4px 6px; height: 30px; width: 100%; }
    .filter-row th { padding: 4px 6px !important; vertical-align: middle; }
    .dataTables_wrapper .dataTables_filter { float: right; }
    .dataTables_wrapper .dataTables_length { float: left; }
    .dataTables_wrapper .dataTables_paginate { float: right; }
    .filter-buttons { display: flex; gap: 5px; }
    .filter-buttons button { padding: 4px 8px; font-size: 12px; }
    .action-buttons { white-space: nowrap; }
    .leave-type-row { margin-bottom: 10px; }
    .leave-type-row:last-child { margin-bottom: 0; }
    .btn-remove-leave-type { margin-top: 27px; }
    .modal-dialog-scrollable .modal-body {
        max-height: calc(100vh - 200px);
        overflow-y: auto;
    }
    .select2-container--default .select2-selection--single {
        height: 38px;
        padding: 5px;
    }
    .assigned-badge {
        background-color: #28a745;
        color: white;
        padding: 3px 8px;
        border-radius: 12px;
        font-size: 11px;
        margin-left: 5px;
    }
    .readonly-field {
        background-color: #f8f9fa;
        cursor: not-allowed;
    }
</style>
@endsection

@section('content')
<div class="container-fluid">
    <div class="row mb-4">
        <div class="col-12">
            <div class="page-title-box d-flex align-items-center justify-content-between">
                <h2 class="mb-0">Leave Policies</h2>
                <div class="page-title-right d-flex align-items-center gap-3">
                    @php
                        $breadcrumb = getCurrentMenuBreadcrumb();
                    @endphp
                    @if(!empty($breadcrumb))
                        <nav aria-label="breadcrumb">
                            <ol class="breadcrumb mb-0 py-2">
                                @foreach($breadcrumb as $index => $item)
                                    @if($index === count($breadcrumb) - 1)
                                        <li class="breadcrumb-item active" aria-current="page">{{ $item }}</li>
                                    @else
                                        <li class="breadcrumb-item"><a href="javascript: void(0);">{{ $item }}</a></li>
                                    @endif
                                @endforeach
                            </ol>
                        </nav>
                    @endif
                </div>
            </div>
        </div>
    </div>
    <!-- end page title -->

    <div id="alert-container"></div>

    <div class="row">
        <div class="col-12">
            <div class="card">
                <div class="card-body">
                    <table id="policies_table" class="table table-bordered dt-responsive fs-13 w-100">
                        <thead>
                            <tr>
                                <th>ID</th>
                                <th>Policy Name</th>
                                <th>Description</th>
                                <th>Total Leaves</th>
                                <th>Business</th>
                                <th>Created At</th>
                                <th>Actions</th>
                            </tr>
                            <!-- Filter Row -->
                            <tr class="filter filter-row">
                                <td>
                                    <input type="text" class="form-control form-filter input-sm" name="id_leave_policy" placeholder="Search ID">
                                </td>
                                <td>
                                    <input type="text" class="form-control form-filter input-sm" name="policy_name" placeholder="Search Policy Name">
                                </td>
                                <td>
                                    <input type="text" class="form-control form-filter input-sm" name="policy_description" placeholder="Search Description">
                                </td>
                                <td>
                                    <input type="text" class="form-control form-filter input-sm" name="total_leaves" placeholder="Search Total Leaves" disabled>
                                </td>
                                <td>
                                    <input type="text" class="form-control form-filter input-sm" name="business_name" placeholder="Search Business">
                                </td>
                                <td>
                                    <input type="text" class="form-control form-filter input-sm date-picker" name="created_at" placeholder="Search Date">
                                </td>
                                <td>
                                    <div class="filter-buttons">
                                        <button class="btn btn-sm btn-ghost-secondary rounded-circle btn-icon filter-submit">
                                            <iconify-icon icon="mdi:magnify" class="fs-22"></iconify-icon>
                                        </button>
                                        <button class="btn btn-sm btn-ghost-secondary rounded-circle btn-icon filter-cancel">
                                            <iconify-icon icon="mdi:refresh" class="fs-22"></iconify-icon>
                                        </button>
                                    </div>
                                </td>
                            </tr>
                        </thead>
                        <tbody></tbody>
                    </table>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Add/Edit Policy Modal -->
<div class="modal fade" id="policyModal" tabindex="-1" aria-labelledby="policyModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-lg modal-dialog-scrollable">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="policyModalLabel">Add New Leave Policy</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
                <form id="policyForm">
                    @csrf
                    <input type="hidden" name="id_leave_policy" id="id_leave_policy">
                    <input type="hidden" name="business_id" id="business_id" value="{{ session('business_id') }}">
                    
                    <div class="row">
                        <div class="col-md-12 mb-3">
                            <label for="policy_name" class="form-label">Policy Name <span class="text-danger">*</span></label>
                            <input type="text" class="form-control" id="policy_name" name="policy_name" required>
                            <div class="invalid-feedback" id="policy_name_error"></div>
                        </div>
                        
                        <div class="col-md-12 mb-3">
                            <label for="policy_description" class="form-label">Policy Description</label>
                            <textarea class="form-control" id="policy_description" name="policy_description" rows="2"></textarea>
                            <div class="invalid-feedback" id="policy_description_error"></div>
                        </div>
                    </div>
                    
                    <!-- Add New Leave Type Section -->
                    <div class="row mb-3 border-top pt-3">
                        <div class="col-12">
                            <h5 class="mb-3">Create New Leave Type</h5>
                            <div class="row g-2">
                                <div class="col-md-8">
                                    <input type="text" class="form-control" id="new_leave_type" placeholder="Enter new leave type name">
                                </div>
                                <div class="col-md-4">
                                    <button type="button" class="btn btn-sm btn-outline-success w-100" id="add-new-leave-type-btn">
                                        <iconify-icon icon="mdi:plus" class="me-1"></iconify-icon> Add Type
                                    </button>
                                </div>
                            </div>
                            <small class="text-muted">Create a new leave type if it doesn't exist in the dropdown</small>
                        </div>
                    </div>
                    
                    <!-- Leave Types Section -->
                    <div class="row mb-3">
                        <div class="col-12">
                            <h5 class="mb-3">Leave Types & Allowed Count <span class="text-danger">*</span></h5>
                            <small class="text-muted">Enter the allowed leave count as per the yearly policy.</small>
                            <div id="leave-types-container">
                                <!-- Leave types will be added dynamically here -->
                            </div>
                            <button type="button" class="btn btn-sm btn-outline-primary mt-2" id="add-leave-type-btn">
                                <iconify-icon icon="mdi:plus" class="me-1"></iconify-icon> Add Leave Type Row
                            </button>
                        </div>
                    </div>
                    
                    <!-- Assigned Warning (for edit mode) -->
                    <div class="row" id="assigned-warning" style="display: none;">
                        <div class="col-12">
                            <div class="alert alert-warning mb-0">
                                <iconify-icon icon="mdi:alert-circle" class="me-2"></iconify-icon>
                                <strong>Note:</strong> This policy is assigned to staff members. You can view details but cannot edit.
                            </div>
                        </div>
                    </div>
                </form>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                <button type="button" class="btn btn-primary" id="savePolicyBtn">Save Policy</button>
            </div>
        </div>
    </div>
</div>

<!-- View Policy Details Modal -->
<div class="modal fade" id="viewPolicyModal" tabindex="-1" aria-labelledby="viewPolicyModalLabel" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="viewPolicyModalLabel">Policy Details</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
                <table class="table table-bordered">
                    <tr>
                        <th width="30%">Policy Name:</th>
                        <td id="view-policy-name"></td>
                    </tr>
                    <tr>
                        <th>Description:</th>
                        <td id="view-policy-description"></td>
                    </tr>
                    <tr>
                        <th>Total Leaves:</th>
                        <td id="view-total-leaves"></td>
                    </tr>
                    <tr>
                        <th>Business:</th>
                        <td id="view-business-name"></td>
                    </tr>
                    <tr>
                        <th>Created At:</th>
                        <td id="view-created-at"></td>
                    </tr>
                    <tr id="view-assigned-row" style="display: none;">
                        <th>Status:</th>
                        <td><span class="assigned-badge">Assigned to Staff</span></td>
                    </tr>
                </table>
                
                <h6 class="mt-3 mb-2">Leave Types:</h6>
                <table class="table table-sm table-bordered">
                    <thead>
                        <tr>
                            <th>Leave Type</th>
                            <th>Allowed Count</th>
                        </tr>
                    </thead>
                    <tbody id="view-leave-types">
                        <!-- Leave types will be populated here -->
                    </tbody>
                </table>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
            </div>
        </div>
    </div>
</div>

@endsection

@section('scripts')
@vite(['resources/js/app.js'])
<script>
    document.addEventListener('DOMContentLoaded', function() {
        initializeDatePickers();
        initializeSelect2();
        initializeLeaveTypeTemplate();
        initializePoliciesTable();
        setupEventHandlers();
    });
    
    let policiesTable = null;
    let policyModal = null;
    let viewPolicyModal = null;
    let leaveTypeTemplate = null;
    let leaveTypesData = @json($leaveTypes ?? []);
    let selectedLeaveTypes = new Set();
    
    function initializeDatePickers() {
        if (typeof flatpickr !== 'undefined') {
            flatpickr('.date-picker', {
                dateFormat: 'Y-m-d',
                allowInput: true
            });
        }
    }
    
    function initializeSelect2() {
        if (typeof $().select2 === 'function') {
            $('.select2').select2({
                width: '100%',
                placeholder: 'Select Option',
                allowClear: true
            });
        }
    }
    
    function initializePoliciesTable() {
        if (typeof $.fn.DataTable === 'undefined') {
            console.error('DataTables is not loaded');
            return;
        }
        
        policiesTable = $('#policies_table').DataTable({
            scrollX: true, 
            orderCellsTop: true,           
            scrollCollapse: true,
            fixedHeader: {
                header: true,
                footer: false
            },
            processing: true,
            serverSide: true,
            stateSave: true,
            responsive: false,
            ajax: {
                url: "{{ route('hrm.get_leave_policies') }}",
                type: 'POST',
                contentType: 'application/json',
                headers: {
                    'X-CSRF-TOKEN': '{{ csrf_token() }}'
                },
                dataType: 'json',
                data: function (d) {
                    d.id_leave_policy = $('input[name="id_leave_policy"]').val();
                    d.policy_name = $('input[name="policy_name"]').val();
                    d.policy_description = $('input[name="policy_description"]').val();
                    d.business_name = $('input[name="business_name"]').val();
                    d.created_at = $('input[name="created_at"]').val();
                    
                    return JSON.stringify(d);
                },
                error: function (xhr, error, thrown) {
                    console.error('DataTables AJAX error:', error, thrown, xhr.responseText);
                    showAlert('error', 'Failed to load data. Please try again.');
                }
            },
            "language": {
                "paginate": {
                    "previous": "<i class='ti ti-chevron-left'></i>",
                    "next": "<i class='ti ti-chevron-right'></i>"
                }
            },
            lengthChange: true,
            pageLength: 10,
            searching: false,
            info: true,
            paging: true,
            buttons: [
                { extend: "copy",  className: "btn btn-sm btn-soft-primary" },
                { extend: "csv",   className: "btn btn-sm btn-soft-primary" },
                { extend: "print", className: "btn btn-sm btn-soft-primary" },
                { extend: "pdf", className: "btn btn-sm btn-soft-primary" }
            ],
            layout: {
                topStart: 'buttons',
                topEnd: function() {
                    return '<div class="d-flex justify-content-end"><button type="button" class="btn btn-outline-primary" id="addPolicyBtn"><iconify-icon icon="mdi:plus" class="me-1"></iconify-icon> Add New Policy</button></div>';
                },
                bottomStart: {
                    pageLength: {
                        menu: [10, 25, 50, 100]
                    },
                    info: {}
                },
                bottomEnd: 'paging'
            },
            columns: [
                { data: 'id_leave_policy', name: 'id_leave_policy', orderable: true, searchable: true },
                { data: 'policy_name', name: 'policy_name', orderable: true, searchable: true },
                { data: 'policy_description', name: 'policy_description', orderable: true, searchable: true },
                { 
                    data: 'total_leaves', 
                    name: 'total_leaves', 
                    orderable: true, 
                    searchable: false,
                    render: function(data, type, row) {
                        return data || '0';
                    }
                },
                { 
                    data: 'business_name', 
                    name: 'business_name', 
                    orderable: true, 
                    searchable: true,
                    render: function(data, type, row) {
                        return data || '-';
                    }
                },
                { 
                    data: 'created_at', 
                    name: 'created_at', 
                    orderable: true, 
                    searchable: true,
                    render: function(data, type, row) {
                        return data || '-';
                    }
                },
                { 
                    data: null, 
                    name: 'actions', 
                    orderable: false, 
                    searchable: false,
                    className: 'action-buttons',
                    render: function(data, type, row) {
                        let buttons = '';
                        
                        if (!row.is_assigned) {
                            buttons += `<button class="btn btn-sm btn-ghost-secondary rounded-circle btn-icon edit-policy-btn" data-id="${row.id_leave_policy}" title="Edit Policy">
                                <iconify-icon icon="line-md:edit-twotone" class="fs-22"></iconify-icon>
                            </button>`;
                            
                            buttons += `<button class="btn btn-sm btn-ghost-danger rounded-circle btn-icon delete-policy-btn" data-id="${row.id_leave_policy}" title="Delete Policy">
                                <iconify-icon icon="mdi:delete-outline" class="fs-22"></iconify-icon>
                            </button>`;
                        } else {
                            buttons += `<span class="assigned-badge">Assigned</span>`;
                        }
                        
                        buttons += `<button class="btn btn-sm btn-ghost-secondary rounded-circle btn-icon view-policy-btn" data-id="${row.id_leave_policy}" title="View Details">
                            <iconify-icon icon="mdi:eye-outline" class="fs-22"></iconify-icon>
                        </button>`;
                        
                        return buttons;
                    }
                }
            ],
            initComplete: function(settings, json) {
                var api = this.api();
                var filterRow = $(api.table().header()).find('tr.filter');
                filterRow.show();
            }
        });
    }
    
    function initializeLeaveTypeTemplate() {
        let optionsHtml = '<option value="">Select Leave Type</option>';
        
        if (leaveTypesData && leaveTypesData.length > 0) {
            leaveTypesData.forEach(function(type) {
                optionsHtml += `<option value="${type.id_leave_type}" data-name="${type.leave_type}">${type.leave_type}</option>`;
            });
        } else {
            optionsHtml += '<option value="" disabled>No leave types available</option>';
        }
        
        leaveTypeTemplate = `
            <div class="row leave-type-row" data-index="{index}">
                <div class="col-md-6">
                    <label class="form-label">Leave Type <span class="text-danger">*</span></label>
                    <select class="form-control leave-type-select" name="leave_types[{index}][leave_type_id]" required>
                        ${optionsHtml}
                    </select>
                    <input type="hidden" name="leave_types[{index}][leave_type_name]" class="leave-type-name">
                </div>
                <div class="col-md-4">
                    <label class="form-label">Allowed Count <span class="text-danger">*</span></label>
                    <input type="number" class="form-control leave-type-count" name="leave_types[{index}][allowed_count]" min="1" required>
                </div>
                <div class="col-md-2">
                    <button type="button" class="btn btn-sm btn-danger btn-remove-leave-type" style="margin-top: 27px;">
                        <iconify-icon icon="mdi:delete"></iconify-icon>
                    </button>
                </div>
            </div>
        `;
    }
    
    function setupEventHandlers() {
        // Initialize Bootstrap modals
        const policyModalEl = document.getElementById('policyModal');
        const viewPolicyModalEl = document.getElementById('viewPolicyModal');
        
        if (policyModalEl) {
            policyModal = new bootstrap.Modal(policyModalEl);
        }
        if (viewPolicyModalEl) {
            viewPolicyModal = new bootstrap.Modal(viewPolicyModalEl);
        }
        
        // Add policy button
        const addPolicyBtn = document.getElementById('addPolicyBtn');
        if (addPolicyBtn) {
            addPolicyBtn.addEventListener('click', function() {
                openPolicyModal();
            });
        }
        
        // Save policy button
        const savePolicyBtn = document.getElementById('savePolicyBtn');
        if (savePolicyBtn) {
            savePolicyBtn.addEventListener('click', function() {
                savePolicy();
            });
        }

        // Add leave type button
        const addLeaveTypeBtn = document.getElementById('add-leave-type-btn');
        if (addLeaveTypeBtn) {
            addLeaveTypeBtn.addEventListener('click', function() {
                const rows = document.querySelectorAll('#leave-types-container .leave-type-row');
                
                if (rows.length > 0) {
                    const lastRow = rows[rows.length - 1];
                    const lastSelect = lastRow.querySelector('.leave-type-select');
                    const lastCount = lastRow.querySelector('input[type="number"].leave-type-count');
                    
                    let hasError = false;
                    
                    if (lastSelect) lastSelect.classList.remove('is-invalid');
                    if (lastCount) lastCount.classList.remove('is-invalid');
                    
                    if (!lastSelect || !lastSelect.value) {
                        if (lastSelect) {
                            lastSelect.classList.add('is-invalid');
                            lastSelect.focus();
                        }
                        hasError = true;
                    }
                    
                    if (!lastCount || !lastCount.value || parseInt(lastCount.value) < 1) {
                        if (lastCount) {
                            lastCount.classList.add('is-invalid');
                            if (!lastSelect || lastSelect.value) {
                                lastCount.focus();
                            }
                        }
                        hasError = true;
                    }
                    
                    if (hasError) {
                        Swal.fire({
                            icon: 'warning',
                            title: 'Validation Error',
                            text: 'Please fill in the current leave type row before adding a new one.',
                            customClass: {
                                confirmButton: "btn btn-danger me-2", 
                                cancelButton: "btn btn-warning"
                            },
                        });
                        return;
                    }
                }
                
                const availableOptions = getAvailableLeaveTypeOptions();
                if (availableOptions.length === 0) {
                    Swal.fire({
                        icon: 'info',
                        title: 'No More Leave Types',
                        text: 'All available leave types have been added.',
                        customClass: {
                            confirmButton: "btn btn-danger me-2", 
                            cancelButton: "btn btn-warning"
                        },
                    });
                    return;
                }
                
                addLeaveTypeRow();
            });
        }
        
        // Add new leave type button
        const addNewLeaveTypeBtn = document.getElementById('add-new-leave-type-btn');
        if (addNewLeaveTypeBtn) {
            addNewLeaveTypeBtn.addEventListener('click', function() {
                addNewLeaveType();
            });
        }
        
        // Enter key for new leave type input
        const newLeaveTypeInput = document.getElementById('new_leave_type');
        if (newLeaveTypeInput) {
            newLeaveTypeInput.addEventListener('keyup', function(e) {
                if (e.key === 'Enter') {
                    addNewLeaveType();
                }
            });
        }
        
        // Delegated events
        document.addEventListener('click', function(e) {
            if (e.target.closest('.edit-policy-btn')) {
                const btn = e.target.closest('.edit-policy-btn');
                const id = btn.getAttribute('data-id');
                editPolicy(id);
            }
            
            if (e.target.closest('.view-policy-btn')) {
                const btn = e.target.closest('.view-policy-btn');
                const id = btn.getAttribute('data-id');
                viewPolicyDetails(id);
            }
            
            if (e.target.closest('.delete-policy-btn')) {
                const btn = e.target.closest('.delete-policy-btn');
                const id = btn.getAttribute('data-id');
                deletePolicy(id);
            }
            
            if (e.target.closest('.btn-remove-leave-type')) {
                const btn = e.target.closest('.btn-remove-leave-type');
                const row = btn.closest('.leave-type-row');
                if (row) {
                    removeLeaveTypeRow(row);
                }
            }
            
            if (e.target.closest('.filter-submit')) {
                e.preventDefault();
                refreshData();
            }
            
            if (e.target.closest('.filter-cancel')) {
                e.preventDefault();
                document.querySelectorAll('.form-filter').forEach(input => {
                    input.value = '';
                });
                refreshData();
            }
        });
        
        document.addEventListener('change', function(e) {
            if (e.target.classList.contains('leave-type-select')) {
                const select = e.target;
                const row = select.closest('.leave-type-row');
                const hiddenInput = row.querySelector('.leave-type-name');
                const selectedOption = select.options[select.selectedIndex];
                
                if (hiddenInput && selectedOption) {
                    hiddenInput.value = selectedOption.getAttribute('data-name');
                }
                
                updateSelectedLeaveTypes();
                updateLeaveTypeDropdowns();
            }
        });
        
        document.addEventListener('keyup', function(e) {
            if (e.target.classList.contains('form-filter') && e.key === 'Enter') {
                refreshData();
            }
        });
        
        if (policyModalEl) {
            policyModalEl.addEventListener('hidden.bs.modal', function () {
                resetModalForm();
                clearValidationErrors();
                selectedLeaveTypes.clear();
            });
        }
    }

    function refreshData() {
        if (policiesTable) {
            policiesTable.ajax.reload(null, false);
        }
    }
    
    function getAvailableLeaveTypeOptions() {
        if (!leaveTypesData || leaveTypesData.length === 0) {
            return [];
        }
        
        const allOptions = leaveTypesData.map(type => ({
            value: type.id_leave_type.toString(),
            text: type.leave_type,
            name: type.leave_type
        }));
        
        const currentlySelected = new Set();
        document.querySelectorAll('.leave-type-select').forEach(select => {
            if (select.value) {
                currentlySelected.add(select.value);
            }
        });
        
        return allOptions.filter(option => !currentlySelected.has(option.value));
    }

    function openPolicyModal() {
        if (!leaveTypesData || leaveTypesData.length === 0) {
            refreshLeaveTypeDropdowns().then(() => {
                if (!leaveTypesData || leaveTypesData.length === 0) {
                    Swal.fire({
                        icon: 'warning',
                        title: 'No Leave Types',
                        text: 'No leave types available. Please create a leave type first.',
                        customClass: {
                            confirmButton: "btn btn-danger me-2", 
                            cancelButton: "btn btn-warning"
                        },
                    });
                    return;
                }
                initializeLeaveTypeTemplate();
                continueOpeningModal();
            });
        } else {
            continueOpeningModal();
        }
        
        function continueOpeningModal() {
            resetModalForm();
            clearValidationErrors();
            document.getElementById('assigned-warning').style.display = 'none';
            document.getElementById('savePolicyBtn').style.display = 'block';
            
            document.getElementById('policyModalLabel').textContent = 'Add New Leave Policy';
            
            addLeaveTypeRow();
            
            if (policyModal) {
                policyModal.show();
            }
        }
    }

    function addLeaveTypeRow() {
        const container = document.getElementById('leave-types-container');
        if (!container || !leaveTypeTemplate) return;
        
        const currentCount = container.querySelectorAll('.leave-type-row').length;
        
        const newRowHtml = leaveTypeTemplate.replace(/{index}/g, currentCount);
        const tempDiv = document.createElement('div');
        tempDiv.innerHTML = newRowHtml.trim();
        
        const newRow = tempDiv.firstChild;
        container.appendChild(newRow);
        
        const newSelect = newRow.querySelector('.leave-type-select');
        if (newSelect) {
            populateLeaveTypeDropdown(newSelect);
            
            if (typeof $.fn.select2 !== 'undefined') {
                $(newSelect).select2({
                    width: '100%',
                    dropdownParent: $('#policyModal'),
                    templateResult: function(data) {
                        if (data.element && data.element.disabled) {
                            return null;
                        }
                        return data.text;
                    }
                }).on('select2:open', function() {
                    updateLeaveTypeDropdowns();
                }).on('select2:select', function(e) {
                    const row = $(this).closest('.leave-type-row');
                    const selectedOption = $(this).find('option:selected');
                    const hiddenInput = row.find('.leave-type-name');
                    if (hiddenInput.length > 0 && selectedOption.length > 0) {
                        hiddenInput.val(selectedOption.data('name'));
                    }
                    updateSelectedLeaveTypes();
                    updateLeaveTypeDropdowns();
                });
            }
        }
        updateLeaveTypeDropdowns();
    }

    function removeLeaveTypeRow(row) {
        const select = row.querySelector('.leave-type-select');
        const removedValue = select.value;
        
        row.remove();
        reindexLeaveTypes();
        if (removedValue) {
            selectedLeaveTypes.delete(removedValue);
        }
        
        updateLeaveTypeDropdowns();
    }

    function updateSelectedLeaveTypes() {
        selectedLeaveTypes.clear();
        document.querySelectorAll('.leave-type-select').forEach(select => {
            if (select.value) {
                selectedLeaveTypes.add(select.value);
            }
        });
    }

    function updateLeaveTypeDropdowns() {
        updateSelectedLeaveTypes();
        
        document.querySelectorAll('.leave-type-select').forEach(select => {
            const currentValue = select.value;
            select.querySelectorAll('option').forEach(option => {
                if (option.value) {
                    if (option.value !== currentValue && selectedLeaveTypes.has(option.value)) {
                        option.disabled = true;
                        option.style.display = 'none';
                    } else {
                        option.disabled = false;
                        option.style.display = '';
                    }
                }
            });

            if (currentValue) {
                const selectedOption = select.querySelector(`option[value="${currentValue}"]`);
                if (selectedOption && selectedOption.disabled) {
                    select.value = '';
                    const event = new Event('change');
                    select.dispatchEvent(event);
                }
            }
            
            if ($(select).hasClass('select2-hidden-accessible')) {
                $(select).trigger('change.select2');
            }
        });
    }

    function addNewLeaveType() {
        const input = document.getElementById('new_leave_type');
        const leaveTypeName = input.value.trim();
        
        if (!leaveTypeName) {
            Swal.fire({
                icon: 'warning',
                title: 'Validation Error',
                text: 'Please enter a leave type name.',
                customClass: {
                    confirmButton: "btn btn-danger me-2", 
                    cancelButton: "btn btn-warning"
                },
            });
            return;
        }
        
        Swal.fire({
            title: 'Creating...',
            text: 'Please wait',
            allowOutsideClick: false,
            customClass: {
                confirmButton: "btn btn-danger me-2", 
                cancelButton: "btn btn-warning"
            },
            didOpen: () => {
                Swal.showLoading();
            }
        });

        const data = {
            _token: '{{ csrf_token() }}',
            leave_type: leaveTypeName
        };
        
        fetch("{{ route('hrm.create_leave_type') }}", {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'X-CSRF-TOKEN': '{{ csrf_token() }}'
            },
            body: JSON.stringify(data)
        })
        .then(response => response.json())
        .then(response => {
            Swal.close();
            
            if (response.success) {
                input.value = '';
                refreshLeaveTypeDropdowns();
                
                Swal.fire({
                    icon: 'success',
                    title: 'Success!',
                    text: 'Leave type created successfully.',
                    timer: 2000,
                    showConfirmButton: false,
                    customClass: {
                        confirmButton: "btn btn-danger me-2", 
                        cancelButton: "btn btn-warning"
                    },
                });
                
            } else {
                Swal.fire({
                    icon: 'error',
                    title: 'Error',
                    text: response.message || 'Failed to create leave type.',
                    customClass: {
                        confirmButton: "btn btn-danger me-2", 
                        cancelButton: "btn btn-warning"
                    },
                });
            }
        })
        .catch(error => {
            Swal.close();
            console.error('Error creating leave type:', error);
            Swal.fire({
                icon: 'error',
                title: 'Error',
                text: 'Error creating leave type. Please try again.',
                customClass: {
                    confirmButton: "btn btn-danger me-2", 
                    cancelButton: "btn btn-warning"
                },
            });
        });
    }

    function refreshLeaveTypeDropdowns() {
        return fetch("{{ route('hrm.leave_policies') }}/leave-types", {
            method: 'GET',
            headers: {
                'Accept': 'application/json',
            }
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                leaveTypesData = data.leaveTypes;
                document.querySelectorAll('.leave-type-select').forEach(select => {
                    populateLeaveTypeDropdown(select);
                });
                
                initializeLeaveTypeTemplate();
                
                return true;
            } else {
                console.error('Failed to refresh leave types:', data.message);
                return false;
            }
        })
        .catch(error => {
            console.error('Error refreshing leave types:', error);
            return false;
        });
    }
   
    function editPolicy(id) {
        Swal.fire({
            title: 'Loading...',
            text: 'Please wait',
            allowOutsideClick: false,
            customClass: {
                confirmButton: "btn btn-danger me-2", 
                cancelButton: "btn btn-warning"
            },
            didOpen: () => {
                Swal.showLoading();
            }
        });
        
        fetch("{{ route('hrm.get_leave_policy', '') }}/" + id, {
            method: 'GET',
            headers: {
                'Accept': 'application/json',
            }
        })
        .then(response => response.json())
        .then(response => {
            Swal.close();
            
            if (response.success) {
                resetModalForm();
                selectedLeaveTypes.clear();
                document.getElementById('id_leave_policy').value = response.data.id_leave_policy;
                document.getElementById('policy_name').value = response.data.policy_name;
                document.getElementById('policy_description').value = response.data.policy_description || '';

                document.getElementById('policyModalLabel').textContent = 'Edit Leave Policy';
                if (response.is_assigned) {
                    document.getElementById('assigned-warning').style.display = 'block';
                    document.getElementById('savePolicyBtn').style.display = 'none';
                    
                    document.getElementById('policy_name').setAttribute('readonly', true);
                    document.getElementById('policy_name').classList.add('readonly-field');
                    document.getElementById('policy_description').setAttribute('readonly', true);
                    document.getElementById('policy_description').classList.add('readonly-field');
                    document.getElementById('new_leave_type').setAttribute('readonly', true);
                    document.getElementById('new_leave_type').classList.add('readonly-field');
                    document.getElementById('add-new-leave-type-btn').style.display = 'none';
                    document.getElementById('add-leave-type-btn').style.display = 'none';
                } else {
                    document.getElementById('assigned-warning').style.display = 'none';
                    document.getElementById('savePolicyBtn').style.display = 'block';
                    document.getElementById('policy_name').removeAttribute('readonly');
                    document.getElementById('policy_name').classList.remove('readonly-field');
                    document.getElementById('policy_description').removeAttribute('readonly');
                    document.getElementById('policy_description').classList.remove('readonly-field');
                    document.getElementById('new_leave_type').removeAttribute('readonly');
                    document.getElementById('new_leave_type').classList.remove('readonly-field');
                    document.getElementById('add-new-leave-type-btn').style.display = 'block';
                    document.getElementById('add-leave-type-btn').style.display = 'block';
                }
                
                const container = document.getElementById('leave-types-container');
                container.innerHTML = '';
                
                if (response.data.definitions && response.data.definitions.length > 0) {
                    response.data.definitions.forEach((definition, index) => {
                        const rowHtml = leaveTypeTemplate.replace(/{index}/g, index);
                        const tempDiv = document.createElement('div');
                        tempDiv.innerHTML = rowHtml.trim();
                        const row = tempDiv.firstChild;
                        
                        const select = row.querySelector('.leave-type-select');
                        const hiddenInput = row.querySelector('.leave-type-name');
                        const countInput = row.querySelector('input[name="leave_types[' + index + '][allowed_count]"]');
                        const removeBtn = row.querySelector('.btn-remove-leave-type');
                        
                        if (select) {
                            select.value = definition.leave_type_id;
                            selectedLeaveTypes.add(definition.leave_type_id);
                            const event = new Event('change');
                            select.dispatchEvent(event);
                        }
                        
                        if (hiddenInput) {
                            hiddenInput.value = definition.leave_type_name || definition.leave_type;
                        }
                        
                        if (countInput) {
                            countInput.value = definition.allowed_count;
                            if (response.is_assigned) {
                                countInput.setAttribute('readonly', true);
                                countInput.classList.add('readonly-field');
                                if (removeBtn) {
                                    removeBtn.style.display = 'none';
                                }
                            }
                        }
                        
                        container.appendChild(row);
                        if (select && typeof $.fn.select2 !== 'undefined') {
                            $(select).select2({
                                width: '100%',
                                dropdownParent: $('#policyModal')
                            });
                        }
                    });
                } else {
                    addLeaveTypeRow();
                }
                
                updateLeaveTypeDropdowns();
                
                clearValidationErrors();
                
                if (policyModal) {
                    policyModal.show();
                }
                
            } else {
                Swal.fire({
                    icon: 'error',
                    title: 'Error',
                    text: response.message || 'Failed to load policy data.',
                    customClass: {
                        confirmButton: "btn btn-danger me-2", 
                        cancelButton: "btn btn-warning"
                    }
                });
            }
        })
        .catch(error => {
            Swal.close();
            console.error('Error:', error);
            Swal.fire({
                icon: 'error',
                title: 'Error',
                text: 'Error loading policy data. Please try again.',
                customClass: {
                    confirmButton: "btn btn-danger me-2", 
                    cancelButton: "btn btn-warning"
                },
            });
        });
    }

    function viewPolicyDetails(id) {
        Swal.fire({
            title: 'Loading...',
            text: 'Please wait',
            allowOutsideClick: false,
            customClass: {
                confirmButton: "btn btn-danger me-2", 
                cancelButton: "btn btn-warning"
            },
            didOpen: () => {
                Swal.showLoading();
            }
        });
        
        fetch("{{ route('hrm.get_leave_policy', '') }}/" + id, {
            method: 'GET',
            headers: {
                'Accept': 'application/json',
            }
        })
        .then(response => response.json())
        .then(response => {
            Swal.close();
            
            if (response.success) {
                document.getElementById('view-policy-name').textContent = response.data.policy_name;
                document.getElementById('view-policy-description').textContent = response.data.policy_description || 'N/A';
                document.getElementById('view-business-name').textContent = response.data.business_name || 'N/A';
                document.getElementById('view-created-at').textContent = response.data.created_at ? new Date(response.data.created_at).toLocaleDateString('en-GB') : 'N/A';
                
                if (response.is_assigned) {
                    document.getElementById('view-assigned-row').style.display = '';
                } else {
                    document.getElementById('view-assigned-row').style.display = 'none';
                }
                
                let totalLeaves = 0;
                const leaveTypesContainer = document.getElementById('view-leave-types');
                leaveTypesContainer.innerHTML = '';
                
                if (response.data.definitions && response.data.definitions.length > 0) {
                    response.data.definitions.forEach(definition => {
                        totalLeaves += parseInt(definition.allowed_count);
                        
                        const row = document.createElement('tr');
                        row.innerHTML = `
                            <td>${definition.leave_type_name || definition.leave_type}</td>
                            <td>${definition.allowed_count}</td>
                        `;
                        leaveTypesContainer.appendChild(row);
                    });
                }
                
                document.getElementById('view-total-leaves').textContent = totalLeaves;
                if (viewPolicyModal) {
                    viewPolicyModal.show();
                }
                
            } else {
                Swal.fire({
                    icon: 'error',
                    title: 'Error',
                    text: response.message || 'Failed to load policy details.',
                    customClass: {
                        confirmButton: "btn btn-danger me-2", 
                        cancelButton: "btn btn-warning"
                    }
                });
            }
        })
        .catch(error => {
            Swal.close();
            console.error('Error:', error);
            Swal.fire({
                icon: 'error',
                title: 'Error',
                text: 'Error loading policy details. Please try again.',
                customClass: {
                    confirmButton: "btn btn-danger me-2", 
                    cancelButton: "btn btn-warning"
                },
            });
        });
    }

    function deletePolicy(id) {
        Swal.fire({
            title: 'Are you sure?',
            text: "This action cannot be undone!",
            icon: 'warning',
            showCancelButton: true,
            confirmButtonText: 'Yes, delete it!',
            cancelButtonText: 'Cancel',
            customClass: {
                confirmButton: "btn btn-danger me-2", 
                cancelButton: "btn btn-warning"
            },
        }).then((result) => {
            if (result.isConfirmed) {
                Swal.fire({
                    title: 'Deleting...',
                    text: 'Please wait',
                    allowOutsideClick: false,
                    customClass: {
                        confirmButton: "btn btn-danger me-2", 
                        cancelButton: "btn btn-warning"
                    },
                    didOpen: () => {
                        Swal.showLoading();
                    }
                });
                
                fetch("{{ route('hrm.delete_leave_policy', '') }}/" + id, {
                    method: 'DELETE',
                    headers: {
                        'Accept': 'application/json',
                        'X-CSRF-TOKEN': '{{ csrf_token() }}'
                    }
                })
                .then(response => response.json())
                .then(response => {
                    Swal.close();
                    
                    if (response.success) {
                        Swal.fire({
                            icon: 'success',
                            title: 'Deleted!',
                            text: response.message,
                            timer: 2000,
                            showConfirmButton: false,
                            customClass: {
                                confirmButton: "btn btn-danger me-2", 
                                cancelButton: "btn btn-warning"
                            },
                        }).then(() => {
                            refreshData();
                        });
                    } else {
                        Swal.fire({
                            icon: 'error',
                            title: 'Error',
                            text: response.message,
                            customClass: {
                                confirmButton: "btn btn-danger me-2", 
                                cancelButton: "btn btn-warning"
                            },
                        });
                    }
                })
                .catch(error => {
                    Swal.close();
                    console.error('Error:', error);
                    Swal.fire({
                        icon: 'error',
                        title: 'Error',
                        text: 'Error deleting policy. Please try again.',
                        customClass: {
                            confirmButton: "btn btn-danger me-2", 
                            cancelButton: "btn btn-warning"
                        },
                    });
                });
            }
        });
    }

    function savePolicy() {
        clearValidationErrors();
        
        const form = document.getElementById('policyForm');
        if (!form) return;
        
        const policyName = document.getElementById('policy_name');
        const leaveTypeRows = document.querySelectorAll('#leave-types-container .leave-type-row');
        
        let isValid = true;
        
        if (!policyName || !policyName.value.trim()) {
            policyName.classList.add('is-invalid');
            document.getElementById('policy_name_error').textContent = 'Policy name is required.';
            isValid = false;
        }
        
        if (leaveTypeRows.length === 0) {
            showAlert('error', 'Please add at least one leave type.');
            isValid = false;
        } else {
            leaveTypeRows.forEach((row, index) => {
                const select = row.querySelector('.leave-type-select');
                const countInput = row.querySelector('.leave-type-count');
                
                if (!select || !select.value) {
                    select.classList.add('is-invalid');
                    isValid = false;
                }
                
                if (!countInput || !countInput.value || parseInt(countInput.value) < 1) {
                    countInput.classList.add('is-invalid');
                    isValid = false;
                }
            });
        }
        
        if (!isValid) {
            Swal.fire({
                icon: 'warning',
                title: 'Validation Error',
                text: 'Please fill in all required fields correctly.',
                customClass: {
                    confirmButton: "btn btn-danger me-2", 
                    cancelButton: "btn btn-warning"
                },
            });
            return;
        }
        
        const leaveTypes = [];
        leaveTypeRows.forEach((row, index) => {
            const select = row.querySelector('.leave-type-select');
            const countInput = row.querySelector('.leave-type-count');
            const hiddenInput = row.querySelector('.leave-type-name');
            
            if (select && countInput && hiddenInput) {
                const selectedOption = select.options[select.selectedIndex];
                const leaveTypeName = selectedOption ? selectedOption.textContent : hiddenInput.value;
                
                leaveTypes.push({
                    leave_type_id: parseInt(select.value),
                    leave_type_name: leaveTypeName,
                    allowed_count: parseInt(countInput.value)
                });
            }
        });
        
        const data = {
            _token: '{{ csrf_token() }}',
            id_leave_policy: document.getElementById('id_leave_policy').value,
            policy_name: document.getElementById('policy_name').value,
            policy_description: document.getElementById('policy_description').value,
            leave_types: leaveTypes
        };
        
        Swal.fire({
            title: 'Saving...',
            text: 'Please wait',
            allowOutsideClick: false,
            customClass: {
                confirmButton: "btn btn-danger me-2", 
                cancelButton: "btn btn-warning"
            },
            didOpen: () => {
                Swal.showLoading();
            }
        });

        fetch("{{ route('hrm.save_leave_policy') }}", {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'X-CSRF-TOKEN': '{{ csrf_token() }}'
            },
            body: JSON.stringify(data)
        })
        .then(response => response.json())
        .then(response => {
            Swal.close();
            
            if (response.success) {
                if (policyModal) {
                    policyModal.hide();
                }
                resetModalForm();
                Swal.fire({
                    icon: 'success',
                    title: 'Success!',
                    text: response.message,
                    timer: 2000,
                    showConfirmButton: false,
                    customClass: {
                        confirmButton: "btn btn-danger me-2", 
                        cancelButton: "btn btn-warning"
                    },
                }).then(() => {
                    refreshData();
                });
                
            } else {
                Swal.fire({
                    icon: 'error',
                    title: 'Error',
                    text: response.message,
                    customClass: {
                        confirmButton: "btn btn-danger me-2", 
                        cancelButton: "btn btn-warning"
                    },
                });
            }
        })
        .catch(error => {
            Swal.close();
            console.error('Save error:', error);
            
            Swal.fire({
                icon: 'error',
                title: 'Error',
                text: 'Error saving policy. Please try again.',
                customClass: {
                    confirmButton: "btn btn-danger me-2", 
                    cancelButton: "btn btn-warning"
                },
            }).then(() => {
                if (policyModal) {
                    policyModal.hide();
                }
                resetModalForm();
            });
        });
    }

    function resetModalForm() {
        const form = document.getElementById('policyForm');
        if (form) {
            form.reset();
        }
        
        document.getElementById('id_leave_policy').value = '';
        document.getElementById('leave-types-container').innerHTML = '';
        document.getElementById('new_leave_type').value = '';
        
        const addBtn = document.getElementById('add-leave-type-btn');
        if (addBtn) {
            addBtn.style.display = 'block';
        }
        
        const addNewBtn = document.getElementById('add-new-leave-type-btn');
        if (addNewBtn) {
            addNewBtn.style.display = 'block';
        }
        
        const policyName = document.getElementById('policy_name');
        const policyDescription = document.getElementById('policy_description');
        const newLeaveTypeInput = document.getElementById('new_leave_type');
        
        if (policyName) {
            policyName.removeAttribute('readonly');
            policyName.classList.remove('readonly-field');
        }
        if (policyDescription) {
            policyDescription.removeAttribute('readonly');
            policyDescription.classList.remove('readonly-field');
        }
        if (newLeaveTypeInput) {
            newLeaveTypeInput.removeAttribute('readonly');
            newLeaveTypeInput.classList.remove('readonly-field');
        }

        selectedLeaveTypes.clear();
    }

    function reindexLeaveTypes() {
        const rows = document.querySelectorAll('#leave-types-container .leave-type-row');
        rows.forEach((row, index) => {
            row.setAttribute('data-index', index);
            
            const inputs = row.querySelectorAll('input, select');
            inputs.forEach(input => {
                const name = input.getAttribute('name');
                if (name) {
                    const newName = name.replace(/leave_types\[\d+\]/, `leave_types[${index}]`);
                    input.setAttribute('name', newName);
                }
            });
        });
    }

    function clearValidationErrors() {
        document.querySelectorAll('.is-invalid').forEach(el => {
            el.classList.remove('is-invalid');
        });
        
        document.querySelectorAll('.invalid-feedback').forEach(el => {
            el.textContent = '';
        });
    }

    function showAlert(type, message) {
        const alertHtml = `
            <div class="row">
                <div class="col-12">
                    <div class="alert alert-${type} alert-dismissible fade show" role="alert">
                        <strong>${type === 'success' ? 'Success!' : 'Error!'}</strong> ${message}
                        <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                    </div>
                </div>
            </div>
        `;
        
        const container = document.getElementById('alert-container');
        if (container) {
            container.innerHTML = alertHtml;
            
            setTimeout(() => {
                container.innerHTML = '';
            }, 5000);
        }
    }

    function populateLeaveTypeDropdown(selectElement) {
        if (!selectElement) return;
        
        $(selectElement).empty();
        $(selectElement).append('<option value="">Select Leave Type</option>');
        
        if (leaveTypesData && leaveTypesData.length > 0) {
            leaveTypesData.forEach(function(type) {
                const option = new Option(type.leave_type, type.id_leave_type, false, false);
                option.setAttribute('data-name', type.leave_type);
                $(selectElement).append(option);
            });
        } else {
            $(selectElement).append('<option value="" disabled>No leave types available</option>');
        }
        
        if ($(selectElement).hasClass('select2-hidden-accessible')) {
            $(selectElement).trigger('change.select2');
        }
    }
</script>
@endsection