@extends('layouts.horizontal', ['title' => 'Loan Report', 'topbarTitle' => 'Loan Report'])

@section('css')
@vite(['node_modules/datatables.net-bs5/css/dataTables.bootstrap5.min.css',
'node_modules/datatables.net-responsive-bs5/css/responsive.bootstrap5.min.css',
'node_modules/datatables.net-fixedcolumns-bs5/css/fixedColumns.bootstrap5.min.css',
'node_modules/datatables.net-fixedheader-bs5/css/fixedHeader.bootstrap5.min.css',
'node_modules/datatables.net-buttons-bs5/css/buttons.bootstrap5.min.css',
'node_modules/datatables.net-select-bs5/css/select.bootstrap5.min.css',
'node_modules/dropzone/dist/dropzone.css',
'node_modules/select2/dist/css/select2.min.css',
'node_modules/flatpickr/dist/flatpickr.min.css'])
<style>
    .filter-row { background-color: #f9f9f9; }
    .form-filter { font-size: 12px; padding: 4px 6px; height: 30px; width: 100%; }
    .filter-row th { padding: 4px 6px !important; vertical-align: middle; }
    .dataTables_wrapper .dataTables_filter { float: right; }
    .dataTables_wrapper .dataTables_length { float: left; }
    .dataTables_wrapper .dataTables_paginate { float: right; }
    .filter-buttons { display: flex; gap: 5px; }
    .filter-buttons button { padding: 4px 8px; font-size: 12px; }
    .action-buttons { white-space: nowrap; }
    .amount-cell { text-align: right; }
    .status-badge {
        padding: 4px 8px;
        border-radius: 4px;
        font-size: 12px;
        font-weight: 500;
    }
    .status-active { background-color: #d4edda; color: #155724; }
    .status-pending { background-color: #fff3cd; color: #856404; }
    .status-paid { background-color: #cce5ff; color: #004085; }
    .status-overdue { background-color: #f8d7da; color: #721c24; }
    
    /* Fix DataTable scrolling issues */
    .dataTables_wrapper .dataTables_scroll {
        position: relative;
        clear: both;
        border: 1px solid #dee2e6;
        border-radius: 4px;
    }
    
    .dataTables_scrollBody {
        max-height: 400px !important;
        min-height: 200px !important;
        overflow-y: auto !important;
        overflow-x: hidden !important;
    }
    
    .dataTables_scrollBody table.dataTable {
        margin-bottom: 0 !important;
        border-collapse: collapse !important;
    }
    
    .dataTables_scrollFoot {
        border-top: 2px solid #dee2e6;
    }
    
    .dataTables_scrollFoot .amount-cell {
        font-weight: bold;
    }
    
    /* Ensure table headers stay fixed */
    table.dataTable thead th {
        position: sticky;
        top: 0;
        background-color: #f8f9fa;
        z-index: 10;
    }
    
    .dataTables_empty {
        padding: 40px !important;
        text-align: center;
        color: #6c757d;
        background-color: #f8f9fa !important;
    }
    
    .btn-icon {
        width: 36px;
        height: 36px;
        display: inline-flex;
        align-items: center;
        justify-content: center;
        padding: 0;
    }
    
    .btn-rounded {
        border-radius: 50%;
    }
</style>
@endsection

@section('content')
<div class="container-fluid">
    <div class="row mb-4">
        <div class="col-12">
            <div class="page-title-box d-flex align-items-center justify-content-between">
                <h2 class="mb-0">Loan Report</h2>
                <div class="page-title-right d-flex align-items-center gap-3">
                    @php
                        $breadcrumb = getCurrentMenuBreadcrumb();
                    @endphp
                    @if(!empty($breadcrumb))
                        <nav aria-label="breadcrumb">
                            <ol class="breadcrumb mb-0 py-2">
                                @foreach($breadcrumb as $index => $item)
                                    @if($index === count($breadcrumb) - 1)
                                        <li class="breadcrumb-item active" aria-current="page">{{ $item }}</li>
                                    @else
                                        <li class="breadcrumb-item"><a href="javascript: void(0);">{{ $item }}</a></li>
                                    @endif
                                @endforeach
                            </ol>
                        </nav>
                    @endif
                </div>
            </div>
        </div>
    </div>
    
    <div id="alert-container"></div>
    
    @if(session('ho') === 'Yes')
        <div class="row mb-3">
            <div class="col-12">
                <div class="card">
                    <div class="card-body">
                        <div class="row align-items-center">
                            <div class="col-md-12">
                                <div class="d-flex align-items-center justify-content-between gap-3">
                                    <div class="flex-grow-1">
                                        <label class="form-label">Select Business</label>
                                        <select class="form-control select2" id="businessSelector">
                                            @foreach($businesses as $business)
                                                <option value="{{ $business->id_business }}" 
                                                    {{ session('selected_business_id') == $business->id_business ? 'selected' : '' }}>
                                                    {{ $business->business_name }}
                                                </option>
                                            @endforeach
                                        </select>
                                    </div>
                                    <div>
                                        <button type="button" class="btn btn-outline-success" id="exportToExcelBtn">
                                            <iconify-icon icon="mdi:microsoft-excel"></iconify-icon> Export Excel
                                        </button>
                                    </div>
                                </div>
                                <input type="hidden" name="filter_business_id" id="filter_business_id" value="{{ session('selected_business_id') ?? '' }}">
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    @else
        <div class="row mb-3">
            <div class="col-12">
                <div class="card">
                    <div class="card-body">
                        <div class="d-flex justify-content-end">
                            <button type="button" class="btn btn-outline-success" id="exportToExcelBtn">
                                <iconify-icon icon="mdi:microsoft-excel"></iconify-icon> Export Excel
                            </button>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    @endif

    <!-- Date Range Filter -->
    <div class="row mb-3">
        <div class="col-12">
            <div class="card">
                <div class="card-body">
                    <div class="row">
                        <div class="col-md-4">
                            <div class="mb-3">
                                <label class="form-label">Date Range</label>
                                <div class="input-group">
                                    <input type="text" class="form-control" id="loan_dateRange" name="loan_date_range" placeholder="Select date range" readonly required>
                                    <input type="hidden" id="loan_start_date" name="loan_start_date" value="{{ $startDate }}">
                                    <input type="hidden" id="loan_end_date" name="loan_end_date" value="{{ $endDate }}">
                                </div>
                            </div>
                        </div>
                        <div class="col-md-3">
                            <div class="mb-3">
                                <label class="form-label">Staff</label>
                                <select class="form-control select2" id="staff_filter" name="staff_filter">
                                    <option value="">All Staff</option>
                                    @foreach($staffList as $staff)
                                        <option value="{{ $staff->id_staff }}">{{ $staff->staff_fullname }}</option>
                                    @endforeach
                                </select>
                            </div>
                        </div>
                        <div class="col-md-3">
                            <div class="mb-3">
                                <label class="form-label">Loan Status</label>
                                <select class="form-control" id="loan_status_filter" name="loan_status_filter">
                                    <option value="All">All</option>
                                    <option value="HasLoans" selected>Has Loans</option>
                                    <option value="NoLoans">No Loans</option>
                                    <option value="PendingLoans">Pending Loans</option>
                                </select>
                            </div>
                        </div>
                        <div class="col-md-2 d-flex align-items-end">
                            <div class="mb-3 w-100">
                                <button class="btn btn-sm btn-ghost-primary rounded-circle btn-icon" id="applyFiltersBtn" title="Search">
                                    <iconify-icon icon="mdi:magnify" width="30px"></iconify-icon>
                                </button>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Main Data Table -->
    <div class="row">
        <div class="col-12">
            <div class="card">
                <div class="card-body">
                    <table id="loan_report_table" class="table table-bordered dt-responsive fs-13 w-100">
                        <thead>
                            <tr>
                                <th>Staff ID</th>
                                <th>Staff Name</th>
                                <th>Business</th>
                                <th>Total Loan Given</th>
                                <th>Approved Loans</th>
                                <th>Pending Requests</th>
                                <th>Loan Returned</th>
                                <th>Loan Remaining</th>
                                <th>Last Loan Date</th>
                                <th>Monthly Installment</th>
                                <th>Status</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody></tbody>
                        <tfoot>
                            <tr>
                                <th colspan="3" style="text-align:right">Totals:</th>
                                <th id="footerTotalLoan" class="amount-cell"></th>
                                <th id="footerApprovedLoans" class="text-center"></th>
                                <th id="footerPendingLoans" class="text-center"></th>
                                <th id="footerLoanReturned" class="amount-cell"></th>
                                <th id="footerLoanRemaining" class="amount-cell"></th>
                                <th colspan="3"></th>
                            </tr>
                        </tfoot>
                    </table>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Loan Details Modal -->
<div class="modal fade" id="loanDetailsModal" tabindex="-1" aria-labelledby="loanDetailsModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="loanDetailsModalLabel">Loan Details - <span id="detailStaffName"></span></h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
                <div class="row mb-3">
                    <div class="col-md-6">
                        <p><strong>Staff Name:</strong> <span id="detailStaffFullName"></span></p>
                        <p><strong>Staff ID:</strong> <span id="detailStaffId"></span></p>
                        <p><strong>Business:</strong> <span id="detailBusinessName"></span></p>
                    </div>
                    <div class="col-md-6">
                        <p><strong>Total Loan Given:</strong> <span id="detailTotalLoan" class="fw-bold"></span></p>
                        <p><strong>Loan Returned:</strong> <span id="detailLoanReturned" class="fw-bold"></span></p>
                        <p><strong>Loan Remaining:</strong> <span id="detailLoanRemaining" class="fw-bold"></span></p>
                    </div>
                </div>
                
                <div class="table-responsive">
                    <table class="table table-bordered" id="loanDetailsTable">
                        <thead>
                            <tr>
                                <th>Loan ID</th>
                                <th>Amount</th>
                                <th>Monthly Installment</th>
                                <th>Total Months</th>
                                <th>Status</th>
                                <th>Approved Date</th>
                            </tr>
                        </thead>
                        <tbody id="loanDetailsBody">
                            <!-- Loan details will be populated here -->
                        </tbody>
                    </table>
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">
                    <iconify-icon icon="mdi:close" class="me-1 fs-22"></iconify-icon> Close
                </button>
            </div>
        </div>
    </div>
</div>
@endsection

@section('scripts')
@vite(['resources/js/app.js'])
<script>
    document.addEventListener('DOMContentLoaded', function() {
        
        initializeDatePicker();
        initializeSelect2();
        
        // Set default filter to HasLoans
        $('#loan_status_filter').val('HasLoans');
        
        initializeLoanReportTable();
        setupEventHandlers();
        setupBusinessChangeHandlers();
        
        // Apply default filter
        refreshData();
    });

    let loanReportTable = null;
    let loanDetailsModal = null;
    let loanDateRangePicker = null;
    
   
    
    function initializeDatePicker() {
        const startDate = new Date('{{ $startDate }}');
        const endDate = new Date('{{ $endDate }}');
        
        loanDateRangePicker = flatpickr("#loan_dateRange", {
            mode: "range",
            dateFormat: "Y-m-d",
            defaultDate: [startDate, endDate],
            onChange: function(selectedDates, dateStr, instance) {
                if (selectedDates.length === 2) {
                    updateLoanDateRangeFields(selectedDates[0], selectedDates[1]);
                } else if (selectedDates.length === 1) {
                    // Waiting for second date selection
                } else {
                    $('#loan_dateRange').val('');
                    $('#loan_start_date').val('');
                    $('#loan_end_date').val('');
                }
            },
        });
        
        updateLoanDateRangeFields(startDate, endDate);
    }
    
    function updateLoanDateRangeFields(startDate, endDate) {
        $('#loan_start_date').val(formatDate(startDate));
        $('#loan_end_date').val(formatDate(endDate));
        
        // Format for display
        const formattedStart = formatDate(startDate, 'd-m-Y');
        const formattedEnd = formatDate(endDate, 'd-m-Y');
        $('#loan_dateRange').val(formattedStart + ' to ' + formattedEnd);
    }
    
    function formatDate(date, format = 'Y-m-d') {
        const d = new Date(date);
        if (format === 'd-m-Y') {
            return d.getDate().toString().padStart(2, '0') + '-' + 
                   (d.getMonth() + 1).toString().padStart(2, '0') + '-' + 
                   d.getFullYear();
        } else {
            return d.getFullYear() + '-' + 
                   (d.getMonth() + 1).toString().padStart(2, '0') + '-' + 
                   d.getDate().toString().padStart(2, '0');
        }
    }
    
    function initializeSelect2() {
        $('#businessSelector').select2({
            width: '100%'
        });
        
        $('#staff_filter').select2({
            width: '100%'
        });
    }

    function setupBusinessChangeHandlers() {
        $('#businessSelector').on('change', function() {
            const businessId = $(this).val();
            $('#filter_business_id').val(businessId);
            
            // Reload staff list for the selected business
            if (businessId) {
                loadStaffForBusiness(businessId);
            } else {
                clearStaffFilter();
            }
            
            if (loanReportTable) {
                loanReportTable.ajax.reload();
            }
        });
    }
    
    function loadStaffForBusiness(businessId) {
        if (!businessId) return;
        
        $.ajax({
            url: '{{ route("hrm.get_staff_by_business") }}',
            method: 'GET',
            data: {
                business_id: businessId
            },
            success: function(response) {
                if (response.success) {
                    const staffSelect = $('#staff_filter');
                    staffSelect.empty();
                    staffSelect.append('<option value="">All Staff</option>');
                    
                    response.staff.forEach(function(staff) {
                        staffSelect.append(
                            $('<option></option>')
                                .val(staff.id_staff)
                                .text(staff.staff_fullname)
                        );
                    });
                    
                    staffSelect.select2({
                        width: '100%'
                    });
                } else {
                    showAlert('error', response.message || 'Failed to load staff.');
                }
            },
            error: function(xhr, status, error) {
                console.error('Error loading staff:', error);
                showAlert('error', 'Error loading staff list.');
            }
        });
    }
    
    function clearStaffFilter() {
        const staffSelect = $('#staff_filter');
        staffSelect.empty();
        staffSelect.append('<option value="">All Staff</option>');
        staffSelect.select2({
            width: '100%'
        });
    }

    function initializeLoanReportTable() {
        if (typeof $.fn.DataTable === 'undefined') {
            console.error('DataTables is not loaded');
            return;
        }
        
        // Destroy existing DataTable if it exists
        if ($.fn.DataTable.isDataTable('#loan_report_table')) {
            $('#loan_report_table').DataTable().destroy();
        }
        
        loanReportTable = $('#loan_report_table').DataTable({
            scrollX: true,
            scrollY: '400px',
            scrollCollapse: true,
            paging: true,
            fixedHeader: {
                header: true
            },
            processing: true,
            serverSide: true,
            stateSave: false,
            stateDuration: 0,
            ajax: {
                url: "{{ route('hrm.get_loan_report_data') }}",
                type: 'POST',
                headers: {
                    'X-CSRF-TOKEN': '{{ csrf_token() }}'
                },
                data: function (d) {
                    // Add custom filters to the request
                    return {
                        start_date: $('#loan_start_date').val(),
                        end_date: $('#loan_end_date').val(),
                        staff_id: $('#staff_filter').val(),
                        loan_status: $('#loan_status_filter').val(),
                        business_id: $('#filter_business_id').val(),
                        draw: d.draw,
                        start: d.start,
                        length: d.length,
                        order: d.order,
                        search: d.search
                    };
                },
                error: function (xhr, error, thrown) {
                    console.error('DataTables AJAX error:', error, thrown, xhr.responseText);
                    showAlert('error', 'Failed to load loan report data. Please try again.');
                }
            },
            "language": {
                "emptyTable": "No loan data available",
                "zeroRecords": "No matching records found",
                "processing": "Processing...",
                "paginate": {
                    "previous": "<i class='ti ti-chevron-left'></i>",
                    "next": "<i class='ti ti-chevron-right'></i>"
                }
            },
            lengthMenu: [[10, 25, 50, 100, -1], [10, 25, 50, 100, "All"]],
            pageLength: 10,
            searching: true,
            info: true,
            columns: [
                { 
                    data: 'staff_code', 
                    name: 'staff_code', 
                    orderable: true, 
                    searchable: true,
                    render: function(data, type, row) {
                        return data || 'N/A';
                    }
                },
                { 
                    data: 'staff_fullname', 
                    name: 'staff_fullname', 
                    orderable: true, 
                    searchable: true,
                    render: function(data, type, row) {
                        return data || 'N/A';
                    }
                },
                { 
                    data: 'business_name', 
                    name: 'business_name', 
                    orderable: true, 
                    searchable: true,
                    render: function(data, type, row) {
                        return data || 'N/A';
                    }
                },
                { 
                    data: 'total_loan_given', 
                    name: 'total_loan_given', 
                    orderable: true, 
                    searchable: true,
                    className: 'amount-cell',
                    render: function(data, type, row) {
                        // Handle null or undefined data
                        if (data === null || data === undefined) {
                            return 'Rs. 0.00';
                        }
                        return 'Rs. ' + data;
                    }
                },
                { 
                    data: 'approved_loans_count', 
                    name: 'approved_loans_count', 
                    orderable: true, 
                    searchable: true,
                    className: 'text-center',
                    render: function(data, type, row) {
                        return data || 0;
                    }
                },
                { 
                    data: 'pending_loans_count', 
                    name: 'pending_loans_count', 
                    orderable: true, 
                    searchable: true,
                    className: 'text-center',
                    render: function(data, type, row) {
                        return data || 0;
                    }
                },
                { 
                    data: 'loan_returned', 
                    name: 'loan_returned', 
                    orderable: true, 
                    searchable: true,
                    className: 'amount-cell',
                    render: function(data, type, row) {
                        // Handle null or undefined data
                        if (data === null || data === undefined) {
                            return 'Rs. 0.00';
                        }
                        return 'Rs. ' + data;
                    }
                },
                { 
                    data: 'loan_remaining', 
                    name: 'loan_remaining', 
                    orderable: true, 
                    searchable: true,
                    className: 'amount-cell',
                    render: function(data, type, row) {
                        // Handle null or undefined data
                        if (data === null || data === undefined) {
                            return '<span class="text-success fw-bold">Rs. 0.00</span>';
                        }
                        
                        // Try to parse the amount safely
                        let amount = 0;
                        try {
                            const cleanData = String(data).replace('Rs. ', '').replace(/,/g, '');
                            amount = parseFloat(cleanData) || 0;
                        } catch (e) {
                            amount = 0;
                        }
                        
                        let className = '';
                        if (amount > 0) {
                            className = 'text-danger fw-bold';
                        } else {
                            className = 'text-success fw-bold';
                        }
                        
                        return '<span class="' + className + '">Rs. ' + data + '</span>';
                    }
                },
                { 
                    data: 'last_loan_date', 
                    name: 'last_loan_date', 
                    orderable: true, 
                    searchable: true,
                    render: function(data, type, row) {
                        return data || 'N/A';
                    }
                },
                { 
                    data: 'total_monthly_installment', 
                    name: 'total_monthly_installment', 
                    orderable: true, 
                    searchable: true,
                    className: 'amount-cell',
                    render: function(data, type, row) {
                        // Handle null or undefined data
                        if (data === null || data === undefined) {
                            return 'Rs. 0.00';
                        }
                        return 'Rs. ' + data;
                    }
                },
                { 
                    data: null, 
                    name: 'status', 
                    orderable: false, 
                    searchable: false,
                    render: function(data, type, row) {
                        // Safely check if staff has loans
                        const hasLoans = row.has_loans === true || 
                                    (row.raw_total_loan && parseFloat(row.raw_total_loan) > 0);
                        const hasPending = row.has_pending === true || 
                                        (row.pending_loans_count && parseInt(row.pending_loans_count) > 0);
                        
                        // Safely parse loan remaining
                        let loanRemaining = 0;
                        try {
                            loanRemaining = parseFloat(row.raw_loan_remaining) || 0;
                        } catch (e) {
                            loanRemaining = 0;
                        }
                        
                        if (hasLoans) {
                            if (hasPending) {
                                return '<span class="status-badge status-pending">Has Pending</span>';
                            } else if (loanRemaining > 0) {
                                return '<span class="status-badge status-active">Active Loan</span>';
                            } else {
                                return '<span class="status-badge status-paid">Paid Off</span>';
                            }
                        } else if (hasPending) {
                            return '<span class="status-badge status-pending">Pending Only</span>';
                        } else {
                            return '<span class="status-badge">No Loans</span>';
                        }
                    }
                },
                { 
                    data: null, 
                    name: 'actions', 
                    orderable: false, 
                    searchable: false,
                    className: 'action-buttons',
                    render: function(data, type, row) {
                        let buttons = '';
                        
                        // Safely check if staff has loans or loan details
                        const hasLoans = row.has_loans === true || 
                                    (row.raw_total_loan && parseFloat(row.raw_total_loan) > 0);
                        const hasLoanDetails = row.loan_details && row.loan_details.length > 0;
                        
                        if (hasLoans || hasLoanDetails) {
                            // Safely get values with defaults
                            const staffId = row.id_staff || 0;
                            const staffName = escapeHtml(row.staff_fullname || '');
                            const businessName = escapeHtml(row.business_name || '');
                            const staffCode = escapeHtml(row.staff_code || '');
                            const totalLoan = row.total_loan_given || '0.00';
                            const loanReturned = row.loan_returned || '0.00';
                            const loanRemaining = row.loan_remaining || '0.00';
                            const loanDetails = row.loan_details || [];
                            
                            buttons += `
                                <button class="btn btn-sm btn-ghost-secondary rounded-circle btn-icon" 
                                        onclick="viewLoanDetails(${staffId}, '${staffName}', '${businessName}', '${staffCode}', '${totalLoan}', '${loanReturned}', '${loanRemaining}', ${JSON.stringify(loanDetails).replace(/"/g, '&quot;')})"
                                        title="View Loan Details">
                                    <iconify-icon icon="mdi:eye-outline" class="fs-22"></iconify-icon>
                                </button>
                            `;
                        }
                        
                        return buttons;
                    }
                }
            ],
            drawCallback: function(settings) {
                // Update footer totals
                const json = settings.json;
                if (json && json.summary) {
                    updateFooterTotals(json.summary);
                }
                
                // Fix scrolling issue
                $('.dataTables_scrollBody').css('overflow', 'auto');
                
                // Store the current data for export
                storeCurrentDataForExport(json);
            },
            footerCallback: function (row, data, start, end, display) {
                // This is handled by drawCallback now
            },
            initComplete: function() {
                // Fix for scrolling issue
                this.api().columns.adjust().draw();
            },
            dom: "<'row'<'col-sm-12 col-md-6'l><'col-sm-12 col-md-6'f>>" +
                 "<'row'<'col-sm-12'tr>>" +
                 "<'row'<'col-sm-12 col-md-5'i><'col-sm-12 col-md-7'p>>"
        });
    }
    
    // Store current data for export
    let currentExportData = [];
    let currentSummary = null;
    
    function storeCurrentDataForExport(json) {
        if (json && json.data) {
            currentExportData = json.data;
            currentSummary = json.summary;
        }
    }
    
    function escapeHtml(text) {
        // Handle null, undefined, or non-string values
        if (text === null || text === undefined) {
            return '';
        }
        
        // Convert to string if it's not already
        const textStr = String(text);
        
        const map = {
            '&': '&amp;',
            '<': '&lt;',
            '>': '&gt;',
            '"': '&quot;',
            "'": '&#039;'
        };
        return textStr.replace(/[&<>"']/g, function(m) { return map[m]; });
    }
        
    function updateFooterTotals(summary) {
        $('#footerTotalLoan').html('<strong>Rs. ' + summary.total_loan_given + '</strong>');
        $('#footerApprovedLoans').html('<strong>' + summary.total_approved_loans + '</strong>');
        $('#footerPendingLoans').html('<strong>' + summary.total_pending_loans + '</strong>');
        $('#footerLoanReturned').html('<strong>Rs. ' + summary.total_loan_returned + '</strong>');
        $('#footerLoanRemaining').html('<strong>Rs. ' + summary.total_loan_remaining + '</strong>');
    }
    
    function setupEventHandlers() {
        const loanDetailsModalEl = document.getElementById('loanDetailsModal');
        if (loanDetailsModalEl) loanDetailsModal = new bootstrap.Modal(loanDetailsModalEl);
        
        // Apply filters button
        const applyFiltersBtn = document.getElementById('applyFiltersBtn');
        if (applyFiltersBtn) {
            applyFiltersBtn.addEventListener('click', function() {
                refreshData();
            });
        }
        
        // Export to Excel button
        const exportToExcelBtn = document.getElementById('exportToExcelBtn');
        if (exportToExcelBtn) {
            exportToExcelBtn.addEventListener('click', function() {
                exportToExcel();
            });
        }
        
        // Handle Enter key in filters
        document.addEventListener('keyup', function(e) {
            if (e.target.closest('#loan_dateRange, #staff_filter, #loan_status_filter') && e.key === 'Enter') {
                refreshData();
            }
        });
    }
    
    function refreshData() {
        if (loanReportTable) {
            loanReportTable.ajax.reload(null, false);
        }
    }
    
    function viewLoanDetails(staffId, staffName, businessName, staffCode, totalLoan, loanReturned, loanRemaining, loanDetails) {
        try {
            let details = [];
            try {
                details = typeof loanDetails === 'string' ? JSON.parse(loanDetails) : (loanDetails || []);
            } catch (e) {
                console.error('Error parsing loan details:', e);
                details = [];
            }
            
            // Update modal title and basic info
            document.getElementById('detailStaffName').textContent = staffName || 'N/A';
            document.getElementById('detailStaffFullName').textContent = staffName || 'N/A';
            document.getElementById('detailStaffId').textContent = staffCode || 'N/A';
            document.getElementById('detailBusinessName').textContent = businessName || 'N/A';
            document.getElementById('detailTotalLoan').textContent = totalLoan || '0.00';
            document.getElementById('detailLoanReturned').textContent = loanReturned || '0.00';
            document.getElementById('detailLoanRemaining').textContent = loanRemaining || '0.00';
            
            // Populate loan details table
            const detailsBody = document.getElementById('loanDetailsBody');
            detailsBody.innerHTML = '';
            
            if (details && details.length > 0) {
                details.forEach(function(loan) {
                    try {
                        const row = document.createElement('tr');
                        const amount = loan.amount ? parseFloat(loan.amount) : 0;
                        const installment = loan.installment ? parseFloat(loan.installment) : 0;
                        
                        row.innerHTML = `
                            <td>${loan.loan_id || 'N/A'}</td>
                            <td class="amount-cell">Rs. ${amount.toLocaleString('en-IN', {
                                minimumFractionDigits: 2,
                                maximumFractionDigits: 2
                            })}</td>
                            <td class="amount-cell">Rs. ${installment.toLocaleString('en-IN', {
                                minimumFractionDigits: 2,
                                maximumFractionDigits: 2
                            })}</td>
                            <td class="text-center">${loan.months || '0'}</td>
                            <td>
                                <span class="status-badge ${getStatusClass(loan.status)}">
                                    ${loan.status || 'N/A'}
                                </span>
                            </td>
                            <td>${loan.approved_date ? formatDisplayDate(loan.approved_date) : 'N/A'}</td>
                        `;
                        detailsBody.appendChild(row);
                    } catch (e) {
                        console.error('Error rendering loan detail row:', e);
                    }
                });
            } else {
                const row = document.createElement('tr');
                row.innerHTML = `
                    <td colspan="7" class="text-center">No active loan details found.</td>
                `;
                detailsBody.appendChild(row);
            }
            
            // Show modal
            if (loanDetailsModal) {
                loanDetailsModal.show();
            }
            
        } catch (error) {
            console.error('Error viewing loan details:', error);
            showAlert('error', 'Error loading loan details.');
        }
    }
    
    function getStatusClass(status) {
        switch(status) {
            case 'Approved': return 'status-active';
            case 'Disbursed': return 'status-active';
            case 'Pending': return 'status-pending';
            case 'Rejected': return 'status-overdue';
            default: return '';
        }
    }
    
    function formatDisplayDate(dateString) {
        if (!dateString) return 'N/A';
        const date = new Date(dateString);
        return date.getDate().toString().padStart(2, '0') + '-' + 
               (date.getMonth() + 1).toString().padStart(2, '0') + '-' + 
               date.getFullYear();
    }
    
    function exportToExcel() {
        // Show loading indicator
        const originalBtnText = $('#exportToExcelBtn').html();
        $('#exportToExcelBtn').html('<span class="spinner-border spinner-border-sm me-2"></span>Exporting...');
        $('#exportToExcelBtn').prop('disabled', true);
        
        try {
            // Create a new table structure for export
            var table = document.getElementById('loan_report_table');
            var tableClone = document.createElement('table');
            
            // Clone the header
            var theadClone = table.getElementsByTagName('thead')[0].cloneNode(true);
            tableClone.appendChild(theadClone);
            
            // Clone the body (data rows)
            var tbodyClone = table.getElementsByTagName('tbody')[0].cloneNode(true);
            tableClone.appendChild(tbodyClone);
            
            // Clone the footer (totals) - this will now be added after body
            var tfootClone = table.getElementsByTagName('tfoot')[0].cloneNode(true);
            tableClone.appendChild(tfootClone);
            
            // Clean up the clone (remove unwanted elements)
            $(tableClone).find('.select2-container, .select2-selection, .select2-search__field, .select2-dropdown').remove();
            $(tableClone).find('[style*="position: sticky"]').removeAttr('style');
            $(tableClone).find('[onclick]').removeAttr('onclick');
            $(tableClone).find('[class*="badge"]').each(function() {
                $(this).replaceWith($('<span>').text($(this).text()));
            });
            
            // Remove iconify icons
            $(tableClone).find('iconify-icon').each(function() {
                $(this).remove();
            });
            
            // Remove action buttons column (last column)
            $(tableClone).find('th:last, td:last').remove();
            
            // Clean up HTML content in cells
            $(tableClone).find('td, th').each(function() {
                const $cell = $(this);
                // Remove HTML tags, keep only text
                const text = $cell.text().replace(/\s+/g, ' ').trim();
                $cell.text(text);
            });
            
            // Generate filename
            const timestamp = new Date().toISOString().slice(0, 10).replace(/-/g, '');
            const filename = `Loan_Report_${timestamp}.xlsx`;
            
            try {
                // Create workbook with proper table structure
                var ws = XLSX.utils.table_to_sheet(tableClone, {sheet: "Loan Report", raw: true});
                
                // Apply some styling to the totals row
                if (ws['!ref']) {
                    const range = XLSX.utils.decode_range(ws['!ref']);
                    const totalRowIndex = range.e.r; // Last row index
                    
                    // Make totals row bold
                    for (let col = range.s.c; col <= range.e.c; col++) {
                        const cellAddress = XLSX.utils.encode_cell({r: totalRowIndex, c: col});
                        if (ws[cellAddress]) {
                            // Create a cell style for the totals row
                            ws[cellAddress].s = {
                                font: { bold: true }
                            };
                        }
                    }
                }
                
                // Create workbook
                var wb = XLSX.utils.book_new();
                XLSX.utils.book_append_sheet(wb, ws, "Loan Report");
                
                // Save the file
                XLSX.writeFile(wb, filename);
                
            } catch (e) {
                console.log('XLSX method failed, using fallback:', e);
                // Fallback method - create HTML table with proper structure
                var html = '<table>';
                html += theadClone.outerHTML;
                html += tbodyClone.outerHTML;
                html += tfootClone.outerHTML;
                html += '</table>';
                
                var blob = new Blob([html], {type: 'application/vnd.ms-excel'});
                var url = URL.createObjectURL(blob);
                var link = document.createElement('a');
                link.download = filename.replace('.xlsx', '.xls');
                link.href = url;
                link.click();
                URL.revokeObjectURL(url);
            }
            
        } catch (error) {
            console.error('Error exporting to Excel:', error);
            showAlert('error', 'Error exporting to Excel: ' + error.message);
        } finally {
            // Reset button
            setTimeout(() => {
                $('#exportToExcelBtn').html(originalBtnText);
                $('#exportToExcelBtn').prop('disabled', false);
            }, 1000);
        }
    }
        
   
    function showAlert(type, message) {
        const alertHtml = `
            <div class="row">
                <div class="col-12">
                    <div class="alert alert-${type} alert-dismissible fade show" role="alert">
                        <strong>${type === 'success' ? 'Success!' : 'Error!'}</strong> ${message}
                        <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                    </div>
                </div>
            </div>
        `;
        
        const container = document.getElementById('alert-container');
        if (container) {
            container.innerHTML = alertHtml;
            
            setTimeout(() => {
                container.innerHTML = '';
            }, 5000);
        }
    }
    
    window.viewLoanDetails = viewLoanDetails;
    window.loadStaffForBusiness = loadStaffForBusiness;
    window.clearStaffFilter = clearStaffFilter;
    window.refreshData = refreshData;
    window.exportToExcel = exportToExcel;
</script>
@endsection