@extends('layouts.horizontal', ['title' => 'Mark Attendance', 'topbarTitle' => 'Mark Attendance'])

@section('css')
@vite(['node_modules/datatables.net-bs5/css/dataTables.bootstrap5.min.css',
'node_modules/datatables.net-responsive-bs5/css/responsive.bootstrap5.min.css',
'node_modules/datatables.net-fixedcolumns-bs5/css/fixedColumns.bootstrap5.min.css',
'node_modules/datatables.net-fixedheader-bs5/css/fixedHeader.bootstrap5.min.css',
'node_modules/datatables.net-buttons-bs5/css/buttons.bootstrap5.min.css',
'node_modules/datatables.net-select-bs5/css/select.bootstrap5.min.css',
'node_modules/dropzone/dist/dropzone.css',
'node_modules/select2/dist/css/select2.min.css',
'node_modules/flatpickr/dist/flatpickr.min.css'])
<style>
    html, body { overflow-y: auto !important; height: auto !important; }
    .container-fluid { max-height: none !important; overflow: visible !important; }
    
    .attendance-container {
        background: #fff;
        border-radius: 15px;
        padding: 10px;
        box-shadow: 0 5px 30px rgba(0,0,0,.08);
        border: 1px solid #eef2f7;
        min-height: 450px;
    }
    
    .qr-container, .fingerprint-container, .instruction-container {
        background: #f8f9fa;
        border-radius: 10px;
        padding: 10px !important;
        border: 2px dashed #dee2e6;
        min-height: 380px;
        height: 100%;
        display: flex;
        flex-direction: column;
        align-items: center;
        justify-content: space-between; 
    }
    
    .section-title {
        font-weight: 600;
        font-size: 16px;
        color: #495057;
        border-bottom: 2px solid #eef2f7;
        padding-bottom: 8px;
        width: 100%;
        text-align: center;
        margin-bottom: 10px; 
    }
    
    #qrcode { 
        min-height: 250px !important; 
        min-width: 250px !important;
        display: flex !important; 
        align-items: center !important; 
        justify-content: center !important;
        margin: 5px 0;
    }
    
    #qrcode img, #qrcode svg {
        width: 220px !important;
        height: 220px !important;
        max-width: none !important;
    }
    
    .timer-badge {
        background: linear-gradient(45deg, #7393B3, #4A6888);
        color: #fff;
        padding: 6px 12px;
        border-radius: 20px;
        font-size: 12px;
        font-weight: 600;
        margin: 8px 0;
    }
    
    .btn-attendance { min-width: 160px; }
    #FPImage { 
        max-width: 200px !important;
        margin: 10px 0;
        width: 200px;
        height: 200px;
        object-fit: contain;
    }
    
    .qr-refresh-area {
        margin-top: 5px;
        text-align: center;
    }
    
    /* Hidden fingerprint status */
    .fingerprint-status {
        margin-top: 10px;
        padding: 8px;
        border-radius: 8px;
        text-align: center;
        min-height: 40px;
        display: none !important;
        align-items: center;
        justify-content: center;
        font-size: 13px;
    }
    
    .status-ready { background: #d1ecf1; color: #0c5460; }
    .status-scanning { background: #fff3cd; color: #856404; }
    .status-success { background: #d4edda; color: #155724; }
    .status-error { background: #f8d7da; color: #721c24; }
    
    .key-display {
        font-size: 12px;
        color: #666;
        margin-top: 5px;
        display: none;
    }
    
    .instruction-item {
        display: flex;
        align-items: center;
        padding: 10px 5px !important; 
        margin-bottom: 5px;
        border: none !important;
    }
    
    .instruction-icon {
        margin-right: 10px;
        font-size: 20px;
        flex-shrink: 0;
    }
    
    .instruction-text {
        text-align: left;
        font-size: 14px;
        color: #495057;
        font-weight: 500;
        line-height: 1.4;
    }
    
    .instruction-container .list-group {
        padding: 0 5px;
    }
    
    /* Ensure all cards have same height */
    .card-height-fix {
        min-height: 380px;
        display: flex;
        flex-direction: column;
    }
    
    /* Custom theme colors for circles */
    .text-primary { color: var(--bs-primary) !important; }
    .text-secondary { color: var(--bs-secondary) !important; }
    .text-danger { color: var(--bs-danger) !important; }
    .text-warning { color: var(--bs-warning) !important; }
    .text-success { color: var(--bs-success) !important; }
    .text-pink { color: #e83e8c !important; }
    .text-muted { color: #6c757d !important; }
    
    /* Center content in all cards */
    .card-content-center {
        display: flex;
        flex-direction: column;
        align-items: center;
        justify-content: center;
        text-align: center;
    }
</style>
@endsection

@section('content')
@php
    // Get the fingerprint license key from config
    $secugenLicense = config('services.fingerprint.license_key', '');
@endphp

<div class="container-fluid">
    <div class="row mb-4">
        <div class="col-12">
            <div class="page-title-box d-flex align-items-center justify-content-between">
                <h2 class="mb-0">Mark Your Attendance</h2>
                <div class="page-title-right d-flex align-items-center gap-3">
                    @php
                        $breadcrumb = getCurrentMenuBreadcrumb();
                    @endphp
                    @if(!empty($breadcrumb))
                        <nav aria-label="breadcrumb">
                            <ol class="breadcrumb mb-0 py-2">
                                @foreach($breadcrumb as $index => $item)
                                    @if($index === count($breadcrumb) - 1)
                                        <li class="breadcrumb-item active" aria-current="page">{{ $item }}</li>
                                    @else
                                        <li class="breadcrumb-item"><a href="javascript: void(0);">{{ $item }}</a></li>
                                    @endif
                                @endforeach
                            </ol>
                        </nav>
                    @endif
                </div>
            </div>
        </div>
    </div>
    <div class="attendance-container">
        <div class="row">
            {{-- ================= QR SECTION ================= --}}
            <div class="col-md-4 mb-3 d-flex">
                <div class="qr-container card-height-fix w-100">
                    <h3 class="section-title">QR Code Scan</h3>
                    <p class="text-muted small mb-2">Scan with registered mobile</p>

                    <div id="qrcode" class="card-content-center">
                        {!! $qrCode ?? '' !!}
                    </div>

                    <div class="key-display" id="keyDisplay">
                        Key: <span id="currentKey">{{ $keyCode ?? '' }}</span>
                    </div>

                    <div class="qr-refresh-area">
                        <!-- <div class="timer-badge">
                            QR Code updates in: <span id="timer">90</span>s
                        </div> -->
                        <button onclick="refreshQR()" class="btn btn-primary btn-attendance mt-1">
                            <iconify-icon icon="mdi:refresh" class="fs-18 me-1"></iconify-icon> Refresh QR
                        </button>
                    </div>
                </div>
            </div>

           {{-- ================= INSTRUCTIONS ================= --}}
           <div class="col-md-4 mb-3 d-flex">
                <div class="instruction-container card-height-fix w-100">
                    <h3 class="section-title">Instructions</h3>
                    <ul class="list-group mb-0 user-list w-100" style="border: none;">
                        <li class="list-group-item instruction-item">
                            <div class="d-flex align-items-center w-100">
                                <iconify-icon icon="mdi:circle" class="instruction-icon text-primary fs-16"></iconify-icon>
                                <div class="instruction-text">
                                    Use your registered Mobile Device only
                                </div>
                            </div>
                        </li>

                        <li class="list-group-item instruction-item">
                            <div class="d-flex align-items-center w-100">
                                <iconify-icon icon="mdi:circle" class="instruction-icon text-success fs-16"></iconify-icon>
                                <div class="instruction-text">
                                    You can only mark your time in once per day
                                </div>
                            </div>
                        </li>

                        <li class="list-group-item instruction-item">
                            <div class="d-flex align-items-center w-100">
                                <iconify-icon icon="mdi:circle" class="instruction-icon text-pink fs-16"></iconify-icon>
                                <div class="instruction-text">
                                    Your time is automatically synchronized with the central HR system
                                </div>
                            </div>
                        </li>

                        <li class="list-group-item instruction-item">
                            <div class="d-flex align-items-center w-100">
                                <iconify-icon icon="mdi:circle" class="instruction-icon text-muted fs-16"></iconify-icon>
                                <div class="instruction-text">
                                    In case your device fails to scan, contact your administrator
                                </div>
                            </div>
                        </li>
                        
                        <li class="list-group-item instruction-item">
                            <div class="d-flex align-items-center w-100">
                                <iconify-icon icon="mdi:circle" class="instruction-icon text-warning fs-16"></iconify-icon>
                                <div class="d-flex flex-column">
                                    <div class="instruction-text">
                                        QR Code will update in:
                                    </div>
                                    <div class="text-muted font-12 mt-1">
                                        <span id="instructionTimer">90</span> seconds
                                    </div>
                                </div>
                            </div>
                        </li>
                    </ul>
                    
                    <div class="mt-2 w-100 text-center">
                        <!-- <iconify-icon icon="mdi:magnify" class="fs-22 text-primary"></iconify-icon> -->
                        <!-- <span class="ms-2 small text-muted">Scan carefully for best results</span> -->
                    </div>
                </div>
            </div>

            {{-- ================= FINGERPRINT ================= --}}
            <div class="col-md-4 mb-3 d-flex">
                <div class="fingerprint-container card-height-fix w-100">
                    <h3 class="section-title">Fingerprint Scan</h3>
                    
                    <div class="card-content-center">
                        <img id="FPImage" 
                             src="{{ asset('images/staff_fingerprint/fingerprint_placeholder.jpg') }}" 
                             alt="Fingerprint" 
                             class="img-fluid">
                    </div>
                    
                    <!-- Hidden fingerprint status - only shown programmatically when needed -->
                    <div class="fingerprint-status status-ready mb-1" id="fingerprintStatus">
                        Ready to scan fingerprint
                    </div>
                    
                    <div class="mb-2" id="scanQuality" style="display: none;">
                        <small class="text-muted">Quality Threshold: 50</small>
                    </div>

                    <button id="fingerprint_scan_button" 
                            class="btn btn-primary btn-attendance mt-1"
                            onclick="scanFingerprint()">
                        <iconify-icon icon="mdi:fingerprint" class="fs-18 me-1"></iconify-icon> Scan Fingerprint
                    </button>
                    
                    <div class="text-center">
                        <small class="text-muted">
                            <iconify-icon icon="mdi:information" class="me-1"></iconify-icon>
                            Ensure Fingerprint device is connected
                        </small>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>
@endsection

@section('scripts')
@vite(['resources/js/app.js'])

<script>
let qrTimer = null;
let instructionTimer = null;
let secondsLeft = 90;
let isScanning = false;
const secugenLicense = "{{ $secugenLicense }}";
let currentLocation = null;
let locationPromise = null;

// Clear all timers
function clearAllTimers() {
    if (qrTimer) {
        clearInterval(qrTimer);
        qrTimer = null;
    }
    if (instructionTimer) {
        clearInterval(instructionTimer);
        instructionTimer = null;
    }
}

// Initialize timer
function startQRTimer() {
    // Clear any existing timers first
    clearAllTimers();
    
    secondsLeft = 90;
    document.getElementById('instructionTimer').textContent = secondsLeft;
    
    instructionTimer = setInterval(function() {
        secondsLeft--;
        document.getElementById('instructionTimer').textContent = secondsLeft;
        
        if (secondsLeft <= 0) {
            clearAllTimers();
            autoRefreshQR();
        }
    }, 1000);
}

// Auto refresh QR
function autoRefreshQR() {
    refreshQR();
}

// Manual QR refresh
function refreshQR() {
    fetch('{{ route("attendance.refresh_qr") }}', {
        method: 'POST',
        headers: {
            'X-Requested-With': 'XMLHttpRequest',
            'X-CSRF-TOKEN': '{{ csrf_token() }}',
            'Accept': 'application/json'
        }
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            // Update QR code
            const qrCodeDiv = document.getElementById('qrcode');
            qrCodeDiv.innerHTML = data.qr_code;
            
            // Ensure QR code is properly sized
            const qrImages = qrCodeDiv.querySelectorAll('img, svg');
            qrImages.forEach(img => {
                img.style.width = '220px';
                img.style.height = '220px';
                img.style.maxWidth = 'none';
            });
            
            // Update key if displayed
            const keyElement = document.getElementById('currentKey');
            if (keyElement) {
                keyElement.textContent = data.key_code;
            }
            
            // Reset and restart timer
            secondsLeft = 90;
            document.getElementById('instructionTimer').textContent = secondsLeft;
            startQRTimer();
            
            toastr.success('QR code refreshed', 'Success');
        } else {
            toastr.error(data.message || 'Failed to refresh QR', 'Error');
        }
    })
    .catch(error => {
        console.error('Error refreshing QR:', error);
        toastr.error('Failed to refresh QR. Please try again.', 'Error');
    });
}

// ================= FINGERPRINT SCANNING =================

function updateFingerprintStatus(status, message) {
    const statusDiv = document.getElementById('fingerprintStatus');
    statusDiv.textContent = message;
    statusDiv.className = 'fingerprint-status';
    
    switch(status) {
        case 'ready':
            statusDiv.classList.add('status-ready');
            statusDiv.style.display = 'none';
            break;
        case 'scanning':
            statusDiv.classList.add('status-scanning');
            statusDiv.style.display = 'flex';
            break;
        case 'success':
            statusDiv.classList.add('status-success');
            statusDiv.style.display = 'flex';
            break;
        case 'error':
            statusDiv.classList.add('status-error');
            statusDiv.style.display = 'flex';
            break;
    }
}

async function scanFingerprint() {
    if (isScanning) return;
    
    isScanning = true;
    const button = document.getElementById('fingerprint_scan_button');
    button.disabled = true;
    button.innerHTML = '<iconify-icon icon="mdi:loading" class="fs-18 me-1 spin"></iconify-icon> Getting location...';
    
    updateFingerprintStatus('scanning', 'Getting your location...');
    
    try {
        // Step 1: Get location FIRST
        // console.log('Getting location before fingerprint scan...');
        const location = await getLocationWithTimeout(8000); // 8 second timeout
        
        // Check if location is valid
        if (isDefaultLocation(location)) {
            console.warn('Using default/fallback location:', location);
            
            // Ask user for permission to use approximate location
            const useLocation = confirm('Unable to get precise location. Would you like to use approximate location for attendance?');
            
            if (!useLocation) {
                throw new Error('Location permission denied by user');
            }
        }
        
        // Step 2: Update button to show fingerprint scanning
        button.innerHTML = '<iconify-icon icon="mdi:loading" class="fs-18 me-1 spin"></iconify-icon> Scanning fingerprint...';
        updateFingerprintStatus('scanning', 'Scanning fingerprint...');
        
        // Step 3: Capture fingerprint
        const fingerprintData = await captureFingerprint();
        
        if (!fingerprintData.success) {
            throw new Error(fingerprintData.error || 'Failed to capture fingerprint');
        }
        
        // Step 4: Match fingerprint
        const matchResult = await matchFingerprint(fingerprintData.template);
        
        if (!matchResult.success) {
            throw new Error(matchResult.message || 'Fingerprint not recognized');
        }
        
        // Step 5: Record attendance WITH location
        const attendanceResult = await recordAttendance(matchResult, location);
        
        updateFingerprintStatus('success', attendanceResult.message);
        toastr.success(attendanceResult.message, 'Attendance Recorded');
        speak(attendanceResult.message);
        
        setTimeout(() => {
            resetFingerprintButton();
        }, 3000);
        
    } catch (error) {
        console.error('Fingerprint scan error:', error);
        updateFingerprintStatus('error', error.message);
        toastr.error(error.message, 'Scan Failed');
        resetFingerprintButton();
    }
}

function resetFingerprintButton() {
    const button = document.getElementById('fingerprint_scan_button');
    button.disabled = false;
    button.innerHTML = '<iconify-icon icon="mdi:fingerprint" class="fs-18 me-1"></iconify-icon> Scan Fingerprint';
    isScanning = false;
    
    setTimeout(() => {
        const statusDiv = document.getElementById('fingerprintStatus');
        statusDiv.style.display = 'none';
    }, 5000);
}

// Check if location is default/fallback
function isDefaultLocation(location) {
    if (!location) return true;
    
    // Check for default Pakistan coordinates
    if (location.address === 'Pakistan' || 
        location.address === 'Geolocation not supported by browser' ||
        location.address === 'Location permission denied' ||
        location.address === 'Location unavailable - please enable location services' ||
        (location.latitude === 30 && location.longitude === 70) ||
        location.latitude === null ||
        location.longitude === null) {
        return true;
    }
    
    return false;
}

// Get location with timeout
function getLocationWithTimeout(timeout = 8000) {
    return new Promise((resolve, reject) => {
        const timer = setTimeout(() => {
            reject(new Error('Location request timed out'));
        }, timeout);
        
        getLocation()
            .then(location => {
                clearTimeout(timer);
                resolve(location);
            })
            .catch(error => {
                clearTimeout(timer);
                reject(error);
            });
    });
}

function captureFingerprint() {
    return new Promise((resolve, reject) => {
        const uri = "https://localhost:8443/SGIFPCapture";
        const params = new URLSearchParams({
            Timeout: "10000",
            Quality: "50",
            licstr: encodeURIComponent(secugenLicense),
            templateFormat: "ISO"
        });

        const xhr = new XMLHttpRequest();
        xhr.timeout = 15000;
        xhr.open("POST", uri, true);
        
        xhr.onreadystatechange = function() {
            if (xhr.readyState === 4) {
                if (xhr.status === 200) {
                    try {
                        const result = JSON.parse(xhr.responseText);
                        if (result.ErrorCode === 0) {
                            if (result.BMPBase64) {
                                document.getElementById('FPImage').src = 
                                    "data:image/bmp;base64," + result.BMPBase64;
                            }
                            
                            resolve({
                                success: true,
                                template: result.TemplateBase64,
                                image: result.BMPBase64,
                                quality: result.Quality
                            });
                        } else {
                            reject(new Error(`Capture Error: ${errorCodeToString(result.ErrorCode)}`));
                        }
                    } catch (e) {
                        reject(new Error("Invalid response from scanner"));
                    }
                } else {
                    reject(new Error(`Scanner not found (Status: ${xhr.status})`));
                }
            }
        };

        xhr.onerror = () => reject(new Error("Network error"));
        xhr.ontimeout = () => reject(new Error("Scanner timeout"));
        
        xhr.send(params);
    });
}

function matchFingerprint(templateBase64) {
    return fetch('{{ route("attendance.fingerprint_match") }}', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
            'X-CSRF-TOKEN': '{{ csrf_token() }}',
            'Accept': 'application/json'
        },
        body: JSON.stringify({
            template_base64: templateBase64,
            quality: 50
        })
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            return data;
        } else {
            throw new Error(data.message || 'Fingerprint match failed');
        }
    });
}

// Modified to accept location parameter
function recordAttendance(matchData, location) {
    const now = new Date();
    const attendanceDate = now.toISOString().split('T')[0];
    const attendanceTime = now.toTimeString().split(' ')[0];
    // Log if location is default/fallback
    if (isDefaultLocation(location)) {
        console.warn('Using fallback/default location for attendance:', location);
    }
    
    return fetch('{{ route("attendance.mark_fingerprint") }}', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
            'X-CSRF-TOKEN': '{{ csrf_token() }}',
            'Accept': 'application/json'
        },
        body: JSON.stringify({
            staff_id: matchData.staff_id,
            attendance_date: attendanceDate,
            attendance_time: attendanceTime,
            latitude: location.latitude,
            longitude: location.longitude,
            address: location.address
        })
    })
    .then(response => {
        if (!response.ok) {
            throw new Error('Network response was not ok');
        }
        return response.json();
    })
    .then(data => {
        // console.log('Server response:', data);
        
        if (data[0] && data[0].status) {
            return {
                success: true,
                message: data[0].message
            };
        } else {
            throw new Error(data[0]?.message || 'Attendance recording failed');
        }
    })
    .catch(error => {
        console.error('Error recording attendance:', error);
        throw error;
    });
}

// Pre-fetch location on page load
function prefetchLocation() {
    // console.log('Pre-fetching location...');
    getLocation()
        .then(location => {
            currentLocation = location;
            // console.log('Location pre-fetched:', location);
        })
        .catch(error => {
            console.warn('Failed to pre-fetch location:', error.message);
        });
}

function getLocation() {
    return new Promise((resolve, reject) => {
        // console.log('Attempting to get location...');
        
        if (!navigator.geolocation) {
            console.warn('Geolocation not supported by browser');
            resolve({ 
                latitude: null, 
                longitude: null, 
                address: 'Geolocation not supported by browser'
            });
            return;
        }
        
        // Check if we have cached location (less than 30 seconds old)
        if (currentLocation && locationPromise) {
            // console.log('Using cached location');
            resolve(currentLocation);
            return;
        }
        
        // Create a new location promise if none exists
        if (!locationPromise) {
            locationPromise = new Promise((innerResolve, innerReject) => {
                const options = {
                    enableHighAccuracy: true,
                    timeout: 10000,
                    maximumAge: 30000
                };

                navigator.geolocation.getCurrentPosition(
                    async (position) => {
                        // console.log('Location obtained successfully:', position.coords);
                        
                        const lat = position.coords.latitude;
                        const lng = position.coords.longitude;
                        
                        let address = `Coordinates: ${lat.toFixed(6)}, ${lng.toFixed(6)}`;
                        
                        try {
                            // Converts GPS coordinates (latitude/longitude) into a human-readable address
                            const response = await fetch(
                                `https://nominatim.openstreetmap.org/reverse?lat=${lat}&lon=${lng}&format=json&addressdetails=1`
                            );
                            const data = await response.json();
                            
                            if (data.display_name) {
                                address = data.display_name;
                            } else if (data.address) {
                                const addrParts = [];
                                if (data.address.road) addrParts.push(data.address.road);
                                if (data.address.suburb) addrParts.push(data.address.suburb);
                                if (data.address.city) addrParts.push(data.address.city);
                                if (data.address.state) addrParts.push(data.address.state);
                                if (data.address.country) addrParts.push(data.address.country);
                                
                                if (addrParts.length > 0) {
                                    address = addrParts.join(', ');
                                }
                            }
                        } catch (error) {
                            // console.log('Address lookup failed:', error.message);
                        }
                        
                        const location = {
                            latitude: lat,
                            longitude: lng,
                            address: address
                        };
                        
                        currentLocation = location;
                        innerResolve(location);
                    },
                );
            });
        }
        
        locationPromise.then(resolve).catch(reject);
    });
}


function getGeolocationError(code) {
    switch(code) {
        case 1: return 'Permission denied';
        case 2: return 'Position unavailable';
        case 3: return 'Timeout';
        default: return 'Unknown error';
    }
}

function speak(text) {
    if ('speechSynthesis' in window) {
        const utterance = new SpeechSynthesisUtterance(text);
        utterance.rate = 1.0;
        utterance.pitch = 1.0;
        window.speechSynthesis.speak(utterance);
    }
}

function errorCodeToString(errorCode) {
    const errors = {
        51: "System file load failure",
        52: "Sensor chip initialization failed",
        53: "Device not found",
        54: "Fingerprint image capture timeout",
        55: "No device available",
        56: "Driver load failed",
        57: "Wrong Image",
        58: "Lack of bandwidth",
        59: "Device Busy",
        60: "Cannot get serial number of the device",
        61: "Unsupported device",
        63: "SgiBioSrv didn't start; Try image capture again"
    };
    return errors[errorCode] || "Unknown error";
}

document.addEventListener('DOMContentLoaded', function() {
    clearAllTimers();
    startQRTimer();
    prefetchLocation();
    setTimeout(() => {
        const qrImages = document.querySelectorAll('#qrcode img, #qrcode svg');
        qrImages.forEach(img => {
            img.style.width = '220px';
            img.style.height = '220px';
            img.style.maxWidth = 'none';
        });
    }, 500);
});

// Clean up timers when page unloads
window.addEventListener('beforeunload', function() {
    clearAllTimers();
});

// Also clean up timers when page is hidden (optional)
document.addEventListener('visibilitychange', function() {
    if (document.hidden) {
        clearAllTimers();
    }
});
</script>
@endsection