@extends('layouts.horizontal', ['title' => 'Staff Leave Report', 'topbarTitle' => 'Staff Leave Report'])

@section('css')
@vite(['node_modules/datatables.net-bs5/css/dataTables.bootstrap5.min.css',
'node_modules/datatables.net-responsive-bs5/css/responsive.bootstrap5.min.css',
'node_modules/datatables.net-fixedcolumns-bs5/css/fixedColumns.bootstrap5.min.css',
'node_modules/datatables.net-fixedheader-bs5/css/fixedHeader.bootstrap5.min.css',
'node_modules/datatables.net-buttons-bs5/css/buttons.bootstrap5.min.css',
'node_modules/datatables.net-select-bs5/css/select.bootstrap5.min.css',
'node_modules/select2/dist/css/select2.min.css',
'node_modules/flatpickr/dist/flatpickr.min.css'])
<style>
    .filter-row { background-color: #f9f9f9; }
    .form-filter { font-size: 12px; padding: 4px 6px; height: 30px; width: 100%; }
    .filter-row th { padding: 4px 6px !important; vertical-align: middle; }
    .filter-buttons { display: flex; gap: 5px; }
    .filter-buttons button { padding: 4px 8px; font-size: 12px; }
    
    .card {
        box-shadow: 0 0.125rem 0.25rem rgba(0, 0, 0, 0.075);
    }
    
    .card-background {
        background: #f8f9fa;
        border-radius: 10px;
        padding: 15px;
        margin-bottom: 15px;
        border: 1px solid #e9ecef;
    }
    
</style>
@endsection

@section('content')
<div class="container-fluid">
    <div class="row mb-4">
        <div class="col-12">
            <div class="page-title-box d-flex align-items-center justify-content-between">
                <h2 class="mb-0">Staff Leave Report</h2>
                 <div class="page-title-right d-flex align-items-center gap-3">
                    @php
                        $breadcrumb = getCurrentMenuBreadcrumb();
                    @endphp
                    @if(!empty($breadcrumb))
                        <nav aria-label="breadcrumb">
                            <ol class="breadcrumb mb-0 py-2">
                                @foreach($breadcrumb as $index => $item)
                                    @if($index === count($breadcrumb) - 1)
                                        <li class="breadcrumb-item active" aria-current="page">{{ $item }}</li>
                                    @else
                                        <li class="breadcrumb-item"><a href="javascript: void(0);">{{ $item }}</a></li>
                                    @endif
                                @endforeach
                            </ol>
                        </nav>
                    @endif
                </div>
            </div>
        </div>
    </div>

    @if(session('Success'))
    <div class="row">
        <div class="col-12">
            <div class="alert alert-success alert-dismissible fade show" role="alert">
                <strong>Success!</strong> {{ session('Success') }}
                <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
            </div>
        </div>
    </div>
    @endif

    @if(session('Error'))
    <div class="row">
        <div class="col-12">
            <div class="alert alert-danger alert-dismissible fade show" role="alert">
                <strong>Error!</strong> {{ session('Error') }}
                <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
            </div>
        </div>
    </div>
    @endif

    @if($isHoUser)
    <div class="row mb-3">
        <div class="col-sm-12">
            <div class="card">
                <div class="card-body">
                    <div class="row align-items-center">
                            <div class="col-md-12">
                            <label class="form-label">Select Business</label>
                            <select class="form-control select2" id="businessSelector">
                                <option value="">All Businesses</option>
                                @foreach($businesses as $biz)
                                    <option value="{{ $biz->id_business }}" 
                                        {{ (session('business_id') == $biz->id_business) ? 'selected' : '' }}>
                                        {{ $biz->business_name }}
                                    </option>
                                @endforeach
                            </select>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
    @endif

    <!-- Date Range Filter -->
    <div class="row mb-3">
        <div class="col-12">
            <div class="card date-range-filter">
                <div class="card-body">
                    <div class="row align-items-center">
                        <div class="col-md-12">
                            <label class="form-label">Date Range</label>
                            <div class="input-group">
                                <input type="text" class="form-control" id="leave_dateRange" name="leave_date_range" placeholder="Select date range" readonly required>
                                <input type="hidden" id="leave_start_date" name="leave_start_date">
                                <input type="hidden" id="leave_end_date" name="leave_end_date">
                                <div class="gap-2" >
                                    <button type="button" class="btn btn-ghost-success btn-icon rounded-circle p-3" id="applyDateFilter">
                                        <iconify-icon icon="mdi:magnify" width="30px"></iconify-icon>
                                    </button>
                                    <button class="btn btn-ghost-success btn-icon rounded-circle p-3" id="resetDateFilter">
                                     <iconify-icon icon="mdi:refresh" width="30px"></iconify-icon>
                                    </button>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Staff Leave Report Table -->
    <div class="row">
        <div class="col-12">
            <div class="card">
                <div class="card-body">
                    <div class="table-responsive">
                        <table id="staff_leave_report_table" class="table table-bordered table-hover fs-13 w-100">
                            <thead>
                                <tr>
                                    <th class="text-center">Staff Name & ID</th>
                                    <th class="text-center">Business</th>
                                    <th class="text-center">Leave Policy Status</th>
                                    <th class="text-center">Policy Leaves (Allowed)</th>
                                    <th class="text-center">Consumed Leaves</th>
                                    <th class="text-center">Remaining Leaves</th>
                                    <th class="text-center">Leave Details</th>
                                    <th class="text-center">Status</th>
                                </tr>
                            </thead>
                            <tbody id="leaveDataBody">
                                <!-- Data will be loaded via JavaScript -->
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>


<div id="loadingOverlay" style="display: none; position: fixed; top: 0; left: 0; width: 100%; height: 100%; background-color: rgba(0,0,0,0.5); z-index: 9999; display: flex; justify-content: center; align-items: center;">
    <div class="spinner-border text-light" role="status">
        <span class="visually-hidden">Loading...</span>
    </div>
</div>
@endsection

@section('scripts')
<script>
    let leaveFlatpickr;
    
    $(document).ready(function() {
        initializeDatePicker();
        initializeSelect2();
        setupEventHandlers();
        loadLeaveData();
    });

   function initializeDatePicker() {
        // Set default date range (current complete year)
        const currentYear = new Date().getFullYear();
        const defaultStartDate = new Date(currentYear, 0, 1); // Jan 1 of current year
        const defaultEndDate = new Date(currentYear, 11, 31); // Dec 31 of current year
        
        leaveFlatpickr = flatpickr("#leave_dateRange", {
            mode: "range",
            dateFormat: "Y-m-d",
            defaultDate: [defaultStartDate, defaultEndDate],
            onChange: function(selectedDates, dateStr, instance) {
                if (selectedDates.length === 2) {
                    updateLeaveDateRangeFields(selectedDates[0], selectedDates[1]);
                } else if (selectedDates.length === 1) {
                    // Waiting for second date selection
                } else {
                    $('#leave_dateRange').val('');
                    $('#leave_start_date').val('');
                    $('#leave_end_date').val('');
                }
            },
        });
        
        // Set initial values
        updateLeaveDateRangeFields(defaultStartDate, defaultEndDate);
    }

    function updateLeaveDateRangeFields(startDate, endDate) {
        const startStr = formatDate(startDate);
        const endStr = formatDate(endDate);
        
        $('#leave_dateRange').val(startStr + ' to ' + endStr);
        $('#leave_start_date').val(startStr);
        $('#leave_end_date').val(endStr);
    }

    function formatDate(date) {
        const d = new Date(date);
        const year = d.getFullYear();
        const month = String(d.getMonth() + 1).padStart(2, '0');
        const day = String(d.getDate()).padStart(2, '0');
        return `${year}-${month}-${day}`;
    }

    function initializeSelect2() {
        $('#businessSelector').select2({
            width: '100%',
            placeholder: 'Select Business'
        });
    }

    function setupEventHandlers() {
        // Apply date filter button
        $('#applyDateFilter').on('click', function() {
            loadLeaveData();
        });
        
       $('#resetDateFilter').on('click', function() {
            const currentYear = new Date().getFullYear();
            const defaultStartDate = new Date(currentYear, 0, 1);
            const defaultEndDate = new Date(currentYear, 11, 31);
            
            leaveFlatpickr.setDate([defaultStartDate, defaultEndDate], true);
            updateLeaveDateRangeFields(defaultStartDate, defaultEndDate);
            
            loadLeaveData();
        });
        
        $('#businessSelector').on('change', function() {
            loadLeaveData();
        });
    }
    
    function loadLeaveData() {
        const businessId = $('#businessSelector').val();
        const startDate = $('#leave_start_date').val();
        const endDate = $('#leave_end_date').val();
        
        if (!startDate || !endDate) {
            Swal.fire({
                title: 'Error',
                text: 'Please select a date range first',
                icon: 'error',
                customClass: {
                    confirmButton: "btn btn-danger"
                },
                buttonsStyling: false
            });
            return;
        }
        
        showLoading();
        
        $.ajax({
            url: "{{ route('hrm.get_staff_leave_report_data') }}",
            type: 'POST',
            data: {
                business_id: businessId,
                date_range: {
                    start: startDate,
                    end: endDate
                },
                _token: "{{ csrf_token() }}"
            },
            success: function(response) {
                if (response.success) {
                    renderLeaveTable(response.data);
                } else {
                    $('#leaveDataBody').html('<tr><td colspan="8" class="text-center text-danger">Error loading data: ' + response.message + '</td></tr>');
                }
            },
            error: function(xhr, status, error) {
                console.error('Error loading data:', error);
                $('#leaveDataBody').html('<tr><td colspan="8" class="text-center text-danger">Error loading data. Please try again.</td></tr>');
                
                if (xhr.responseJSON && xhr.responseJSON.error) {
                    Swal.fire({
                        title: 'Error',
                        text: xhr.responseJSON.error,
                        icon: 'error',
                        customClass: {
                            confirmButton: "btn btn-danger"
                        },
                        buttonsStyling: false
                    });
                }
            },
            complete: function() {
                hideLoading();
            }
        });
    }

    function renderLeaveTable(data) {
        const tbody = $('#leaveDataBody');
        tbody.empty();
        
        if (!data || data.length === 0) {
            tbody.html('<tr><td colspan="8" class="text-center">No leave data found for the selected criteria</td></tr>');
            return;
        }
        
        data.forEach(function(staff) {
            const staffRow = $('<tr>').addClass('staff-info').appendTo(tbody);
            
            // Staff Name & ID
            $('<td>')
                .addClass('staff-info-cell')
                .html(`<div class="d-flex flex-column">
                          <span class="fw-bold text-primary mb-1">${staff.staff_name}</span>
                          <small class="text-muted">ID: ${staff.staff_id}</small>
                       </div>`)
                .appendTo(staffRow);
            
            // Business name
            $('<td>')
                .addClass('align-middle')
                .html(`<div class="fw-semibold">${staff.business_name}</div>`)
                .appendTo(staffRow);
            
            // Leave Policy Status
            const policyCell = $('<td>')
                .addClass('align-middle text-center')
                .appendTo(staffRow);
            
            if (staff.no_policy) {
                policyCell.html('<span class="badge badge-outline-danger">No Policy Assigned</span>');
            } else {
                policyCell.html(`<span class="badge badge-outline-success">${staff.policy_name}</span>`);
            }
            
            // Policy Leaves (Allowed)
            $('<td>')
                .addClass('align-middle text-center')
                .html(staff.policy_leaves || '<span class="text-muted">N/A</span>')
                .appendTo(staffRow);
            
            // Consumed Leaves
            $('<td>')
                .addClass('align-middle text-center fw-bold')
                .html(`<span class="${staff.exceeded ? 'text-danger' : 'text-success'}">${staff.consumed_leaves}</span>`)
                .appendTo(staffRow);
            
            // Remaining Leaves
            $('<td>')
                .addClass('align-middle text-center fw-bold')
                .html(staff.remaining_leaves_display)
                .appendTo(staffRow);
            
            // Leave Details
            $('<td>')
                .addClass('align-middle')
                .html(staff.leave_details)
                .appendTo(staffRow);
            
            // Status
            $('<td>')
                .addClass('align-middle text-center')
                .html(staff.status_display)
                .appendTo(staffRow);
        });
    }

    function showLoading() {
        $('#loadingOverlay').show();
    }
    
    function hideLoading() {
        $('#loadingOverlay').hide();
    }
</script>
@endsection