@extends('layouts.settings', ['title' => 'Role Permissions', 'topbarTitle' => 'Settings'])

@section('css')
<style>
    .permission-group {
        border: 1px solid #e9ecef;
        border-radius: 8px;
        padding: 15px;
        margin-bottom: 20px;
        background: #f8f9fa;
    }
    .permission-group-title {
        font-weight: 600;
        color: #495057;
        margin-bottom: 15px;
        text-transform: capitalize;
        font-size: 16px;
    }
    .permission-item {
        display: flex;
        align-items: center;
        padding: 8px 12px;
        margin-bottom: 8px;
        background: white;
        border-radius: 4px;
        border: 1px solid #dee2e6;
    }
    .permission-item:hover {
        background: #f8f9fa;
    }
    .role-card {
        border: 1px solid #dee2e6;
        border-radius: 8px;
        margin-bottom: 20px;
    }
    .role-card-header {
        background: #f8f9fa;
        padding: 15px;
        border-bottom: 1px solid #dee2e6;
        font-weight: 600;
    }
    .role-card-body {
        padding: 15px;
    }
</style>
@endsection

@section('content')
<div class="container-fluid">
    <div class="row">
        <div class="col-12">
            <div class="page-title-box d-sm-flex align-items-center justify-content-between">
                <h4 class="mb-sm-0">Role Permissions Management</h4>
                <div class="page-title-right">
                    <ol class="breadcrumb m-0">
                        <li class="breadcrumb-item"><a href="javascript: void(0);">Settings</a></li>
                        <li class="breadcrumb-item active">Role Permissions</li>
                    </ol>
                </div>
            </div>
        </div>
    </div>

    <!-- Success/Error Messages -->
    @if(session('success'))
    <div class="alert alert-success alert-dismissible fade show" role="alert">
        {{ session('success') }}
        <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
    </div>
    @endif

    @if(session('error'))
    <div class="alert alert-danger alert-dismissible fade show" role="alert">
        {{ session('error') }}
        <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
    </div>
    @endif

    <div class="row">
        <!-- Permissions Section -->
        <div class="col-xl-4">
            <div class="card">
                <div class="card-header d-flex justify-content-between align-items-center">
                    <h5 class="card-title mb-0">Permissions</h5>
                    <button type="button" class="btn btn-sm btn-primary" data-bs-toggle="modal" data-bs-target="#addPermissionModal">
                        <i class="ti ti-plus"></i> Add Permission
                    </button>
                </div>
                <div class="card-body">
                    @if($permissions->isEmpty())
                        <p class="text-muted text-center">No permissions created yet.</p>
                    @else
                        @foreach($groupedPermissions as $group => $groupPermissions)
                            <div class="permission-group">
                                <div class="permission-group-title">
                                    <i class="ti ti-folder me-2"></i>{{ $group }}
                                </div>
                                @foreach($groupPermissions as $permission)
                                    <div class="permission-item">
                                        <div class="flex-grow-1">
                                            <strong>{{ $permission->name }}</strong>
                                            <br>
                                            <small class="text-muted">Guard: {{ $permission->guard_name }}</small>
                                        </div>
                                        <div class="ms-2">
                                            <button class="btn btn-sm btn-info" onclick="editPermission({{ $permission->id }})" title="Edit">
                                                <i class="ti ti-edit"></i>
                                            </button>
                                            <button class="btn btn-sm btn-danger" onclick="deletePermission({{ $permission->id }}, '{{ $permission->name }}')" title="Delete">
                                                <i class="ti ti-trash"></i>
                                            </button>
                                        </div>
                                    </div>
                                @endforeach
                            </div>
                        @endforeach
                    @endif
                </div>
            </div>
        </div>

        <!-- Roles Section -->
        <div class="col-xl-8">
            <div class="card">
                <div class="card-header">
                    <h5 class="card-title mb-0">Roles & Permissions</h5>
                </div>
                <div class="card-body">
                    @if($roles->isEmpty())
                        <p class="text-muted text-center">No roles found. Please sync roles first.</p>
                    @else
                        @foreach($roles as $role)
                            <div class="role-card">
                                <div class="role-card-header">
                                    <div class="d-flex justify-content-between align-items-center">
                                        <div>
                                            <strong>{{ $role->name }}</strong>
                                            <small class="text-muted ms-2">({{ $role->permissions->count() }} permissions)</small>
                                        </div>
                                        <button class="btn btn-sm btn-primary" onclick="openRolePermissionsModal({{ $role->id }}, '{{ $role->name }}')">
                                            <i class="ti ti-settings"></i> Manage Permissions
                                        </button>
                                    </div>
                                </div>
                                <div class="role-card-body">
                                    @if($role->permissions->isEmpty())
                                        <p class="text-muted mb-0">No permissions assigned to this role.</p>
                                    @else
                                        <div class="d-flex flex-wrap gap-2">
                                            @foreach($role->permissions as $permission)
                                                <span class="badge bg-primary">{{ $permission->name }}</span>
                                            @endforeach
                                        </div>
                                    @endif
                                </div>
                            </div>
                        @endforeach
                    @endif
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Add Permission Modal -->
<div class="modal fade" id="addPermissionModal" tabindex="-1" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Add New Permission</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <form id="addPermissionForm" action="{{ route('settings.role-permissions.permissions.store') }}" method="POST">
                @csrf
                <div class="modal-body">
                    <div class="mb-3">
                        <label for="permission_name" class="form-label">Permission Name <span class="text-danger">*</span></label>
                        <input type="text" class="form-control" id="permission_name" name="name" required placeholder="e.g., users.create">
                        <small class="text-muted">Use dot notation (e.g., users.create, posts.edit)</small>
                    </div>
                    <div class="mb-3">
                        <label for="permission_guard" class="form-label">Guard Name</label>
                        <input type="text" class="form-control" id="permission_guard" name="guard_name" value="web" placeholder="web">
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-primary">Create Permission</button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Edit Permission Modal -->
<div class="modal fade" id="editPermissionModal" tabindex="-1" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Edit Permission</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <form id="editPermissionForm" method="POST">
                @csrf
                @method('PUT')
                <div class="modal-body">
                    <div class="mb-3">
                        <label for="edit_permission_name" class="form-label">Permission Name <span class="text-danger">*</span></label>
                        <input type="text" class="form-control" id="edit_permission_name" name="name" required>
                    </div>
                    <div class="mb-3">
                        <label for="edit_permission_guard" class="form-label">Guard Name</label>
                        <input type="text" class="form-control" id="edit_permission_guard" name="guard_name" value="web">
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-primary">Update Permission</button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Assign Permissions to Role Modal -->
<div class="modal fade" id="rolePermissionsModal" tabindex="-1" aria-hidden="true">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Manage Permissions for: <span id="roleNameDisplay"></span></h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <form id="assignPermissionsForm" method="POST">
                @csrf
                <div class="modal-body">
                    <div class="mb-3">
                        <button type="button" class="btn btn-sm btn-link" onclick="selectAllPermissions()">Select All</button>
                        <button type="button" class="btn btn-sm btn-link" onclick="deselectAllPermissions()">Deselect All</button>
                    </div>
                    <div style="max-height: 400px; overflow-y: auto;">
                        @foreach($groupedPermissions as $group => $groupPermissions)
                                <div class="permission-group mb-3" data-group="{{ $group }}">
                                <div class="permission-group-title">
                                    <input type="checkbox" class="form-check-input group-checkbox" data-group="{{ $group }}" onchange="toggleGroupPermissions('{{ $group }}', this.checked)">
                                    <label class="ms-2">{{ $group }}</label>
                                </div>
                                <div class="ms-4">
                                    @foreach($groupPermissions as $permission)
                                        <div class="form-check mb-2">
                                            <input class="form-check-input permission-checkbox" type="checkbox" name="permissions[]" value="{{ $permission->id }}" id="perm_{{ $permission->id }}" data-group="{{ $group }}">
                                            <label class="form-check-label" for="perm_{{ $permission->id }}">
                                                {{ $permission->name }}
                                            </label>
                                        </div>
                                    @endforeach
                                </div>
                            </div>
                        @endforeach
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-primary">Save Permissions</button>
                </div>
            </form>
        </div>
    </div>
</div>
@endsection

@section('scripts')
<script>
    function editPermission(id) {
        fetch(`{{ url('settings/role-permissions/permissions') }}/${id}`)
            .then(response => response.json())
            .then(data => {
                document.getElementById('edit_permission_name').value = data.name;
                document.getElementById('edit_permission_guard').value = data.guard_name;
                document.getElementById('editPermissionForm').action = `{{ url('settings/role-permissions/permissions') }}/${id}`;
                new bootstrap.Modal(document.getElementById('editPermissionModal')).show();
            })
            .catch(error => {
                console.error('Error:', error);
                alert('Failed to load permission data.');
            });
    }

    function deletePermission(id, name) {
        if (confirm(`Are you sure you want to delete the permission "${name}"? This action cannot be undone.`)) {
            const form = document.createElement('form');
            form.method = 'POST';
            form.action = `{{ url('settings/role-permissions/permissions') }}/${id}`;
            form.innerHTML = `
                @csrf
                @method('DELETE')
            `;
            document.body.appendChild(form);
            form.submit();
        }
    }

    function openRolePermissionsModal(roleId, roleName) {
        document.getElementById('roleNameDisplay').textContent = roleName;
        document.getElementById('assignPermissionsForm').action = `{{ url('settings/role-permissions/roles') }}/${roleId}/permissions`;
        
        // Clear all checkboxes first
        document.querySelectorAll('.permission-checkbox').forEach(cb => cb.checked = false);
        document.querySelectorAll('.group-checkbox').forEach(cb => cb.checked = false);
        
        // Load current permissions for this role
        fetch(`{{ url('settings/role-permissions/roles') }}/${roleId}`)
            .then(response => response.json())
            .then(data => {
                if (data.permissions && data.permissions.length > 0) {
                    data.permissions.forEach(perm => {
                        const checkbox = document.getElementById(`perm_${perm.id}`);
                        if (checkbox) {
                            checkbox.checked = true;
                        }
                    });
                    updateGroupCheckboxes();
                }
                new bootstrap.Modal(document.getElementById('rolePermissionsModal')).show();
            })
            .catch(error => {
                console.error('Error:', error);
                new bootstrap.Modal(document.getElementById('rolePermissionsModal')).show();
            });
    }

    function selectAllPermissions() {
        document.querySelectorAll('.permission-checkbox').forEach(cb => cb.checked = true);
        document.querySelectorAll('.group-checkbox').forEach(cb => cb.checked = true);
    }

    function deselectAllPermissions() {
        document.querySelectorAll('.permission-checkbox').forEach(cb => cb.checked = false);
        document.querySelectorAll('.group-checkbox').forEach(cb => cb.checked = false);
    }

    function toggleGroupPermissions(group, checked) {
        const groupPermissions = document.querySelectorAll(`.permission-checkbox[data-group="${group}"]`);
        groupPermissions.forEach(cb => cb.checked = checked);
    }

    function updateGroupCheckboxes() {
        document.querySelectorAll('.group-checkbox').forEach(groupCb => {
            const group = groupCb.dataset.group;
            const groupPermissions = document.querySelectorAll(`.permission-checkbox[data-group="${group}"]`);
            const allChecked = groupPermissions.length > 0 && Array.from(groupPermissions).every(cb => cb.checked);
            groupCb.checked = allChecked;
        });
    }

    // Update group checkboxes when individual permissions change
    document.addEventListener('DOMContentLoaded', function() {
        document.querySelectorAll('.permission-checkbox').forEach(cb => {
            cb.addEventListener('change', updateGroupCheckboxes);
        });
    });
</script>
@endsection
