@extends('layouts.horizontal', ['title' => 'Custom SMS', 'topbarTitle' => 'Custom SMS'])

@section('css')
@vite(['node_modules/select2/dist/css/select2.min.css'])
@endsection

@section('content')
<div class="container-fluid">
    <div class="row page-titles mb-3">
        <div class="col-md-5 align-self-center">
            <h2 class="text-themecolor">Custom SMS</h2>
        </div>
    </div>

    <!-- Container 1: SMS Composer -->
    <div class="row">
        <div class="col-12">
            <div class="card">
                <div class="card-header">
                    <h4 class="card-title mb-0">SMS Composer</h4>
                </div>
                <div class="card-body">
                    <form id="smsForm">
                        <div class="row">
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label for="cellNumber" class="form-label">Cell Number</label>
                                    <div class="input-group">
                                        <span class="input-group-text">+92</span>
                                        <input type="text" class="form-control" id="cellNumber" 
                                               placeholder="Enter 10 digits" maxlength="10"
                                               autocomplete="off">
                                    </div>
                                    <div id="cellNumberError" class="text-danger small mt-1" style="display: none;"></div>
                                    <div id="customerSuggestions" class="list-group position-absolute w-100" style="z-index: 1000; display: none; max-height: 300px; overflow-y: auto;"></div>
                                </div>
                            </div>
                        </div>
                        <div class="row">
                            <div class="col-md-8">
                                <div class="mb-3">
                                    <label for="smsMessage" class="form-label">Message</label>
                                    <textarea class="form-control" id="smsMessage" rows="4" 
                                              placeholder="Enter your message"></textarea>
                                    <div class="mt-1">
                                        <small id="charCounter" class="text-muted">Character limit: 0 of 160</small>
                                    </div>
                                </div>
                            </div>
                        </div>
                        <div class="row">
                            <div class="col-12">
                                <button type="button" id="sendBtn" class="btn btn-primary me-2" onclick="sendSMS()">
                                    <i class="ti ti-send me-1"></i> Send
                                </button>
                                <button type="button" class="btn btn-secondary" onclick="clearMessage()">
                                    <i class="ti ti-eraser me-1"></i> Clear
                                </button>
                            </div>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    </div>

    <!-- Container 2: Customers Table -->
    <div class="row">
        <div class="col-12">
            <div class="card">
                <div class="card-header">
                    <h4 class="card-title mb-0">Customers</h4>
                </div>
                <div class="card-body">
                    <!-- Server-Side Filters -->
                    <div class="row mb-3">
                        <div class="col-md-4">
                            <label for="filterGender" class="form-label">Gender</label>
                            <select id="filterGender" class="form-select" data-toggle="select2">
                                <option value="">All</option>
                                <option value="Male">Male</option>
                                <option value="Female">Female</option>
                                <option value="Other">Other</option>
                            </select>
                        </div>
                        <div class="col-md-4">
                            <label for="filterMobileLength" class="form-label">Mobile Number Length</label>
                            <select id="filterMobileLength" class="form-select" data-toggle="select2">
                                <option value="">All</option>
                                <option value="11">11</option>
                                <option value="12">12</option>
                                <option value="13">13</option>
                                <option value="14">14</option>
                                <option value="15">15</option>
                            </select>
                        </div>
                        <div class="col-md-4">
                            <label for="filterDisplayFormat" class="form-label">Display Format</label>
                            <select id="filterDisplayFormat" class="form-select" data-toggle="select2">
                                <option value="as_saved">Show as saved</option>
                                <option value="prefix_0">Prefix with 0</option>
                                <option value="prefix_country">Prefix with country code</option>
                            </select>
                        </div>
                    </div>
                    <div class="table-responsive">
                        <table id="customersTable" class="table table-striped table-bordered w-100 fs-13">
                            <thead>
                                <tr>
                                    <th>ID</th>
                                    <th>Customer Name</th>
                                    <th>Gender</th>
                                    <th>Care Of</th>
                                    <th>Cell Number</th>
                                </tr>
                            </thead>
                            <tbody>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>
@endsection

@section('scripts')
<script>
    let searchTimeout = null;

    document.addEventListener('DOMContentLoaded', function() {
        initCustomersTable();
        initCellNumberInput();
        initCharacterCounter();
        initFilters();
    });

    function initFilters() {
        // Reload table when any filter changes
        $('#filterGender, #filterMobileLength, #filterDisplayFormat').on('change', function() {
            if (customersTable) {
                customersTable.ajax.reload();
            }
        });
    }

    let customersTable;

    function initCustomersTable() {
        if ($.fn.DataTable.isDataTable('#customersTable')) {
            $('#customersTable').DataTable().clear().destroy();
        }

        customersTable = $('#customersTable').DataTable({
            processing: true,
            serverSide: true,
            ajax: {
                url: '{{ route("sms.customers-data") }}',
                type: 'GET',
                data: function(d) {
                    d.filter_gender = $('#filterGender').val();
                    d.filter_mobile_length = $('#filterMobileLength').val();
                    d.filter_display_format = $('#filterDisplayFormat').val();
                }
            },
            columns: [
                { data: 'id_customers', name: 'id_customers' },
                { data: 'customer_name', name: 'customer_name' },
                { data: 'customer_gender', name: 'customer_gender' },
                { data: 'care_of', name: 'care_of' },
                { data: 'customer_cell', name: 'customer_cell' }
            ],
            pageLength: 10,
            order: [[0, 'desc']],
            layout: {
                topStart: 'buttons',
                topEnd: 'search',
                bottomStart: {
                    pageLength: {
                        menu: [10, 20, 50, 100]
                    },
                    info: {}
                },
                bottomEnd: 'paging'
            },
            buttons: [
                { extend: 'copy', className: 'btn btn-sm btn-soft-primary' },
                { extend: 'csv', className: 'btn btn-sm btn-soft-primary' },
                { extend: 'print', className: 'btn btn-sm btn-soft-primary' }
            ],
            language: {
                search: 'Search:',
                searchPlaceholder: 'Customer Name or ID'
            }
        });
    }

    function initCellNumberInput() {
        const cellInput = document.getElementById('cellNumber');
        const suggestionsDiv = document.getElementById('customerSuggestions');

        cellInput.addEventListener('input', function(e) {
            // Only allow digits
            this.value = this.value.replace(/\D/g, '');
            
            validateCellNumber();

            // Debounced customer lookup
            clearTimeout(searchTimeout);
            if (this.value.length >= 4) {
                searchTimeout = setTimeout(() => {
                    searchCustomers(this.value);
                }, 300);
            } else {
                suggestionsDiv.style.display = 'none';
            }
        });

        // Hide suggestions when clicking outside
        document.addEventListener('click', function(e) {
            if (!cellInput.contains(e.target) && !suggestionsDiv.contains(e.target)) {
                suggestionsDiv.style.display = 'none';
            }
        });
    }

    function validateCellNumber() {
        const cellInput = document.getElementById('cellNumber');
        const errorDiv = document.getElementById('cellNumberError');
        const value = cellInput.value;

        if (value.length === 0) {
            errorDiv.style.display = 'none';
            return false;
        }

        if (value.length !== 10) {
            errorDiv.textContent = 'Cell number must be exactly 10 digits';
            errorDiv.style.display = 'block';
            return false;
        }

        errorDiv.style.display = 'none';
        return true;
    }

    function searchCustomers(term) {
        const suggestionsDiv = document.getElementById('customerSuggestions');

        $.ajax({
            url: '{{ route("sms.customer-lookup") }}',
            type: 'GET',
            data: { term: term },
            success: function(response) {
                if (response.length > 0) {
                    let html = '';
                    response.forEach(function(customer) {
                        html += `<a href="#" class="list-group-item list-group-item-action" 
                                    onclick="selectCustomer('${customer.cell_normalized}'); return false;">
                                    ${customer.customer_name} - ${customer.customer_cell}
                                </a>`;
                    });
                    suggestionsDiv.innerHTML = html;
                    suggestionsDiv.style.display = 'block';
                } else {
                    suggestionsDiv.style.display = 'none';
                }
            }
        });
    }

    function selectCustomer(cellNumber) {
        document.getElementById('cellNumber').value = cellNumber;
        document.getElementById('customerSuggestions').style.display = 'none';
        validateCellNumber();
    }

    const MAX_MESSAGE_LENGTH = 320;
    const SMS_CHAR_LIMIT = 160;

    function initCharacterCounter() {
        const messageInput = document.getElementById('smsMessage');
        const counter = document.getElementById('charCounter');

        messageInput.setAttribute('maxlength', MAX_MESSAGE_LENGTH);

        messageInput.addEventListener('input', function() {
            const length = this.value.length;
            const smsCount = Math.ceil(length / SMS_CHAR_LIMIT) || 1;
            counter.textContent = `Character limit: ${length} of ${MAX_MESSAGE_LENGTH} (${smsCount} SMS)`;
            
            if (length > SMS_CHAR_LIMIT) {
                counter.classList.remove('text-muted');
                counter.classList.add('text-danger');
            } else {
                counter.classList.remove('text-danger');
                counter.classList.add('text-muted');
            }
        });
    }

    function sendSMS() {
        const cellInput = document.getElementById('cellNumber');
        const messageInput = document.getElementById('smsMessage');
        const sendBtn = document.getElementById('sendBtn');

        if (!validateCellNumber()) {
            Swal.fire({
                icon: 'error',
                title: 'Invalid Cell Number',
                text: 'Please enter a valid 10-digit cell number'
            });
            return;
        }

        if (messageInput.value.trim() === '') {
            Swal.fire({
                icon: 'error',
                title: 'Empty Message',
                text: 'Please enter a message'
            });
            return;
        }

        const charCount = messageInput.value.length;
        const smsCount = Math.ceil(charCount / SMS_CHAR_LIMIT);

        // If message exceeds 160 chars, show confirmation
        if (charCount > SMS_CHAR_LIMIT) {
            Swal.fire({
                icon: 'warning',
                title: 'Multiple SMS Warning',
                html: `Your message is <strong>${charCount}</strong> characters.<br>This message will be charged as <strong>${smsCount} SMS messages</strong>.`,
                showCancelButton: true,
                confirmButtonText: 'Send Anyway',
                cancelButtonText: 'Cancel',
                customClass: {
                    confirmButton: 'btn btn-success me-2',
                    cancelButton: 'btn btn-primary'
                }
            }).then((result) => {
                if (result.isConfirmed) {
                    performSend(cellInput.value, messageInput.value, sendBtn);
                }
            });
            return;
        }

        performSend(cellInput.value, messageInput.value, sendBtn);
    }

    function performSend(cellNumber, message, sendBtn) {
        const fullNumber = '92' + cellNumber;

        // Disable send button
        sendBtn.disabled = true;
        sendBtn.innerHTML = '<i class="ti ti-loader me-1"></i> Sending...';

        $.ajax({
            url: '{{ route("sms.send") }}',
            type: 'POST',
            data: {
                _token: '{{ csrf_token() }}',
                cell_number: fullNumber,
                message: message
            },
            success: function(response) {
                Swal.fire({
                    icon: response.success ? 'success' : 'error',
                    title: response.success ? 'SMS Sent' : 'SMS Failed',
                    text: response.message || JSON.stringify(response.raw_response),
                    customClass: {
                        confirmButton: response.success ? 'btn btn-success' : 'btn btn-danger'
                    }
                });
            },
            error: function(xhr) {
                Swal.fire({
                    icon: 'error',
                    title: 'Error',
                    text: xhr.responseJSON?.message || 'Failed to send SMS',
                    customClass: {
                        confirmButton: 'btn btn-danger'
                    }
                });
            },
            complete: function() {
                sendBtn.disabled = false;
                sendBtn.innerHTML = '<i class="ti ti-send me-1"></i> Send';
            }
        });
    }

    function clearMessage() {
        document.getElementById('smsMessage').value = '';
        document.getElementById('charCounter').textContent = 'Character limit: 0 of 160';
        document.getElementById('charCounter').classList.remove('text-danger');
        document.getElementById('charCounter').classList.add('text-muted');
    }
</script>
@endsection

