@extends('layouts.horizontal', ['title' => 'SMS Log', 'topbarTitle' => 'SMS Log'])

@section('css')
@vite(['node_modules/flatpickr/dist/flatpickr.min.css'])
@vite(['node_modules/select2/dist/css/select2.min.css'])
<style>
    /* Allow text wrapping in Select2 dropdown options */
    .select2-results__option {
        white-space: normal !important;
        word-wrap: break-word !important;
        overflow-wrap: break-word !important;
    }
    
    /* Make message cell clickable */
    .message-cell-popover {
        cursor: pointer;
        text-decoration: none;
    }
    
    .message-cell-popover:hover {
        text-decoration: underline;
    }
</style>
@endsection

@section('content')
<div class="container-fluid">
    <div class="row page-titles mb-3">
        <div class="col-md-5 align-self-center">
            <h2 class="text-themecolor">SMS Log</h2>
        </div>
    </div>

    <div class="row">
        <div class="col-12">
            <div class="card">
                <div class="card-body">
                    <div class="table-responsive">
                        <table id="smsLogTable" class="table table-striped table-bordered w-100 fs-13">
                            <thead>
                                <tr>
                                    <th>ID</th>
                                    <th>SMS Message</th>
                                    <th>Receiver</th>
                                    <th>Date Sent</th>
                                    <th>Using</th>
                                    <th class="text-center">Actions</th>
                                </tr>
                                <tr class="filters">
                                    <td><input type="text" class="form-control form-control-sm" id="filter_id" placeholder="ID"></td>
                                    <td><input type="text" class="form-control form-control-sm" id="filter_message" placeholder="SMS Message"></td>
                                    <td><input type="text" class="form-control form-control-sm" id="filter_receiver" placeholder="Receiver"></td>
                                    <td><input type="text" class="form-control form-control-sm flatpickr" id="filter_date" placeholder="Date Sent"></td>
                                    <td>
                                        <select class="form-select form-select-sm" id="filter_using" data-toggle="select2">
                                            <option value="">All</option>
                                        </select>
                                    </td>
                                    <td></td>
                                </tr>
                            </thead>
                            <tbody>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>
@endsection

@section('scripts')

<script>
    let smsLogTable;
    let filterTimeout = null;

    document.addEventListener('DOMContentLoaded', function() {
        loadUsingOptions();
        initSmsLogTable();
        initFilters();
        initDatePicker();
    });

    function loadUsingOptions() {
        $.ajax({
            url: '{{ route("sms.log-using-options") }}',
            type: 'GET',
            success: function(response) {
                const select = $('#filter_using');
                response.forEach(function(option) {
                    if (option) {
                        select.append(new Option(option, option));
                    }
                });
                // Initialize Select2 after options are loaded
                if (select.data('select2') === undefined) {
                    select.select2({
                        placeholder: 'All',
                        allowClear: true,
                        width: '100%'
                    });
                } else {
                    // If already initialized, trigger change to refresh
                    select.trigger('change');
                }
            }
        });
    }

    function initSmsLogTable() {
        if ($.fn.DataTable.isDataTable('#smsLogTable')) {
            $('#smsLogTable').DataTable().clear().destroy();
        }

        smsLogTable = $('#smsLogTable').DataTable({
            processing: true,
            serverSide: true,
            autoWidth: false,
            ajax: {
                url: '{{ route("sms.log-data") }}',
                type: 'GET',
                data: function(d) {
                    d.filter_id = $('#filter_id').val();
                    d.filter_message = $('#filter_message').val();
                    d.filter_receiver = $('#filter_receiver').val();
                    d.filter_date = $('#filter_date').val();
                    d.filter_using = $('#filter_using').val();
                }
            },
            columns: [
                { data: 'id_sms_log', name: 'id_sms_log', className: 'text-center' },
                { 
                    data: 'msgdata', 
                    name: 'msgdata',
                    className: 'text-center',
                    render: function(data, type, row) {
                        if (!data) {
                            return '';
                        }
                        
                        // Escape HTML to prevent XSS
                        const escapedData = $('<div>').text(data).html();
                        const uniqueId = 'msg-popover-' + row.id_sms_log;
                        
                        // Escape quotes and other special chars for HTML attribute
                        const attrSafeMessage = escapedData.replace(/"/g, '&quot;').replace(/'/g, '&#39;');
                        
                        // If message is longer than 50 characters, show truncated version
                        if (data.length > 50) {
                            const truncated = $('<div>').text(data.substring(0, 50)).html();
                            return '<span class="message-cell-popover" id="' + uniqueId + '" data-bs-toggle="popover" data-bs-trigger="click" data-bs-placement="top" data-message="' + attrSafeMessage + '" data-bs-title="Full Message">' + truncated + '...</span>';
                        }
                        
                        // Short messages can still be clicked to show in popover
                        return '<span class="message-cell-popover" id="' + uniqueId + '" data-bs-toggle="popover" data-bs-trigger="click" data-bs-placement="top" data-message="' + attrSafeMessage + '" data-bs-title="Full Message">' + escapedData + '</span>';
                    }
                },
                { data: 'receiver', name: 'receiver', className: 'text-center' },
                { data: 'senton', name: 'senton', className: 'text-center' },
                { data: 'using', name: 'using', className: 'text-center' },
                { 
                    data: null,
                    orderable: false,
                    className: 'text-center',
                    render: function(data, type, row) {
                        return '<button class="btn btn-sm btn-primary" onclick="resendSms(' + row.id_sms_log + ')"><i class="ti ti-send me-1"></i>Resend</button>';
                    }
                }
            ],
            columnDefs: [
                {
                    targets: 0,            // ID column
                    width: '100px',
                    className: 'text-center'
                },
                {
                    targets: [1, 2, 3],    // SMS Message, Receiver, Date Sent columns
                    className: 'text-center'
                },
                {
                    targets: 4,            // Using column
                    width: '300px',
                    className: 'text-center'
                },
                {
                    targets: 5,            // Actions
                    width: '120px',
                    orderable: false,
                    className: 'text-center'
                }
            ],
            pageLength: 10,
            order: [[0, 'desc']],
            orderCellsTop: true,
            searching: false,
            layout: {
                topStart: 'buttons',
                topEnd: null,
                bottomStart: {
                    pageLength: {
                        menu: [10, 20, 50, 100]
                    },
                    info: {}
                },
                bottomEnd: 'paging'
            },
            buttons: [
                { extend: 'copy', className: 'btn btn-sm btn-soft-primary' },
                { extend: 'csv', className: 'btn btn-sm btn-soft-primary' },
                { extend: 'print', className: 'btn btn-sm btn-soft-primary' }
            ],
            drawCallback: function() {
                // Initialize Bootstrap popovers after table draw
                $('.message-cell-popover').each(function() {
                    const element = this;
                    const popover = bootstrap.Popover.getInstance(element);
                    if (popover) {
                        popover.dispose();
                    }
                    
                    // Get message from data attribute (use attr instead of data to avoid jQuery parsing issues)
                    const message = $(element).attr('data-message') || '';
                    
                    // Initialize popover with the message content
                    new bootstrap.Popover(element, {
                        content: message,
                        title: 'Full Message',
                        trigger: 'click',
                        placement: 'top',
                        html: false
                    });
                });
            }
        });
    }

    function initFilters() {
        const filterInputs = ['filter_id', 'filter_message', 'filter_receiver', 'filter_using'];
        
        filterInputs.forEach(function(id) {
            const element = document.getElementById(id);
            if (element.tagName === 'SELECT') {
                element.addEventListener('change', function() {
                    smsLogTable.ajax.reload();
                });
            } else {
                element.addEventListener('input', function() {
                    clearTimeout(filterTimeout);
                    filterTimeout = setTimeout(function() {
                        smsLogTable.ajax.reload();
                    }, 300);
                });
            }
        });
    }

    function initDatePicker() {
        flatpickr('#filter_date', {
            dateFormat: 'Y-m-d',
            allowInput: true,
            onChange: function() {
                smsLogTable.ajax.reload();
            }
        });
    }

    function resendSms(id) {
        Swal.fire({
            title: 'Resend SMS?',
            text: 'Are you sure you want to resend this SMS?',
            icon: 'question',
            showCancelButton: true,
            confirmButtonText: 'Yes, Resend',
            cancelButtonText: 'Cancel',
            customClass: {
                confirmButton: 'btn btn-success me-2',
                cancelButton: 'btn btn-primary'
            }
        }).then((result) => {
            if (result.isConfirmed) {
                $.ajax({
                    url: '{{ route("sms.resend") }}',
                    type: 'POST',
                    data: {
                        _token: '{{ csrf_token() }}',
                        id: id
                    },
                    success: function(response) {
                        Swal.fire({
                            icon: response.success ? 'success' : 'error',
                            title: response.success ? 'SMS Resent' : 'Failed',
                            text: response.message,
                            customClass: {
                                confirmButton: 'btn btn-success'
                            }   
                        });
                        smsLogTable.ajax.reload();
                    },
                    error: function(xhr) {
                        Swal.fire({
                            icon: 'error',
                            title: 'Error',
                            customClass: {
                                confirmButton: 'btn btn-error'
                            },
                            text: xhr.responseJSON?.message || 'Failed to resend SMS'
                        });
                    }
                });
            }
        });
    }
</script>
@endsection

