@extends('layouts.horizontal', ['title' => 'Tax Settings', 'topbarTitle' => 'Tax Settings'])

@section('css')
@vite(['node_modules/datatables.net-bs5/css/dataTables.bootstrap5.min.css',
'node_modules/datatables.net-responsive-bs5/css/responsive.bootstrap5.min.css',
'node_modules/datatables.net-fixedheader-bs5/css/fixedHeader.bootstrap5.min.css',
'node_modules/datatables.net-buttons-bs5/css/buttons.bootstrap5.min.css',
'node_modules/datatables.net-select-bs5/css/select.bootstrap5.min.css'])
<style>
    .filter-row { background-color: #f9f9f9; }
    .form-filter { font-size: 12px; padding: 4px 6px; height: 30px; width: 100%; }
    .filter-row th { padding: 4px 6px !important; vertical-align: middle; }
    .dataTables_wrapper .dataTables_filter { float: right; }
    .dataTables_wrapper .dataTables_length { float: left; }
    .dataTables_wrapper .dataTables_paginate { float: right; }
    .filter-buttons { display: flex; gap: 5px; }
    .filter-buttons button { padding: 4px 8px; font-size: 12px; }
    .action-buttons { white-space: nowrap; }
    /* Prevent body padding adjustment on modal open/close */
    body.modal-open {
        padding-right: 0 !important;
    }
</style>
@endsection

@section('content')
<div class="container-fluid">
    <div class="row mb-4">
        <div class="col-12">
            <div class="page-title-box d-flex align-items-center justify-content-between">
                <h2 class="mb-0">Tax Settings</h2>
                <div class="page-title-right">
                    @can('create_tax_settings')
                    <button class="btn btn-sm btn-outline-success" id="addTaxBtn" onclick="openAddTaxModal()">
                        <iconify-icon icon="mdi:plus" class="me-1 fs-22"></iconify-icon> Add Tax Setting
                    </button>
                    @endcan
                </div>
            </div>
        </div>
    </div>
    <!-- end page title -->

    <!-- Success/Error Messages -->
    <div id="alert-container"></div>


    <div class="row">
        <div class="col-12">
            <div class="card">
                <div class="card-body">
                    <table id="tax_settings_table" class="table table-striped table-bordered w-100 fs-13">
                        <thead>
                            <tr>
                                <th>ID</th>
                                <th>Business</th>
                                <th>Title</th>
                                <th>Percentage</th>
                                <th>Status</th>
                                <th>Apply On</th>
                                <th>When Payment Mode Is</th>
                                <th>Actions</th>
                            </tr>
                            <!-- Filter Row -->
                            <tr class="filter filter-row">
                                <td>
                                    <input type="text" class="form-control form-filter input-sm" name="id_business_taxes" placeholder="Search ID">
                                </td>
                                <td>
                                    <input type="text" class="form-control form-filter input-sm" name="business_name" placeholder="Search Business">
                                </td>
                                <td>
                                    <input type="text" class="form-control form-filter input-sm" name="tax_name" placeholder="Search Title">
                                </td>
                                <td></td>
                                <td>
                                    <select class="form-control form-filter input-sm" name="tax_active">
                                        <option value="">All</option>
                                        <option value="Y">Active</option>
                                        <option value="N">Inactive</option>
                                    </select>
                                </td>
                                <td>
                                    <select class="form-control form-filter input-sm" name="tax_invoice_type">
                                        <option value="">All</option>
                                        <option value="service">Service</option>
                                        <option value="retail">Retail</option>
                                    </select>
                                </td>
                                <td>
                                    <select class="form-control form-filter input-sm" name="tax_payment_mode">
                                        <option value="">All</option>
                                        <option value="All">All</option>
                                        <option value="cash">Cash</option>
                                        <option value="card">Card</option>
                                        <option value="check">Check</option>
                                        <option value="online">Online</option>
                                        <option value="voucher">Voucher</option>
                                        <option value="mixed">Mixed</option>
                                    </select>
                                </td>
                                <td>
                                    <div class="filter-buttons">
                                        <button class="btn btn-sm btn-ghost-secondary rounded-circle btn-icon filter-submit">
                                            <iconify-icon icon="mdi:magnify" class="fs-22"></iconify-icon>
                                        </button>
                                        <button class="btn btn-sm btn-ghost-secondary rounded-circle btn-icon filter-cancel">
                                            <iconify-icon icon="mdi:refresh" class="fs-22"></iconify-icon>
                                        </button>
                                    </div>
                                </td>
                            </tr>
                        </thead>
                        <tbody></tbody>
                    </table>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Add/Edit Tax Modal -->
<div class="modal fade" id="taxModal" tabindex="-1" aria-labelledby="taxModalLabel" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="taxModalLabel">Add Tax Setting</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
                <form id="taxForm">
                    @csrf
                    <input type="hidden" name="id_business_taxes" id="id_business_taxes">
                    
                    @if(session('ho') === 'Yes')
                        <div class="row">
                            <div class="col-12 mb-3">
                                <label for="modal_business_id" class="form-label">Business <span class="text-danger">*</span></label>
                                <select class="form-control" id="modal_business_id" name="business_id" required>
                                    <option value="">Select Business</option>
                                    @foreach($businesses as $biz)
                                        <option value="{{ $biz->id_business }}"
                                            {{ (session('selected_business_id') ?? session('business_id')) == $biz->id_business ? 'selected' : '' }}>
                                            {{ $biz->business_name }}
                                        </option>
                                    @endforeach
                                </select>
                            </div>
                        </div>
                    @else
                        <input type="hidden" name="business_id" id="modal_business_id" value="{{ session('business_id') }}">
                    @endif
                    
                    <div class="row">
                        <div class="col-12 mb-3">
                            <label for="tax_name" class="form-label">Title <span class="text-danger">*</span></label>
                            <input type="text" class="form-control" id="tax_name" name="tax_name" required>
                            <div class="invalid-feedback" id="tax_name_error"></div>
                        </div>
                        
                        <div class="col-12 mb-3">
                            <label for="tax_percentage" class="form-label">Percentage <span class="text-danger">*</span></label>
                            <input type="number" class="form-control" id="tax_percentage" name="tax_percentage" 
                                   step="0.01" min="0" max="100" required>
                            <small class="text-muted">Enter percentage value (e.g., 15 for 15%)</small>
                            <div class="invalid-feedback" id="tax_percentage_error"></div>
                        </div>
                        
                        <div class="col-12 mb-3">
                            <label for="tax_active" class="form-label">Status <span class="text-danger">*</span></label>
                            <select class="form-control" id="tax_active" name="tax_active" required>
                                <option value="Y">Active</option>
                                <option value="N">Inactive</option>
                            </select>
                            <div class="invalid-feedback" id="tax_active_error"></div>
                        </div>
                        
                        <div class="col-12 mb-3">
                            <label for="tax_invoice_type" class="form-label">Apply On <span class="text-danger">*</span></label>
                            <select class="form-control" id="tax_invoice_type" name="tax_invoice_type" required>
                                <option value="service">Service</option>
                                <option value="retail">Retail</option>
                            </select>
                            <div class="invalid-feedback" id="tax_invoice_type_error"></div>
                        </div>
                        
                        <div class="col-12 mb-3">
                            <label for="tax_payment_mode" class="form-label">When Payment Mode Is <span class="text-danger">*</span></label>
                            <select class="form-control" id="tax_payment_mode" name="tax_payment_mode" required>
                                <option value="All">All</option>
                                <option value="cash">Cash</option>
                                <option value="card">Card</option>
                                <option value="check">Check</option>
                                <option value="online">Online</option>
                                <option value="voucher">Voucher</option>
                                <option value="mixed">Mixed</option>
                            </select>
                            <div class="invalid-feedback" id="tax_payment_mode_error"></div>
                        </div>
                    </div>
                </form>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                <button type="button" class="btn btn-primary" id="saveTaxBtn">Save Tax</button>
            </div>
        </div>
    </div>
</div>

@endsection

@section('scripts')
@vite(['resources/js/app.js'])
<script>
    document.addEventListener('DOMContentLoaded', function() {
        initializeTaxTable();
        setupEventHandlers();
    });
    
    let taxTable = null;
    let taxModal = null;
    
    function initializeTaxTable() {
        if (typeof $.fn.DataTable === 'undefined') {
            console.error('DataTables is not loaded');
            return;
        }
        
        taxTable = $('#tax_settings_table').DataTable({
            scrollX: true, 
            orderCellsTop: true,           
            scrollCollapse: true,
            autoWidth: false,
            fixedHeader: {
                header: true,
                footer: false
            },
            layout: {
                topStart: 'buttons',
                topEnd: 'search',
                bottomStart: {
                    pageLength: {
                        menu: [10, 20, 50, 100]
                    },
                    info: {}
                },
                bottomEnd: 'paging'
            },
            processing: true,
            serverSide: true,
            stateSave: true,
            stateDuration: -1,
            responsive: false,
            ajax: {
                url: "{{ route('tax-settings.data') }}",
                type: 'POST',
                contentType: 'application/json',
                headers: {
                    'X-CSRF-TOKEN': '{{ csrf_token() }}'
                },
                dataType: 'json',
                data: function (d) {
                    d.id_business_taxes = $('input[name="id_business_taxes"]').val();
                    d.business_name = $('input[name="business_name"]').val();
                    d.tax_name = $('input[name="tax_name"]').val();
                    d.tax_active = $('select[name="tax_active"]').val();
                    d.tax_invoice_type = $('select[name="tax_invoice_type"]').val();
                    d.tax_payment_mode = $('select[name="tax_payment_mode"]').val();
                    
                    // Add business_id filter if HO user
                    @if(session('ho') === 'Yes')
                    const businessFilter = $('#business_filter').val();
                    if (businessFilter) {
                        d.business_id = businessFilter;
                    }
                    @endif
                    
                    return JSON.stringify(d);
                },
                error: function (xhr, error, thrown) {
                    console.error('DataTables AJAX error:', error, thrown, xhr.responseText);
                    showAlert('error', 'Failed to load data. Please try again.');
                }
            },
            "language": {
                "paginate": {
                    "previous": "<i class='ti ti-chevron-left'></i>",
                    "next": "<i class='ti ti-chevron-right'></i>"
                }
            },
            lengthChange: true,
            pageLength: 10,
            searching: false,
            info: true,
            paging: true,
            buttons: [
                { extend: 'copy', className: 'btn btn-sm btn-soft-primary' },
                { extend: 'csv', className: 'btn btn-sm btn-soft-primary' },
                { extend: 'print', className: 'btn btn-sm btn-soft-primary' }
            ],
            columns: [
                { data: 'id_business_taxes', name: 'id_business_taxes', orderable: true, searchable: true },
                { data: 'business_name', name: 'business_name', orderable: true, searchable: true },
                { data: 'tax_name', name: 'tax_name', orderable: true, searchable: true },
                { data: 'percentage', name: 'tax_percentage', orderable: true, searchable: false },
                { data: 'status', name: 'tax_active', orderable: true, searchable: false },
                { data: 'apply_on', name: 'tax_invoice_type', orderable: true, searchable: false },
                { data: 'payment_mode', name: 'tax_payment_mode', orderable: true, searchable: false },
                { 
                    data: 'action', 
                    name: 'actions', 
                    orderable: false, 
                    searchable: false,
                    className: 'action-buttons'
                }
            ],
            order: [[0, 'desc']]
        });
        
        // Initialize modal
        taxModal = new bootstrap.Modal(document.getElementById('taxModal'));
        
        // Fix horizontal shrink issue when modal closes
        const taxModalElement = document.getElementById('taxModal');
        if (taxModalElement) {
            taxModalElement.addEventListener('hidden.bs.modal', function() {
                // Force DataTables to recalculate width
                if (taxTable) {
                    taxTable.columns.adjust().draw(false);
                }
                // Remove any body padding that might have been added
                document.body.style.paddingRight = '';
                // Re-enable save button when modal closes
                const saveBtn = document.getElementById('saveTaxBtn');
                if (saveBtn) {
                    saveBtn.disabled = false;
                }
            });
            
            
            taxModalElement.addEventListener('show.bs.modal', function() {
                // Store original padding
                const originalPadding = window.getComputedStyle(document.body).paddingRight;
                taxModalElement.dataset.originalPadding = originalPadding;
            });
        }
    }

    function setupEventHandlers() {
        // Add Tax Button
        document.getElementById('addTaxBtn')?.addEventListener('click', function() {
            openTaxModal();
        });
        
        // Save Tax Button
        document.getElementById('saveTaxBtn')?.addEventListener('click', function() {
            saveTax();
        });
        
        // Filter Submit
        document.querySelector('.filter-submit')?.addEventListener('click', function(e) {
            e.preventDefault();
            refreshData();
        });
        
        // Filter Cancel
        document.querySelector('.filter-cancel')?.addEventListener('click', function(e) {
            e.preventDefault();
            document.querySelectorAll('.form-filter').forEach(input => {
                input.value = '';
            });
            refreshData();
        });
        
        // Enter key in filters
        document.addEventListener('keyup', function(e) {
            if (e.target.classList.contains('form-filter') && e.key === 'Enter') {
                refreshData();
            }
        });
    }

    function refreshData() {
        if (taxTable) {
            taxTable.ajax.reload(null, false);
        }
    }

    function openTaxModal(id = null) {
        resetModalForm();
        clearValidationErrors();
        
        // Disable save button when modal opens
        const saveBtn = document.getElementById('saveTaxBtn');
        if (saveBtn) {
            saveBtn.disabled = false;
        }
        
        const modalTitle = document.getElementById('taxModalLabel');
        if (modalTitle) {
            modalTitle.textContent = id ? 'Edit Tax Setting' : 'Add Tax Setting';
        }
        
        if (id) {
            loadTaxData(id);
        } else {
            // Set default business for HO users
            @if(session('ho') === 'Yes')
            const businessFilter = $('#business_filter').val();
            if (businessFilter) {
                $('#modal_business_id').val(businessFilter).trigger('change');
            }
            @endif
            
            taxModal.show();
        }
    }

    function resetModalForm() {
        const form = document.getElementById('taxForm');
        if (form) {
            form.reset();
        }
        
        document.getElementById('id_business_taxes').value = '';
        document.getElementById('tax_active').value = 'Y';
        document.getElementById('tax_invoice_type').value = 'service';
        document.getElementById('tax_payment_mode').value = 'All';
    }

    function loadTaxData(id) {
        Swal.fire({
            title: 'Loading...',
            allowOutsideClick: false,
            didOpen: () => Swal.showLoading()
        });
        
        fetch("{{ url('tax-settings') }}/" + id, {
            method: 'GET',
            headers: {
                'Accept': 'application/json',
            }
        })
        .then(response => response.json())
        .then(response => {
            Swal.close();
            
            if (response.success && response.tax) {
                const tax = response.tax;
                
                document.getElementById('id_business_taxes').value = tax.id_business_taxes;
                document.getElementById('tax_name').value = tax.tax_name || '';
                document.getElementById('tax_percentage').value = tax.tax_percentage || '';
                
                document.getElementById('tax_active').value = tax.tax_active || 'Y';
                document.getElementById('tax_invoice_type').value = tax.tax_invoice_type || 'service';
                document.getElementById('tax_payment_mode').value = tax.tax_payment_mode || 'All';
                
                clearValidationErrors();
                taxModal.show();
            } else {
                Swal.fire({
                    icon: 'error',
                    title: 'Error',
                    text: response.message || 'Failed to load tax data.',
                });
            }
        })
        .catch(error => {
            Swal.close();
            console.error('Error:', error);
            Swal.fire({
                icon: 'error',
                title: 'Error',
                text: 'Error loading tax data. Please try again.',
            });
        });
    }

    function saveTax() {
        const saveBtn = document.getElementById('saveTaxBtn');
        
        // Disable save button immediately to prevent multiple submissions
        if (saveBtn) {
            saveBtn.disabled = true;
        }
        
        const form = document.getElementById('taxForm');
        if (!form.checkValidity()) {
            form.classList.add('was-validated');
            // Re-enable button if validation fails
            if (saveBtn) {
                saveBtn.disabled = false;
            }
            return;
        }
        
        const formData = new FormData(form);
        
        Swal.fire({
            title: 'Saving...',
            allowOutsideClick: false,
            didOpen: () => Swal.showLoading()
        });
        
        fetch("{{ route('tax-settings.save') }}", {
            method: 'POST',
            headers: {
                'X-CSRF-TOKEN': '{{ csrf_token() }}',
                'Accept': 'application/json',
            },
            body: formData
        })
        .then(response => response.json())
        .then(response => {
            Swal.close();
            
            if (response.success) {
                Swal.fire({
                    icon: 'success',
                    title: 'Success',
                    text: response.message || 'Tax setting saved successfully.',
                    timer: 2000,
                    showConfirmButton: false
                }).then(() => {
                    taxModal.hide();
                    refreshData();
                });
            } else {
                let errorMsg = response.message || 'Error saving tax setting.';
                if (response.errors) {
                    displayValidationErrors(response.errors);
                    errorMsg = 'Please fix the validation errors.';
                }
                Swal.fire({
                    icon: 'error',
                    title: 'Error',
                    text: errorMsg,
                });
            }
        })
        .catch(error => {
            Swal.close();
            console.error('Error:', error);
            Swal.fire({
                icon: 'error',
                title: 'Error',
                text: 'Error saving tax setting. Please try again.',
            });
        });
    }

    function deleteTax(id) {
        Swal.fire({
            title: 'Are you sure?',
            text: "You won't be able to revert this!",
            icon: 'warning',
            showCancelButton: true,
            confirmButtonColor: '#d33',
            cancelButtonColor: '#3085d6',
            confirmButtonText: 'Yes, delete it!',
            customClass: {
                confirmButton: 'btn btn-danger me-2',
                cancelButton: 'btn btn-warning'
            }
        }).then((result) => {
            if (result.isConfirmed) {
                Swal.fire({
                    title: 'Deleting...',
                    allowOutsideClick: false,
                    didOpen: () => Swal.showLoading()
                });
                
                fetch("{{ url('tax-settings') }}/" + id + "/delete", {
                    method: 'POST',
                    headers: {
                        'X-CSRF-TOKEN': '{{ csrf_token() }}',
                        'Accept': 'application/json',
                    }
                })
                .then(response => response.json())
                .then(response => {
                    Swal.close();
                    
                    if (response.success) {
                        Swal.fire({
                            icon: 'success',
                            title: 'Deleted!',
                            text: response.message || 'Tax setting deleted successfully.',
                            timer: 2000,
                            showConfirmButton: false
                        }).then(() => {
                            refreshData();
                        });
                    } else {
                        Swal.fire({
                            icon: 'error',
                            title: 'Error',
                            text: response.message || 'Error deleting tax setting.',
                        });
                    }
                })
                .catch(error => {
                    Swal.close();
                    console.error('Error:', error);
                    Swal.fire({
                        icon: 'error',
                        title: 'Error',
                        text: 'Error deleting tax setting. Please try again.',
                    });
                });
            }
        });
    }

    function openEditModal(id) {
        openTaxModal(id);
    }

    function clearValidationErrors() {
        document.querySelectorAll('.invalid-feedback').forEach(el => {
            el.textContent = '';
        });
        document.getElementById('taxForm')?.classList.remove('was-validated');
    }

    function displayValidationErrors(errors) {
        clearValidationErrors();
        Object.keys(errors).forEach(field => {
            const errorElement = document.getElementById(field + '_error');
            if (errorElement) {
                errorElement.textContent = errors[field][0];
            }
        });
    }

    function showAlert(type, message) {
        const alertContainer = document.getElementById('alert-container');
        const alertClass = type === 'error' ? 'alert-danger' : 'alert-success';
        alertContainer.innerHTML = `
            <div class="alert ${alertClass} alert-dismissible fade show" role="alert">
                ${message}
                <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
            </div>
        `;
        setTimeout(() => {
            alertContainer.innerHTML = '';
        }, 5000);
    }
</script>
@endsection
