@extends('layouts.horizontal', ['title' => 'Support Tickets', 'topbarTitle' => 'Support Tickets'])

@section('css')
@endsection

@section('content')
<div class="container-fluid">
    <div class="row page-titles mb-3">
        <div class="col-md-5 align-self-center">
            <h2 class="text-themecolor">Support Tickets</h2>
        </div>
    </div>

    <!-- Container 1: Ticket Submission Form -->
    <div class="row">
        <div class="col-12">
            <div class="card">
                <div class="card-header">
                    <h4 class="card-title mb-0">Submit New Ticket</h4>
                </div>
                <div class="card-body">
                    <form id="ticketForm">
                        <div class="row">
                            <div class="col-md-12">
                                <div class="mb-3">
                                    <label for="ticketTitle" class="form-label">Title <span class="text-danger">*</span></label>
                                    <input type="text" class="form-control" id="ticketTitle" 
                                           placeholder="Enter ticket title" maxlength="50" required>
                                    <div id="titleError" class="text-danger small mt-1" style="display: none;"></div>
                                </div>
                            </div>
                        </div>
                        <div class="row">
                            <div class="col-md-12">
                                <div class="mb-3">
                                    <label for="ticketDescription" class="form-label">Description <span class="text-danger">*</span></label>
                                    <textarea class="form-control" id="ticketDescription" rows="4" 
                                              placeholder="Enter ticket description" maxlength="1000" required></textarea>
                                    <div class="mt-1">
                                        <small id="charCounter" class="text-muted">Character count: 0 of 1000</small>
                                    </div>
                                    <div id="descriptionError" class="text-danger small mt-1" style="display: none;"></div>
                                </div>
                            </div>
                        </div>
                        <div class="row">
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label for="ticketSeverity" class="form-label">Severity <span class="text-danger">*</span></label>
                                    <select class="form-select" id="ticketSeverity" required>
                                        <option value="">-- Select Severity --</option>
                                        <option value="Critical">Critical</option>
                                        <option value="High">High</option>
                                        <option value="Medium">Medium</option>
                                        <option value="Low">Low</option>
                                    </select>
                                    <div id="severityError" class="text-danger small mt-1" style="display: none;"></div>
                                </div>
                            </div>
                        </div>
                        <div class="row">
                            <div class="col-12">
                                <button type="button" id="submitBtn" class="btn btn-primary me-2" onclick="submitTicket()">
                                    <i class="ti ti-send me-1"></i> Submit Ticket
                                </button>
                                <button type="button" class="btn btn-secondary" onclick="clearForm()">
                                    <i class="ti ti-eraser me-1"></i> Clear
                                </button>
                            </div>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    </div>

    <!-- Container 2: Tickets Table -->
    <div class="row">
        <div class="col-12">
            <div class="card">
                <div class="card-header">
                    <h4 class="card-title mb-0">My Tickets</h4>
                </div>
                <div class="card-body">
                    <div class="table-responsive">
                        <table id="ticketsTable" class="table table-striped table-bordered w-100 fs-13">
                            <thead>
                                <tr>
                                    <th>ID</th>
                                    <th>Title</th>
                                    <th>Severity</th>
                                    <th>Status</th>
                                    <th>Created At</th>
                                </tr>
                            </thead>
                            <tbody>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>
@endsection

@section('scripts')
<script>
    let isSubmitting = false;

    document.addEventListener('DOMContentLoaded', function() {
        initTicketsTable();
        initCharacterCounter();
    });

    function initTicketsTable() {
        if ($.fn.DataTable.isDataTable('#ticketsTable')) {
            $('#ticketsTable').DataTable().clear().destroy();
        }

        $('#ticketsTable').DataTable({
            processing: true,
            serverSide: true,
            ajax: {
                url: '{{ route("tickets.data") }}',
                type: 'GET'
            },
            columns: [
                { data: 'id_tickets', name: 'id_tickets' },
                { data: 'title', name: 'title' },
                { data: 'severity', name: 'severity' },
                { data: 'status', name: 'status' },
                { data: 'created_at', name: 'created_at' }
            ],
            pageLength: 10,
            order: [[0, 'desc']],
            layout: {
                topStart: 'buttons',
                topEnd: 'search',
                bottomStart: {
                    pageLength: {
                        menu: [10, 20, 50, 100]
                    },
                    info: {}
                },
                bottomEnd: 'paging'
            },
            buttons: [
                { extend: 'copy', className: 'btn btn-sm btn-soft-primary' },
                { extend: 'csv', className: 'btn btn-sm btn-soft-primary' },
                { extend: 'print', className: 'btn btn-sm btn-soft-primary' }
            ],
            language: {
                search: 'Search:',
                searchPlaceholder: 'Title, ID, Severity, or Status'
            }
        });
    }

    function initCharacterCounter() {
        const descriptionInput = document.getElementById('ticketDescription');
        const counter = document.getElementById('charCounter');

        descriptionInput.addEventListener('input', function() {
            const length = this.value.length;
            counter.textContent = `Character count: ${length} of 1000`;
            
            if (length > 1000) {
                counter.classList.remove('text-muted');
                counter.classList.add('text-danger');
            } else {
                counter.classList.remove('text-danger');
                counter.classList.add('text-muted');
            }
        });
    }

    function validateForm() {
        const title = document.getElementById('ticketTitle').value.trim();
        const description = document.getElementById('ticketDescription').value.trim();
        const severity = document.getElementById('ticketSeverity').value;
        let isValid = true;

        // Clear previous errors
        document.getElementById('titleError').style.display = 'none';
        document.getElementById('descriptionError').style.display = 'none';
        document.getElementById('severityError').style.display = 'none';

        if (!title) {
            document.getElementById('titleError').textContent = 'Title is required';
            document.getElementById('titleError').style.display = 'block';
            isValid = false;
        } else if (title.length > 50) {
            document.getElementById('titleError').textContent = 'Title must not exceed 50 characters';
            document.getElementById('titleError').style.display = 'block';
            isValid = false;
        }

        if (!description) {
            document.getElementById('descriptionError').textContent = 'Description is required';
            document.getElementById('descriptionError').style.display = 'block';
            isValid = false;
        } else if (description.length > 1000) {
            document.getElementById('descriptionError').textContent = 'Description must not exceed 1000 characters';
            document.getElementById('descriptionError').style.display = 'block';
            isValid = false;
        }

        if (!severity) {
            document.getElementById('severityError').textContent = 'Severity is required';
            document.getElementById('severityError').style.display = 'block';
            isValid = false;
        }

        return isValid;
    }

    function submitTicket() {
        if (isSubmitting) {
            return;
        }

        if (!validateForm()) {
            return;
        }

        isSubmitting = true;
        const submitBtn = document.getElementById('submitBtn');
        submitBtn.disabled = true;
        submitBtn.innerHTML = '<i class="ti ti-loader me-1"></i> Submitting...';

        $.ajax({
            url: '{{ route("tickets.store") }}',
            type: 'POST',
            data: {
                _token: '{{ csrf_token() }}',
                title: document.getElementById('ticketTitle').value.trim(),
                description: document.getElementById('ticketDescription').value.trim(),
                severity: document.getElementById('ticketSeverity').value
            },
            success: function(response) {
                if (response.success) {
                    Swal.fire({
                        icon: 'success',
                        title: 'Ticket Submitted',
                        text: response.message || 'Your ticket has been submitted successfully',
                        timer: 2000,
                        showConfirmButton: false
                    });
                    
                    // Clear form
                    clearForm();
                    
                    // Reload tickets table
                    if ($.fn.DataTable.isDataTable('#ticketsTable')) {
                        $('#ticketsTable').DataTable().ajax.reload();
                    }
                } else {
                    Swal.fire({
                        icon: 'error',
                        title: 'Submission Failed',
                        text: response.message || 'Failed to submit ticket'
                    });
                }
            },
            error: function(xhr) {
                let errorMessage = 'Failed to submit ticket. Please try again.';
                if (xhr.responseJSON && xhr.responseJSON.message) {
                    errorMessage = xhr.responseJSON.message;
                } else if (xhr.responseJSON && xhr.responseJSON.errors) {
                    const errors = Object.values(xhr.responseJSON.errors).flat();
                    errorMessage = errors.join('\n');
                }
                
                Swal.fire({
                    icon: 'error',
                    title: 'Error',
                    text: errorMessage
                });
            },
            complete: function() {
                isSubmitting = false;
                submitBtn.disabled = false;
                submitBtn.innerHTML = '<i class="ti ti-send me-1"></i> Submit Ticket';
            }
        });
    }

    function clearForm() {
        document.getElementById('ticketTitle').value = '';
        document.getElementById('ticketDescription').value = '';
        document.getElementById('ticketSeverity').value = '';
        document.getElementById('charCounter').textContent = 'Character count: 0 of 1000';
        document.getElementById('charCounter').classList.remove('text-danger');
        document.getElementById('charCounter').classList.add('text-muted');
        
        // Clear errors
        document.getElementById('titleError').style.display = 'none';
        document.getElementById('descriptionError').style.display = 'none';
        document.getElementById('severityError').style.display = 'none';
    }
</script>
@endsection

