@extends('layouts.horizontal', ['title' => 'Bookings', 'topbarTitle' => 'Customer Visits', 'nonav' => true])

@section('css')

@vite(['node_modules/select2/dist/css/select2.min.css'])
@vite(['node_modules/flatpickr/dist/flatpickr.min.css'])

@endsection

@section('content')

<style>
     tr.strike-through > td {text-decoration:line-through !important; color:red;}      
     tr.border-top > td {border-top: 2px solid #dee2e6 !important;} 
</style>


<div class="BookingsPage">
    <!-- start page title -->
    <div class="row mb-4">
        <div class="col-12">
            <div class="page-title-box d-flex align-items-center justify-content-between">
                <h2 class="mb-0">Create Booking</h2>
                <div class="page-title-right d-flex align-items-center gap-3">
                    @php
                        $breadcrumb = getCurrentMenuBreadcrumb();
                    @endphp
                    @if(!empty($breadcrumb))
                        <nav aria-label="breadcrumb">
                            <ol class="breadcrumb mb-0 py-2">
                                @foreach($breadcrumb as $index => $item)
                                    @if($index === count($breadcrumb) - 1)
                                        <li class="breadcrumb-item active" aria-current="page">{{ $item }}</li>
                                    @else
                                        <li class="breadcrumb-item"><a href="javascript: void(0);">{{ $item }}</a></li>
                                    @endif
                                @endforeach
                            </ol>
                        </nav>
                    @endif
                </div>
            </div>
        </div>
    </div>
    <!-- end page title -->
    <div class="row">
        <div class="col-12">
            <div class="card">
                <div class="card-body">
                    <div class="row">
                        <div class="col-md-6">
                            <label for="selectCustomer" class="form-label">Select Customer</label>
                            <select id="selectCustomer" class="form-select select2">                                
                            </select>
                        </div>
                        <div class="col-md-6">
                            <form id="branchForm">
                                @csrf
                                <label for="branches" class="form-label">Select Branch</label>
                                <select id="branches" name="business_id" class="form-select" onchange="updateFlexiPricingStatus()">
                                    @foreach($branches as $branch)
                                        <option {{ $business_id == $branch->id_business ? 'selected' : '' }} value="{{ $branch->id_business }}" data-flexi-pricing="{{ $branch->flexi_pricing_enabled ?? 'No' }}">{{ $branch->business_name }}</option>
                                    @endforeach
                                </select>
                            </form>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <div class="row" id="bookingDetailsSection" style="display:none;">
        <div class="col-12">
            <div class="card">
                <div class="card-body">
                    <div class="row">
                        <div class="col-md-12">                            
                            <!-- Booking details form will go here -->
                            <div class="alert alert-info" id="customerAlert" style="display:none;">
                                <!-- Customer alert message will be populated here -->
                            </div>
                        </div>
                        <div class="col-md-12 g-3" id="customerRecoveryBtn">
                            <!-- Recovery Invoice Buttons will go here -->
                        </div>
                        <div class="row mt-3">
                            <input type="hidden" id="customerId">
                            <div class="col-md-3">
                                <label for="customerName" class="form-label">Customer Name</label>
                                <input type="text" id="customerName" class="form-control" readonly>
                            </div>
                            <div class="col-md-3">
                                <label for="customerCell" class="form-label">Customer Cell</label>
                                <input type="text" id="customerCell" class="form-control" readonly>
                            </div>
                            <div class="col-md-3">
                                <label for="customerGender" class="form-label">Customer Gender</label>
                                <input type="text" id="customerGender" class="form-control" readonly>
                            </div>
                            <div class="col-md-3">
                                <label for="customerType" class="form-label">Customer Type</label>
                                <input type="text" id="customerType" class="form-control" readonly>
                            </div>
                        </div>
                    </div>
                    <div class="row mt-4">
                        <div class="col-md-4">
                            <label class="form-label">Select Package</label>
                            <select onchange="handlePackageChange()" id="packageSelect" class="form-select select2">
                                <option value="">-- Select Package --</option>
                                @foreach($packages as $package)
                                    <option value="{{ $package->id_package_type }}">{{ $package->service_type }} - {{ config('constants.CURRENCY') }}{{ number_format($package->package_price, 2) }}</option>
                                @endforeach
                            </select>
                        </div>
                        <div class="col-md-4">
                            <label class="form-label">Select Event Date & Time</label>
                            <input onchange="handleDateTimeChange()"  type="text" id="bookingDateTime" class="form-control" placeholder="Select Date & Time">
                        </div>
                        <div class="col-md-4">
                            <label class="form-label">Referring Staff</label>
                            <select id="referringStaffSelect" class="form-select select2">
                                <option value="">-- Select Staff --</option>
                                @foreach($staff as $member)
                                    <option value="{{ $member->id_staff }}" data-image="{{ $member->staff_image }}">{{ $member->staff_fullname }}</option>                                    
                                @endforeach
                            </select>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
    <div class="row" id="bookingCategoriesSection" style="display:none;">
        <div class="col-12">
            <div class="card">
                <div class="card-body">
                    <div class="row">
                        <div class="col-md-12">                                                        
                            <table class="table table-bordered">
                                <thead>
                                    <tr>
                                        <th style="width: 10%;">Category</th>
                                        <th style="width: 25%;">Services</th>
                                        <th style="width: 10%;" class="text-end">Rate</th>
                                        <th style="width: 10%;">Date</th>
                                        <th style="width: 10%;">Time</th>
                                        <th style="width: 25%;">Staff</th>                                        
                                        <th style="width: 30%;" class="text-end">Total</th>
                                    </tr>
                                </thead>
                                <tbody id="bookingCategoriesTableBody">
                                    <!-- Booking categories and services will be populated here -->
                                </tbody>
                            </table>
                        </div>                        
                    </div>
                    
                    <div class="row mb-4">
                        <div class="col-12 text-end">
                            <button id="createBookingButton" class="btn btn-primary">
                                <span class="spinner-border spinner-border-sm me-2 d-none" role="status" aria-hidden="true" id="createBookingSpinner"></span>
                                Create Booking</button>
                        </div>
                    </div>                    
                </div>
            </div>
        </div>
    </div>

    <!-- Flexi-Price Offers Modal -->
    <div class="modal fade" id="flexiPriceModal" tabindex="-1" aria-labelledby="flexiPriceModalLabel" aria-hidden="true">
        <div class="modal-dialog modal-xl">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title" id="flexiPriceModalLabel">% Discount Suggestions</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body">
                    <div id="flexi-price-content">
                        <div class="text-center">
                            <div class="spinner-border" role="status">
                                <span class="visually-hidden">Loading...</span>
                            </div>
                        </div>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                </div>
            </div>
        </div>
    </div>

    
</div>

@endsection

@section('scripts')
<script>
    const baseURL = '{{ url("") }}'; // Use relative paths
    const staffImageBase = "{{ asset('images/staff') }}/";
    
    // ============================================================================
    // FLEXI-PRICING MODULE: Status Management
    // ============================================================================
    // Global variable to store flexi-pricing status for the current business.
    // This is updated dynamically when the user changes branches.
    var flexiPricingEnabled = {{ $business->flexi_pricing_enabled == 'Yes' ? 'true' : 'false' }};
    
    /**
     * Check if flexi-pricing is enabled for a specific business
     * 
     * First checks the branch dropdown data attribute (fast, no API call).
     * Falls back to API call if not found in dropdown.
     * Updates the global flexiPricingEnabled variable.
     * 
     * @param {string|number} businessId - Business ID to check
     * @param {function} callback - Optional callback function(enabled)
     */
    function checkFlexiPricingStatus(businessId, callback) {
        if (!businessId) {
            if (callback) callback(false);
            return;
        }
        
        // First check if we have it in the branch dropdown (fastest method)
        var branchOption = document.querySelector(`#branches option[value="${businessId}"]`);
        if (branchOption) {
            var status = branchOption.getAttribute('data-flexi-pricing') === 'Yes';
            flexiPricingEnabled = status;
            if (callback) callback(status);
            return;
        }
        
        // Fallback: fetch from API if not in dropdown
        $.ajax({
            url: baseURL + '/bookings/check_flexi_pricing_status',
            method: 'POST',
            data: {
                business_id: businessId,
                _token: $('meta[name="csrf-token"]').attr('content')
            },
            success: function(response) {
                if (response.success) {
                    flexiPricingEnabled = response.flexi_pricing_enabled;
                    if (callback) callback(response.flexi_pricing_enabled);
                } else {
                    flexiPricingEnabled = false;
                    if (callback) callback(false);
                }
            },
            error: function() {
                flexiPricingEnabled = false;
                if (callback) callback(false);
            }
        });
    }
    
    /**
     * Update flexi-pricing status when branch selection changes
     * 
     * Called when user selects a different branch. Refreshes categories
     * if they're already displayed to show/hide flexi-pricing offers accordingly.
     */
    function updateFlexiPricingStatus() {
        var selectedBranchId = document.getElementById('branches').value;

        $("#branchForm").submit();
        return false;

        checkFlexiPricingStatus(selectedBranchId, function(enabled) {
            // Refresh categories if they're already displayed
            var bookingCategoriesSection = document.getElementById('bookingCategoriesSection');
            if (bookingCategoriesSection && bookingCategoriesSection.style.display === 'block') {
                var selectedPackageId = document.getElementById('packageSelect').value;
                if (selectedPackageId) {
                    handlePackageChange();
                }
            }
        });
    }
    
    //document ready
    document.addEventListener('DOMContentLoaded', function() {
        // Initialize flexi-pricing status on page load
        var initialBranchId = document.getElementById('branches').value;
        checkFlexiPricingStatus(initialBranchId);
        
        document.getElementById('createBookingButton').addEventListener('click', create_visits_from_booking);

        //initialize packageSelect select2
        $('#packageSelect').select2({
            dropdownParent: $('#packageSelect').parent(),
            placeholder: 'Select Package'
        });

        //initialize referringStaffSelect select2
        $('#referringStaffSelect').select2({
            dropdownParent: $('#referringStaffSelect').parent(),
            templateResult: formatStaff,
            templateSelection: formatStaff,
            placeholder: 'Select Staff'
        });

        //flatpickr for booking date and time
        flatpickr("#bookingDateTime", {
            enableTime: true,
            dateFormat: "Y-m-d H:i",
            minDate: "today"
        });

        // ============================================================================
        // FLEXI-PRICING MODULE: Discount Field Protection (Global)
        // ============================================================================
        // Prevent manual editing of discount field when flexi-pricing is enabled
        // These event listeners catch any attempt to edit the discount field
        if (flexiPricingEnabled) {
            $(document).on('focus', 'input[name="booking_discount"]', function() {
                if ($(this).attr('readonly')) {
                    $(this).blur();
                    toastr.info('Discount is auto-calculated from Flexi-Pricing offers. Please use the "Offers" button to apply discounts.');
                }
            });
            
            $(document).on('keydown keypress input', 'input[name="booking_discount"]', function(e) {
                if ($(this).attr('readonly')) {
                    e.preventDefault();
                    e.stopPropagation();
                    return false;
                }
            });
        }

        ////initialize select2 for Customer Search
        $('#selectCustomer').select2({
            dropdownParent: $('#selectCustomer').parent(),
            placeholder: 'Select Customer',
            minimumInputLength: 3,
            ajax: {
                url: '{{ route("customer_search") }}',
                dataType: 'json',
                delay: 250,
                data: function (params) {
                    return { term: params.term };
                },
                transport: function (params, success, failure) {
                    let $select = $('#selectCustomer');
                    $.ajax(params).then(function (data) {
                        success(data);

                        // Executed AFTER results loaded
                        setTimeout(() => {
                            let select2 = $select.data('select2');
                            let $first = select2.$results.find('.select2-results__option:first');
                            if ($first.length) {
                                $first.addClass('select2-results__option--highlighted');
                            }
                        }, 500);
                    }, failure);
                },
                processResults: function (data) {
                    return {
                        results: data.map(function(item){
                            return {
                                id: item.id,
                                text: item.customer_name || (item.customer_name + ' (' + item.customer_cell + ')'),
                                customer_name: item.customer_name,
                                customer_cell: item.customer_cell,
                                customer_type: item.customer_type // <- make sure server sends this
                            };
                        })
                    };
                },
                cache: true
            },
            templateResult: formatCustomer,      // dropdown items
            templateSelection: formatCustomerSel // selected item
        });

        function formatCustomer(customer) {
            if (!customer || !customer.id) {
                return customer && customer.text ? customer.text : '';
            }

            var type = (customer.customer_type || '').toString().toLowerCase();
            var iconClass = 'ti-user fs-4 text-secondary';
            var nameClass = '';

            if (type === 'orange' || type === '1') {
                iconClass = 'ti-star fs-4 text-warning';
                nameClass = 'text-warning';
            } else if (type === 'green' || type === '2') {
                iconClass = 'ti-flag fs-4 text-success';
                nameClass = 'text-success';
            } else if (type === 'red' || type === '3') {
                iconClass = 'ti-flag-question fs-4 text-danger';
            }

            var nameEsc = $('<div/>').text(customer.customer_name || customer.text || '').html();
            var cellEsc = $('<div/>').text(customer.customer_cell || '').html();

            var $node = $(`
                <div class="d-flex justify-content-between align-items-center">
                    <div>
                        <div class="fw-bold ${nameClass}">${nameEsc}</div>
                        <div class="small">${cellEsc}</div>
                    </div>
                    <i class="ti ${iconClass} ms-2" aria-hidden="true"></i>
                </div>
            `);

            //important: return plain DOM node, not jQuery object
            return $node[0];
        }

        function formatCustomerSel(customer) {
            if (!customer || !customer.id) {
                return customer && customer.text ? customer.text : '';
            }
            var name = customer.customer_name || customer.text;
            var cell = customer.customer_cell ? ' (' + customer.customer_cell + ')' : '';
            // return a simple node — keep selection compact
            return $('<span>').text(name + cell);
        }

        $("#selectCustomer").on("select2:select", function (e) {
            var selectedCustomerId = e.params.data.id;
            //get custome details
            
            console.log("Selected Customer: " + selectedCustomerId);
            if(selectedCustomerId != "" && selectedCustomerId != null){
                $.ajax({
                    url : "{{ route('get_customer_details') }}",
                    data: { "id_customer" : selectedCustomerId },
                    type: "GET", // The GET method
                    dataType: "json", // Expected data type of the response
                    headers: {'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content')},
                    success: function(data) {
                        var customer_details = data.customer_details;
                        var recoveries = data.recoveries;
                        
                        // console.log(data.customer_details.customer_name);
                        $("#customerId").val(data.customer_details.id_customers);
                        $("#customerName").val(data.customer_details.customer_name);
                        $("#customerCell").val(data.customer_details.customer_cell);   
                        if(customer_details.customer_gender == "M"){
                            customer_details.customer_gender = "Male";
                        } else if(customer_details.customer_gender == "F"){
                            customer_details.customer_gender = "Female";
                        } else {
                            customer_details.customer_gender = "Other";
                        }                     
                        $("#customerGender").val(customer_details.customer_gender);

                        if(customer_details.customer_flag == "red"){
                            customer_details.customer_flag = "Red Flagged Customer";
                        } else if(customer_details.customer_flag == "green"){
                            customer_details.customer_flag = "Important Customer";
                        } else if(customer_details.customer_flag == "orange"){
                            customer_details.customer_flag = "Star Customer";
                        } else {
                            customer_details.customer_flag = "Regular Customer";
                        }
                        $("#customerType").val(customer_details.customer_flag);

                        if(data.customer_details.customer_alert != null && data.customer_details.customer_alert != ""){
                            $("#customerAlert").html(data.customer_details.customer_alert+"<br>");
                            $("#customerAlert").show();
                            document.getElementById('customerAlert').classList.remove('d-none'); 
                        
                            if(data.customer_details.customer_allergies != null && data.customer_details.customer_allergies != ""){
                                $("#customerAlert").append("Allergies: " + data.customer_details.customer_allergies);
                                document.getElementById('customerAlert').classList.remove('d-none'); 
                            }
                        } else {
                            document.getElementById('customerAlert').classList.add('d-none'); 
                            $("#customerAlert").html("");
                        }

                       
                        //Recovery Invoice Buttons
                        var bhtml="Balances: ";
                        $("#customerRecoveryBtn").html();
                        if(recoveries.length > 0){
                            $("#customerRecoveryBtn").removeClass('d-none');                        
                        } else { 
                            $("#customerRecoveryBtn").addClass('d-none');                        
                        }

                        $.each(recoveries, function(index, invoice){
                            bhtml += '<button class="btn btn-soft-danger me-1" onclick="open_new_recovery_invoice(' + invoice.id_invoice+ ')" data-id-invoice = ' + invoice.id_invoice + '>' + invoice.balance + '</button>';
                        });
                        $("#customerRecoveryBtn").html(bhtml);
                    },
                    error: function(xhr, status, error) {
                        console.error("Error fetching customer details: " + error);
                    }
                });
            }

            // Show booking details section
            $("#bookingDetailsSection").show();
            // You can also load additional booking details via AJAX here if needed

        });
        
    });

   function handlePackageChange() {
        // Handle package change logic here
        var selectedPackageId = document.getElementById('packageSelect').value;
        var selectedDateTime = document.getElementById('bookingDateTime').value;

        //console.log("Selected Package ID: " + selectedPackageId);

        if(selectedPackageId != "" && selectedPackageId != null){
            if(selectedDateTime == "" || selectedDateTime == null){
                document.getElementById('bookingCategoriesSection').style.display = 'none';
                toastr.info('Please select date and time for the booking . . .');
                return;
            }
            // You can add AJAX calls here to fetch and update booking categories based on the selected package
            document.getElementById('bookingCategoriesSection').style.display = 'none';
            fetch(`${baseURL}/bookings/get_package_categories/${selectedPackageId}`, {
                method: 'POST',
                headers: {
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content'),
                    'Content-Type': 'application/json'
                }
            })
            .then(response => response.json())
            .then(data => {
                // Handle the response data here
                //console.log(data);
                // You can update the UI or perform other actions based on the received data
                var tableBody = document.getElementById('bookingCategoriesTableBody');
                var dtParts = selectedDateTime.split(' ');
                var package_total = 0;
                tableBody.innerHTML = ''; // Clear existing rows
                data.forEach(category => {                    
                    // Show "Offers" button if this category has a flexi-pricing mapping
                    var flexiPriceButton = '';
                    if (category.has_mapping) {
                        flexiPriceButton = `<button class="btn btn-sm btn-info ms-2" onclick="showFlexiPriceOffers(${category.id_package_category}, '${dtParts[0]} ${dtParts[1]}', ${category.category_price})" title="View Flexi-Price Offers">
                            <i class="ri-price-tag-3-line"></i> Offers
                        </button>`;
                    }

                    
                    var serviceHTML = '';
                    var serviceRatesHTML = '';

                    var ServiceIds = category.service_ids.split(',');
                    var mainServiceIds = category.package_service_ids.split(',');
                    var mainServiceNames = category.service_names.split('<br>');
                    var mainServiceRates = category.service_rates.split('<br>');

                    mainServiceIds.forEach((serviceId, index) => {
                        var options = category.options[serviceId];
                        var mainRate = parseFloat(mainServiceRates[index]);
                        var service_id = parseFloat(ServiceIds[index]);
                        var serviceName = mainServiceNames[index];

                        if (options && options.length > 0) {
                            
                            var optionHTML = `<select class="service-option form-select-sm mb-1" data-service-id="${service_id}" style="width:100%;">`;
                            options.forEach(opt => {
                                optionHTML += `<option value="${opt.service_id}" data-rate="${mainRate}">${opt.service_name}</option>`;
                            });
                            optionHTML += `</select>`;
                            // serviceHTML += `<strong></strong><br>${optionHTML}<br>`;
                            serviceHTML += `${optionHTML}<br>`;
                            serviceRatesHTML += mainRate + '<br>'; 
                        } else {
                          
                            serviceHTML += `<input type='hidden' class="service-option" data-service-id="${service_id}" value="${service_id}" data-rate="${mainRate}">${serviceName}<br>`;
                            serviceRatesHTML += mainRate + '<br>';
                        }
                    });    

                    var row = `<tr>                       
                        <td data-type-id="${category.id_package_type}" data-category-id="${category.id_package_category}" data-service-id="${category.service_ids}" data-category-price="${category.category_price}">${category.service_category}${flexiPriceButton}</td>
                        <td>${serviceHTML}</td>
                        <td class="text-end">${serviceRatesHTML}</td>
                        <td><input class="form-control service-date-input" type="date" name="service_date_${category.id_package_category}" value="${dtParts[0]}" data-category-id="${category.id_package_category}"></td>
                        <td><input class="form-control service-time-input" type="time" name="service_time_${category.id_package_category}" value="${dtParts[1]}" data-category-id="${category.id_package_category}"></td>
                        <td>
                            <select class="form-select staff-select" name="service_staff_${category.id_package_category}">
                                <option value="">Select Staff</option>
                                @foreach($staff as $member)
                                    <option value="{{ $member->id_staff }}" data-image="{{ $member->staff_image }}">{{ $member->staff_fullname }}</option>
                                @endforeach
                            </select>
                        </td>
                        <td class="text-end category-price" data-category-id="${category.id_package_category}">${category.category_price}</td>
                    </tr>`;
                    tableBody.innerHTML += row;
                    package_total += parseFloat(category.category_price);
                });
                
                // ============================================================================
                // FLEXI-PRICING MODULE: Date/Time Field Protection
                // ============================================================================
                // Prevent manual changes to date when flexi discount is applied
                $(document).on('change', '.service-date-input', function() {
                    if ($(this).attr('data-flexi-applied') === 'true') {
                        Swal.fire({
                            title: 'Date Locked',
                            text: 'This date is locked because a flexi-pricing discount has been applied. Please reopen the Offers modal to change the date.',
                            icon: 'info',
                            confirmButtonText: 'OK',
                            customClass: {
                                confirmButton: 'btn btn-primary'
                            },
                            buttonsStyling: false
                        });
                        // Revert to original date
                        var categoryRow = $(this).closest('tr');
                        var originalDate = categoryRow.attr('data-flexi-date');
                        if (originalDate) {
                            $(this).val(originalDate);
                        }
                    }
                });
                
                // Validate time is within timeslot range if timeslot-specific discount was applied
                $(document).on('change', '.service-time-input', function() {
                    if ($(this).attr('data-flexi-timeslot-applied') === 'true') {
                        var timeslotStart = $(this).attr('data-timeslot-start');
                        var timeslotEnd = $(this).attr('data-timeslot-end');
                        var selectedTime = $(this).val();
                        
                        if (timeslotStart && timeslotEnd && selectedTime) {
                            if (selectedTime < timeslotStart || selectedTime > timeslotEnd) {
                                Swal.fire({
                                    title: 'Invalid Time',
                                    text: `Time must be between ${timeslotStart} and ${timeslotEnd} for this discount to apply.`,
                                    icon: 'warning',
                                    confirmButtonText: 'OK',
                                    customClass: {
                                        confirmButton: 'btn btn-primary'
                                    },
                                    buttonsStyling: false
                                });
                                // Revert to timeslot start time
                                $(this).val(timeslotStart);
                            }
                        }
                    }
                });
                // Append total row
                var totalRow = `<tr class="border-top">
                    <td colspan="6" class="text-end"><strong>Total</strong></td>
                    <td class="text-end package-total"><strong>${package_total.toFixed(2)}</strong></td>
                </tr>`;
                tableBody.innerHTML += totalRow;

                // ============================================================================
                // FLEXI-PRICING MODULE: Discount Field Setup
                // ============================================================================
                // Create discount input field. If flexi-pricing is enabled:
                // - Create hidden input for form submission
                // - Create display-only input (readonly/disabled) for user viewing
                // - Auto-calculate discount from category discounts
                var discountReadonlyAttr = flexiPricingEnabled ? 'readonly' : '';
                var discountDisabledAttr = flexiPricingEnabled ? 'disabled' : '';
                var discountClass = flexiPricingEnabled ? 'form-control text-end bg-light' : 'form-control text-end';
                var discountOnChange = flexiPricingEnabled ? '' : 'onkeyup="updateTotals()"';
                var discountRow = `<tr>
                    <td colspan="6" class="text-end"><strong>Discount</strong>${flexiPricingEnabled ? ' <small class="text-muted">(Auto-calculated from Flexi-Pricing)</small>' : ''}</td>
                    <td class="text-end">
                        ${flexiPricingEnabled ? `<input type="hidden" name="booking_discount" value="0.00">` : ''}
                        <input type="text" ${discountOnChange} class="${discountClass}" ${flexiPricingEnabled ? 'id="booking_discount_display"' : 'name="booking_discount"'} value="0.00" ${discountReadonlyAttr} ${discountDisabledAttr}>
                    </td>
                </tr>`;
                tableBody.innerHTML += discountRow;
                
                // Enforce readonly/disabled state if flexi-pricing is enabled
                // Multiple layers of protection to prevent manual editing
                if (flexiPricingEnabled) {
                    setTimeout(function() {
                        var discountInput = document.querySelector('#booking_discount_display');
                        if (discountInput) {
                            discountInput.setAttribute('readonly', 'readonly');
                            discountInput.setAttribute('disabled', 'disabled');
                            discountInput.removeAttribute('onkeyup');
                            discountInput.classList.add('bg-light');
                            
                            // Prevent all forms of manual editing
                            discountInput.addEventListener('focus', function(e) {
                                e.preventDefault();
                                this.blur();
                                toastr.info('Discount is auto-calculated from Flexi-Pricing offers. Please use the "Offers" button to apply discounts.');
                                return false;
                            }, true);
                            discountInput.addEventListener('keydown', function(e) {
                                e.preventDefault();
                                e.stopPropagation();
                                return false;
                            }, true);
                            discountInput.addEventListener('keypress', function(e) {
                                e.preventDefault();
                                e.stopPropagation();
                                return false;
                            }, true);
                            discountInput.addEventListener('input', function(e) {
                                e.preventDefault();
                                e.stopPropagation();
                                updateFlexiDiscountTotal();
                                return false;
                            }, true);
                            discountInput.addEventListener('paste', function(e) {
                                e.preventDefault();
                                e.stopPropagation();
                                return false;
                            }, true);
                            discountInput.addEventListener('contextmenu', function(e) {
                                e.preventDefault();
                                return false;
                            });
                        }
                    }, 100);
                }

                var totalAfterDiscountRow = `<tr>
                    <td colspan="6" class="text-end"><strong>Total after Discount</strong></td>
                    <td class="text-end"><input type="text" onkeyup="updateTotals()" class="form-control text-end" disabled name="booking_total_after_discount" value="${package_total.toFixed(2)}" readonly></td>
                </tr>`;
                tableBody.innerHTML += totalAfterDiscountRow;

                var advanceRow = `<tr>
                    <td colspan="6" class="text-end"><strong>Advance Payment</strong></td>
                    <td class="text-end"><input type="text" onkeyup="updateTotals()" class="form-control text-end" name="booking_advance" value="0.00"></td>
                </tr>`;
                tableBody.innerHTML += advanceRow;

                //append payment mode row
                var paymentModeRow = `<tr>
                    <td colspan="6" class="text-end"><strong>Payment Mode</strong></td>
                    <td class="text-end">
                        <select class="form-select" name="booking_payment_mode">
                            <option value="Cash">Cash</option>
                            <option value="Card">Card</option>
                            <option value="Online">Online</option>
                            <option value="Check">Check</option>
                        </select>
                    </td>
                </tr>`;
                tableBody.innerHTML += paymentModeRow;

                //append instrument number row
                var instrumentNumberRow = `<tr>
                    <td colspan="6" class="text-end"><strong>Instrument Number</strong></td>
                    <td class="text-end">
                        <input type="text" class="form-control text-end" name="booking_instrument_number" value="">
                    </td>
                </tr>`;
                tableBody.innerHTML += instrumentNumberRow;

                var remainingTotalRow = `<tr class="border-top">
                    <td colspan="6" class="text-end"><strong>Remaining Total</strong></td>
                    <td class="text-end"><input type="text" onkeyup="updateTotals()" class="form-control text-end" name="booking_remaining_total" value="${package_total.toFixed(2)}" readonly></td>
                </tr>`;
                tableBody.innerHTML += remainingTotalRow;

                // Show booking categories section
                document.getElementById('bookingCategoriesSection').style.display = 'block';

                //initialize select2 for staff selects
                $('#bookingCategoriesTableBody select.staff-select').select2({
                    dropdownParent: $('#bookingCategoriesTableBody').parent(),
                    placeholder: 'Select Staff',
                    templateResult: formatStaff,
                    templateSelection: formatStaff
                });
                
                // Initialize flexi discount total (in case there are any pre-applied discounts)
                updateFlexiDiscountTotal();

                //initialize flatpickr                
                flatpickr(".flatpickr-time-input", {
                        enableTime: true,
                        noCalendar: true,
                        dateFormat: "H:i",
                        time_24hr: true,
                        minuteIncrement: 15,
                        minTime: "{{ $business->business_opening_time }}",
                        maxTime: "{{ $business->business_closing_time }}",
                    });

                //enable button and hide spinner
                document.getElementById('createBookingButton').disabled = false;
                document.getElementById('createBookingSpinner').classList.add('d-none');

            })
            .catch(error => {
                toastr.error('Error fetching package categories:', error);
            });

        } else {
            // Handle case when no package is selected
            toastr.info('Please select a package to continue . . .');
        }
        
    }

    function handleDateTimeChange() {
        // Handle date and time change logic here
        var selectedDateTime = document.getElementById('bookingDateTime').value;
        console.log("Selected Date & Time: " + selectedDateTime);
        // You can add logic to validate or update booking details based on the selected date and time
        if(selectedDateTime == "" || selectedDateTime == null){
            toastr.info('Please select date and time for the booking . . .');
            return;
        }
        
        // If categories are already displayed, refresh them with new date/time
        if(bookingCategoriesSection.style.display == 'block'){
            var selectedPackageId = document.getElementById('packageSelect').value;
            if(selectedPackageId){
                // Refresh categories with new date/time
                handlePackageChange();
            }
        } else {
            // If categories not displayed yet, just trigger package change
            handlePackageChange();
        }
    }
    
    /**
     * ============================================================================
     * FLEXI-PRICING MODULE: Discount Calculation
     * ============================================================================
     * Calculate total discount by summing discounts from all categories that have
     * flexi-pricing discounts applied. Updates the main discount field accordingly.
     */
    function updateFlexiDiscountTotal() {
        // Sum all flexi discounts from categories (stored in data-flexi-discount attribute)
        var totalFlexiDiscount = 0;
        var categoryRows = document.querySelectorAll('tr[data-flexi-discount]');
        
        categoryRows.forEach(function(row) {
            var categoryDiscount = parseFloat(row.getAttribute('data-flexi-discount')) || 0;
            totalFlexiDiscount += categoryDiscount;
        });
        
        // Update discount input based on flexi-pricing status
        if (flexiPricingEnabled) {
            // Update both hidden input (for form submission) and display input
            var discountHidden = document.querySelector('input[name="booking_discount"][type="hidden"]');
            var discountDisplay = document.querySelector('#booking_discount_display');
            if (discountHidden) {
                discountHidden.value = totalFlexiDiscount.toFixed(2);
            }
            if (discountDisplay) {
                discountDisplay.value = totalFlexiDiscount.toFixed(2);
                discountDisplay.setAttribute('readonly', 'readonly');
                discountDisplay.setAttribute('disabled', 'disabled');
            }
        } else {
            // Normal mode: update single discount input
            var discountInput = document.querySelector('input[name="booking_discount"]:not([type="hidden"])');
            if (discountInput) {
                discountInput.value = totalFlexiDiscount.toFixed(2);
            }
        }
        updateTotals();
    }

    function updateTotals() {
        var tableBody = document.getElementById('bookingCategoriesTableBody');
        var rows = tableBody.getElementsByTagName('tr');
        var package_total = 0;
        for (var i = 0; i < rows.length; i++) {
            var totalCell = rows[i].querySelector('.package-total');
            if(totalCell){
                package_total = parseFloat(totalCell.textContent) || 0;
                break;
            }
        }

        // Get discount input (hidden if flexi-pricing enabled, regular otherwise)
        var discountInput = flexiPricingEnabled 
            ? tableBody.querySelector('input[name="booking_discount"][type="hidden"]') 
            : tableBody.querySelector('input[name="booking_discount"]:not([type="hidden"])');
        var advanceInput = tableBody.querySelector('input[name="booking_advance"]');
        var totalAfterDiscountInput = tableBody.querySelector('input[name="booking_total_after_discount"]');
        var remainingTotalInput = tableBody.querySelector('input[name="booking_remaining_total"]');

        var discount = parseFloat(discountInput ? discountInput.value : 0) || 0;
        var advance = parseFloat(advanceInput.value) || 0;
        

        var totalAfterDiscount = package_total - discount;
        totalAfterDiscountInput.value = totalAfterDiscount.toFixed(2);

        var remainingTotal = totalAfterDiscount - advance;
        remainingTotalInput.value = remainingTotal.toFixed(2);
    }

    function formatStaff (option) {
       if (!option.id) {
            return option.text;
        }

        var imgSrc = staffImageBase + $(option.element).data('image');
        return $('<span><img src="'+ imgSrc +'" class="avatar-sm rounded-circle me-2" /> ' + option.text + '</span>');
    }

    function create_visits_from_booking() {


        var customerId = document.getElementById('customerId').value;
        if(customerId == "" || customerId == null){
            toastr.info('Please select a customer to create visits from booking . . .');
            return;        
        }

        var packageId = document.getElementById('packageSelect').value;
        if(packageId == "" || packageId == null){
            toastr.info('Please select a package to create visits from booking . . .');
            return;
        }

        var event_date = document.getElementById('bookingDateTime').value;
        if(event_date == "" || event_date == null){
            toastr.info('Please select event date and time to create visits from booking . . .');
            return;
        }

        var customer_name = document.getElementById('customerName').value;
        var referringStaff =  $('#referringStaffSelect').select2('data');
        var referringStaffName = referringStaff.length > 0 ? referringStaff[0].text : '';
        // Note: flexiPricingEnabled is now a global variable
        var discountInput = flexiPricingEnabled 
            ? document.querySelector('input[name="booking_discount"][type="hidden"]') 
            : document.querySelector('input[name="booking_discount"]:not([type="hidden"])');
        var advanceInput = document.querySelector('input[name="booking_advance"]');
        var paymentModeSelect = document.querySelector('select[name="booking_payment_mode"]');
        var instrumentNumberInput = document.querySelector('input[name="booking_instrument_number"]');
        var totalAfterDiscountInput = document.querySelector('input[name="booking_total_after_discount"]');
        var remainingTotalInput = document.querySelector('input[name="booking_remaining_total"]');
        var business_id = document.getElementById('branches').value;
        var payment_mode = paymentModeSelect.value;
        var instrument_number = instrumentNumberInput.value;

        // Collect booking data from table
        var categoryData = [];
        var tableBody = document.getElementById('bookingCategoriesTableBody');
        var rows = tableBody.getElementsByTagName('tr');
        for (var i = 0; i < rows.length; i++) {
            var cells = rows[i].getElementsByTagName('td');
            if(cells.length > 2){ // Skip total and other rows
                var id_service_type = cells[0].getAttribute('data-type-id');
                var category_price = cells[0].getAttribute('data-category-price');
                var id_service_category = cells[0].getAttribute('data-category-id');
                // var id_service = cells[0].getAttribute('data-service-id');                
                var category_name = cells[0].textContent.trim();
                var type_name = $("#packageSelect option:selected").text().trim();
                var dateInput = cells[3].querySelector('input[type="date"]');
                
                var timeInput = cells[4].querySelector('input[type="time"]');

                var category_start_time = dateInput.value + 'T' + timeInput.value + ":00";
                var visit_date = event_date;
               
                
                if(i==0){
                    visit_date = dateInput.value + 'T' + timeInput.value + ":00";
                }
                var category_staff = cells[5].querySelector('select.form-select');
                //if staff not selected, show error and return
                if(category_staff.value == "" || category_staff.value == null){
                    toastr.info('Please select staff for category: ' + category_name);
                    //enable button and hide spinner
                    //document.getElementById('createBookingButton').disabled = false;
                    document.getElementById('createBookingSpinner').classList.add('d-none');
                    return;
                }

                var id_service_new = [];
                var serviceRate = [];

                var serviceEls = rows[i].querySelectorAll('.service-option');

                serviceEls.forEach(el => {

                    // CASE 1: dropdown
                    if (el.tagName === 'SELECT') {
                        var selectedOption = el.options[el.selectedIndex];
                        id_service_new.push(selectedOption.value);
                        serviceRate.push(selectedOption.getAttribute('data-rate'));
                    }

                    // CASE 2: input
                    if (el.tagName === 'INPUT') {
                        id_service_new.push(el.getAttribute('data-service-id'));
                        serviceRate.push(el.getAttribute('data-rate'));
                    }
                });

                // convert to comma separated
                id_service_new = id_service_new.join(',');
                serviceRate = serviceRate.join(',');



                categoryData.push({
                    id_service_type: id_service_type,                    
                    id_service_category: id_service_category,
                    // id_service_old: id_service,                    
                    id_service: id_service_new,                    
                    category_name: category_name,
                    type_name: type_name,
                    category_date_time: category_start_time,                    
                    category_staff: category_staff.value,
                    serviceRate: serviceRate,
                    category_price: category_price,
                });
            }
        }
        //disable button and show spinner
        document.getElementById('createBookingButton').disabled = true;
        document.getElementById('createBookingSpinner').classList.remove('d-none');
        $.ajax({
            url : "{{ route('create_visits_from_booking') }}",
            data: { 
                customer_id: customerId,
                customer_name: customer_name,
                business_id: business_id,
                package_id: packageId,
                event_date: event_date,
                visit_date: visit_date,
                referring_staff: referringStaffName,
                category_data: categoryData,
                booking_discount: discountInput.value,
                booking_advance: advanceInput.value,
                booking_payment_mode: payment_mode,
                booking_instrument_number: instrument_number,
                booking_total_after_discount: totalAfterDiscountInput.value,
                booking_remaining_total: remainingTotalInput.value
             },
            type: "POST", // The POST method
            dataType: "json", // Expected data type of the response
            headers: {'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content')},
            success: function(data) {
                document.getElementById('createBookingButton').disabled = false;
                document.getElementById('createBookingSpinner').classList.add('d-none');
                if(data.message_type === "error"){
                    toastr.error(data.message);
                } else {
                    toastr.success(data.message);
                    //open print page 
                    var printURL = baseURL + '/bookings/print_booking_receipt/' + data.booking_id;
                    window.open(printURL, '_self');
                    // window.location.reload();
                }

                //enable button and hide spinner
                //document.getElementById('createBookingButton').disabled = false;
                document.getElementById('createBookingSpinner').classList.add('d-none');
                
                // Additional success handling
            },
            error: function(xhr, status, error) {
                //enable button and hide spinner
                document.getElementById('createBookingButton').disabled = false;
                document.getElementById('createBookingSpinner').classList.add('d-none');
                console.error("Error creating visits from booking: " + error);
                toastr.error('Error creating visits from booking.');
            }
        });
                
    }

    /**
     * ============================================================================
     * FLEXI-PRICING MODULE: Show Discount Offers Modal
     * ============================================================================
     * Opens a modal displaying flexi-pricing discount offers for a package category.
     * Shows a 7-day window (3 days before, selected day, 3 days after) with
     * capacity, bookings, and available discounts for each day/timeslot.
     * 
     * @param {number} packageCategoryId - Package category ID
     * @param {string} eventDateTime - Selected event date and time
     * @param {number} categoryPrice - Price of the category (for discount calculation)
     */
    function showFlexiPriceOffers(packageCategoryId, eventDateTime, categoryPrice) {
        // Store category info for discount application
        window.currentFlexiCategoryId = packageCategoryId;
        window.currentFlexiCategoryPrice = parseFloat(categoryPrice) || 0;
        
        // Check if this category already has a flexi discount applied
        var categoryRow = document.querySelector(`tr[data-category-id="${packageCategoryId}"]`);
        if (categoryRow && categoryRow.getAttribute('data-flexi-discount')) {
            Swal.fire({
                title: 'Discount Already Applied',
                text: 'This category already has a flexi-pricing discount applied. Reopening the modal will allow you to change it.',
                icon: 'info',
                confirmButtonText: 'Continue',
                customClass: {
                    confirmButton: 'btn btn-primary'
                },
                buttonsStyling: false
            }).then((result) => {
                if (result.isConfirmed) {
                    // Reset the category fields
                    var dateInput = categoryRow.querySelector(`input[name="service_date_${packageCategoryId}"]`);
                    var timeInput = categoryRow.querySelector(`input[name="service_time_${packageCategoryId}"]`);
                    if (dateInput) {
                        dateInput.readOnly = false;
                        dateInput.classList.remove('bg-light');
                        dateInput.removeAttribute('data-flexi-applied');
                    }
                    if (timeInput) {
                        timeInput.removeAttribute('min');
                        timeInput.removeAttribute('max');
                        timeInput.removeAttribute('data-flexi-timeslot-applied');
                        timeInput.removeAttribute('data-timeslot-start');
                        timeInput.removeAttribute('data-timeslot-end');
                        timeInput.removeAttribute('title');
                    }
                    categoryRow.removeAttribute('data-flexi-discount');
                    categoryRow.removeAttribute('data-flexi-date');
                    categoryRow.removeAttribute('data-flexi-timeslot');
                    
                    // Update discount total after removing this category's discount
                    updateFlexiDiscountTotal();
                    
                    // Open modal
                    var modalElement = document.getElementById('flexiPriceModal');
                    var modal = new bootstrap.Modal(modalElement);
                    modal.show();
                    loadFlexiPriceOffers(packageCategoryId, eventDateTime);
                }
            });
            return;
        }
        
        // Show modal using Bootstrap 5 API
        var modalElement = document.getElementById('flexiPriceModal');
        var modal = new bootstrap.Modal(modalElement);
        modal.show();
        loadFlexiPriceOffers(packageCategoryId, eventDateTime);
    }
    
    function loadFlexiPriceOffers(packageCategoryId, eventDateTime) {
        
        
        $('#flexi-price-content').html('<div class="text-center"><div class="spinner-border" role="status"><span class="visually-hidden">Loading...</span></div></div>');
        
        $.ajax({
            url: baseURL + '/bookings/get_flexi_price',
            type: 'GET',
            data: {
                package_category_id: packageCategoryId,
                event_date: eventDateTime
            },
            success: function(response) {
                if (response.success) {
                    // Build comprehensive table structure showing discounts for 7-day window
                    let html = `<div class="mb-3">`;
                    html += `<p class="mb-1"><strong>Package:</strong> ${response.package_category_name} ${response.package_category_price ? '{{ config("constants.CURRENCY") }}' + parseFloat(response.package_category_price).toFixed(2) : ''} | <strong>Mapping:</strong> ${response.mapping_name}</p>`;
                    html += `<p class="text-muted small mb-0"><i class="ri-information-line"></i> Discounts are based on booking density for the week around your selected date.</p>`;
                    html += `</div>`;
                    
                    if (response.discounts && response.discounts.length > 0) {
                        // Get all unique timeslots (only if mapping uses timeslot-specific bands)
                        let timeslots = response.timeslots || [];
                        let hasTimeslots = timeslots.length > 0;
                        
                        // Build table header
                        html += '<div class="table-responsive" style="max-height: 600px; overflow-y: auto;">';
                        html += '<table class="table table-bordered table-sm" style="font-size: 0.875rem;">';
                        html += '<thead class="table-light" style="position: sticky; top: 0; z-index: 10;">';
                        html += '<tr>';
                        html += '<th rowspan="2" class="align-middle">Date</th>';
                        html += '<th rowspan="2" class="align-middle">Floor</th>';
                        
                        if (hasTimeslots) {
                            // Time slot columns
                            timeslots.forEach(function(ts) {
                                html += `<th colspan="4" class="text-center" style="background-color: #e7f3ff;">${ts.name}<br><small>(${ts.start_time} - ${ts.end_time})</small></th>`;
                            });
                        } else {
                            // All day column
                            html += '<th colspan="4" class="text-center" style="background-color: #e7f3ff;">All Day</th>';
                        }
                        
                        html += '</tr><tr>';
                        
                        // Sub-columns for each timeslot
                        let subColumns = ['Capacity', 'Booked', 'Available', 'Discount'];
                        let repeatCount = hasTimeslots ? timeslots.length : 1;
                        for (let i = 0; i < repeatCount; i++) {
                            subColumns.forEach(function(col) {
                                html += `<th class="text-center" style="font-weight: normal; font-size: 0.8rem;">${col}</th>`;
                            });
                        }
                        
                        html += '</tr></thead><tbody>';
                        
                        // Build table rows - one row per date
                        response.discounts.forEach(function(discount) {
                            let dateObj = new Date(discount.date);
                            let dayAbbr = ['Sun', 'Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat'][dateObj.getDay()];
                            let monthAbbr = ['Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun', 'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec'][dateObj.getMonth()];
                            let dayNum = dateObj.getDate();
                            let year = dateObj.getFullYear();
                            
                            // Check if this is the selected date
                            let isSelectedDate = discount.date === response.event_date;
                            let dateDisplay = `${dayAbbr}, ${monthAbbr} ${dayNum}, ${year}`;
                            if (isSelectedDate) {
                                dateDisplay = '<i class="ri-star-fill text-warning"></i> ' + dateDisplay;
                            }
                            
                            html += `<tr>`;
                            html += `<td class="fw-bold">${dateDisplay}</td>`;
                            
                            // Floor column - show all floors or first floor
                            let floorNames = response.floors.map(f => f.name).join(', ') || 'N/A';
                            html += `<td>${floorNames}</td>`;
                            
                            if (hasTimeslots) {
                                // For each timeslot, show data
                                timeslots.forEach(function(ts) {
                                    let tsData = discount.time_slots ? discount.time_slots.find(t => t.timeslot_id == ts.id) : null;
                                    
                                    if (tsData) {
                                        let capacity = tsData.capacity || 0;
                                        let booked = tsData.bookings || 0;
                                        let available = capacity - booked;
                                        let discountVal = tsData.band.discount || 0;
                                        let discountType = tsData.band.discount_type || 'percentage';
                                        
                                        // Calculate discounted price
                                        let originalPrice = parseFloat(response.package_category_price) || 0;
                                        let discountedPrice = 0;
                                        if (discountType === 'percentage') {
                                            discountedPrice = originalPrice * (1 - discountVal / 100);
                                        } else {
                                            discountedPrice = originalPrice - discountVal;
                                        }
                                        
                                        let discountDisplay = discountVal > 0 
                                            ? `${discountVal}%<br><small class="text-success fw-bold">${discountedPrice.toFixed(2)} Rs</small>`
                                            : '0%';
                                        
                                        html += `<td class="text-center">${capacity}</td>`;
                                        html += `<td class="text-center">${booked}</td>`;
                                        html += `<td class="text-center">${available}</td>`;
                                        html += `<td class="text-center">`;
                                        if (discountVal > 0) {
                                            // Pass date, timeslot start time, and timeslot info
                                            html += `<button class="btn btn-sm btn-primary" style="font-size: 0.85rem; line-height: 1.2; padding: 8px 12px; white-space: normal;" onclick="applyFlexiDiscount(${discountVal}, '${discountType}', ${packageCategoryId}, '${discount.date}', '${ts.start_time}', '${ts.end_time}', '${ts.id}')" title="Apply ${discountVal}% discount">${discountDisplay}</button>`;
                                        } else {
                                            html += discountDisplay;
                                        }
                                        html += `</td>`;
                                    } else {
                                        // No data for this timeslot
                                        html += `<td class="text-center">-</td>`;
                                        html += `<td class="text-center">-</td>`;
                                        html += `<td class="text-center">-</td>`;
                                        html += `<td class="text-center">-</td>`;
                                    }
                                });
                            } else {
                                // All day data
                                let capacity = discount.capacity || 0;
                                let booked = discount.bookings || 0;
                                let available = capacity - booked;
                                let discountVal = discount.band.discount || 0;
                                let discountType = discount.band.discount_type || 'percentage';
                                
                                // Calculate discounted price
                                let originalPrice = parseFloat(response.package_category_price) || 0;
                                let discountedPrice = 0;
                                if (discountType === 'percentage') {
                                    discountedPrice = originalPrice * (1 - discountVal / 100);
                                } else {
                                    discountedPrice = originalPrice - discountVal;
                                }
                                
                                let discountDisplay = discountVal > 0 
                                    ? `${discountVal}%<br><small class="text-success fw-bold">${discountedPrice.toFixed(2)} Rs</small>`
                                    : '0%';
                                
                                html += `<td class="text-center">${capacity}</td>`;
                                html += `<td class="text-center">${booked}</td>`;
                                html += `<td class="text-center">${available}</td>`;
                                html += `<td class="text-center">`;
                                if (discountVal > 0) {
                                    // Pass date for all-day discount (no timeslot)
                                    html += `<button class="btn btn-sm btn-primary" style="font-size: 0.85rem; line-height: 1.2; padding: 8px 12px; white-space: normal;" onclick="applyFlexiDiscount(${discount.band.discount}, '${discount.band.discount_type}', ${packageCategoryId}, '${discount.date}', null, null, null)" title="Apply ${discountVal}% discount">${discountDisplay}</button>`;
                                } else {
                                    html += discountDisplay;
                                }
                                html += `</td>`;
                            }
                            
                            html += `</tr>`;
                        });
                        
                        html += '</tbody></table></div>';
                    } else {
                        html += '<div class="alert alert-info">No discount offers available for the selected date range.</div>';
                    }
                    
                    $('#flexi-price-content').html(html);
                } else {
                    $('#flexi-price-content').html('<div class="alert alert-warning">' + (response.message || 'No offers available') + '</div>');
                }
            },
            error: function(xhr) {
                let errorMsg = xhr.responseJSON?.message || 'Error loading flexi-price offers';
                $('#flexi-price-content').html('<div class="alert alert-danger">' + errorMsg + '</div>');
            }
        });
    }

    /**
     * ============================================================================
     * FLEXI-PRICING MODULE: Apply Discount to Category
     * ============================================================================
     * Applies a flexi-pricing discount to a specific package category.
     * 
     * - Locks the date field (readonly) to the selected date
     * - Sets time constraints if timeslot-specific discount (allows selection within range)
     * - Calculates discount amount based on category price
     * - Updates the main discount field (sum of all category discounts)
     * 
     * @param {number} discount - Discount value (percentage or fixed amount)
     * @param {string} discountType - 'percentage' or 'fixed'
     * @param {number} packageCategoryId - Package category ID
     * @param {string} selectedDate - Selected date (YYYY-MM-DD)
     * @param {string} timeslotStartTime - Timeslot start time (HH:mm) or null for all-day
     * @param {string} timeslotEndTime - Timeslot end time (HH:mm) or null for all-day
     * @param {number} timeslotId - Timeslot ID or null for all-day
     */
    function applyFlexiDiscount(discount, discountType, packageCategoryId, selectedDate, timeslotStartTime, timeslotEndTime, timeslotId) {
        // Get discount input (hidden if flexi-pricing enabled, regular otherwise)
        var discountInput = flexiPricingEnabled 
            ? document.querySelector('input[name="booking_discount"][type="hidden"]') 
            : document.querySelector('input[name="booking_discount"]:not([type="hidden"])');
        
        if (!discountInput) {
            toastr.error('Could not find discount input');
            return;
        }
        
        // Get the category price cell and row
        var categoryPriceCell = document.querySelector(`td.category-price[data-category-id="${packageCategoryId}"]`);
        if (!categoryPriceCell) {
            toastr.error('Could not find category price');
            return;
        }
        
        var categoryRow = categoryPriceCell.closest('tr');
        if (!categoryRow) {
            toastr.error('Could not find category row');
            return;
        }
        
        // Lock date field to selected date (readonly)
        var dateInput = categoryRow.querySelector(`input[name="service_date_${packageCategoryId}"]`);
        if (dateInput && selectedDate) {
            dateInput.value = selectedDate;
            dateInput.readOnly = true;
            dateInput.classList.add('bg-light');
            dateInput.setAttribute('data-flexi-applied', 'true');
        }
        
        // Set time constraints if timeslot-specific discount (user can still choose within range)
        var timeInput = categoryRow.querySelector(`input[name="service_time_${packageCategoryId}"]`);
        if (timeInput) {
            if (timeslotStartTime && timeslotEndTime) {
                // Timeslot-specific: set default to start time, allow selection within range
                timeInput.value = timeslotStartTime;
                timeInput.setAttribute('min', timeslotStartTime);
                timeInput.setAttribute('max', timeslotEndTime);
                timeInput.setAttribute('data-timeslot-start', timeslotStartTime);
                timeInput.setAttribute('data-timeslot-end', timeslotEndTime);
                timeInput.setAttribute('data-flexi-timeslot-applied', 'true');
                timeInput.setAttribute('title', `Time must be between ${timeslotStartTime} and ${timeslotEndTime}`);
            } else {
                // All-day discount: no time constraints
                timeInput.removeAttribute('min');
                timeInput.removeAttribute('max');
                timeInput.removeAttribute('data-timeslot-start');
                timeInput.removeAttribute('data-timeslot-end');
                timeInput.removeAttribute('data-flexi-timeslot-applied');
            }
        }
        
        // Calculate discount amount for this category
        var categoryPrice = parseFloat(categoryPriceCell.textContent.replace(/,/g, '')) || 0;
        var newDiscountAmount = 0;
        
        if (discountType === 'percentage') {
            newDiscountAmount = (categoryPrice * discount) / 100;
        } else {
            newDiscountAmount = discount; // Fixed amount
        }
        
        // Store discount info on category row (used by updateFlexiDiscountTotal)
        categoryRow.setAttribute('data-flexi-discount', newDiscountAmount.toFixed(2));
        categoryRow.setAttribute('data-flexi-date', selectedDate || '');
        categoryRow.setAttribute('data-flexi-timeslot', timeslotId || '');
        
        // Update main discount field (sum of all category discounts)
        updateFlexiDiscountTotal();
        updateTotals();
        
        // Close modal
        var modalElement = document.getElementById('flexiPriceModal');
        var modal = bootstrap.Modal.getInstance(modalElement);
        if (modal) {
            modal.hide();
        }
        
        toastr.success('Discount applied successfully. Date and time have been updated.');
    }

</script>
@endsection