<?php

namespace Tests\Feature;

use App\Models\BlockEvent;
use App\Models\Business;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Support\Facades\DB;
use Tests\TestCase;

class BlockEventsTest extends TestCase
{
    use RefreshDatabase;

    protected function setUp(): void
    {
        parent::setUp();
        $this->business = Business::factory()->create();
        $this->user = User::factory()->create(['business_id' => $this->business->id_business]);
        session(['business_id' => $this->business->id_business]);
        session(['common_services' => 'No']);
    }

    /** @test */
    public function block_events_page_loads_successfully()
    {
        $response = $this->actingAs($this->user)->get('/block_events');
        $response->assertStatus(200);
        $response->assertViewIs('block_events.block_events_listing');
    }

    /** @test */
    public function can_create_new_block_event()
    {
        $response = $this->actingAs($this->user)->post('/block_events/save_or_update', [
            'block_event_name' => 'Lunch Break',
            'block_event_desc' => 'Staff lunch break time',
        ]);

        $response->assertStatus(200);
        $response->assertJson(['success' => true]);
        
        $this->assertDatabaseHas('block_events', [
            'block_event_name' => 'Lunch Break',
            'block_event_desc' => 'Staff lunch break time',
            'business_id' => $this->business->id_business,
            'block_event_duration' => '00:00:00', // Default duration
        ]);
    }

    /** @test */
    public function can_update_existing_block_event()
    {
        $blockEvent = BlockEvent::factory()->create([
            'business_id' => $this->business->id_business,
            'block_event_name' => 'Old Event',
            'block_event_desc' => 'Old Description',
            'block_event_duration' => '01:00:00',
        ]);

        $response = $this->actingAs($this->user)->post('/block_events/save_or_update', [
            'id_block_events' => $blockEvent->id_block_events,
            'block_event_name' => 'Updated Event',
            'block_event_desc' => 'Updated Description',
        ]);

        $response->assertStatus(200);
        $response->assertJson(['success' => true]);
        
        $this->assertDatabaseHas('block_events', [
            'id_block_events' => $blockEvent->id_block_events,
            'block_event_name' => 'Updated Event',
            'block_event_desc' => 'Updated Description',
            'block_event_duration' => '01:00:00', // Preserved existing duration
        ]);
    }

    /** @test */
    public function block_event_validation_requires_name()
    {
        $response = $this->actingAs($this->user)->post('/block_events/save_or_update', [
            'block_event_desc' => 'Description',
        ]);

        $response->assertStatus(500);
        $response->assertJson(['success' => false]);
    }

    /** @test */
    public function block_event_validation_requires_description()
    {
        $response = $this->actingAs($this->user)->post('/block_events/save_or_update', [
            'block_event_name' => 'Event Name',
        ]);

        $response->assertStatus(500);
        $response->assertJson(['success' => false]);
    }

    /** @test */
    public function block_events_use_ho_business_id_when_common_services_enabled()
    {
        // Create Head Office business
        $hoBusiness = Business::factory()->create(['ho' => 'Yes']);
        $this->business = Business::factory()->create(['ho' => 'No']);
        $this->user = User::factory()->create(['business_id' => $this->business->id_business]);
        
        session(['business_id' => $this->business->id_business]);
        session(['common_services' => 'Yes']);

        $response = $this->actingAs($this->user)->post('/block_events/save_or_update', [
            'block_event_name' => 'Common Event',
            'block_event_desc' => 'Common Description',
        ]);

        $response->assertStatus(200);
        $response->assertJson(['success' => true]);
        
        // Should be saved with HO business_id
        $this->assertDatabaseHas('block_events', [
            'block_event_name' => 'Common Event',
            'business_id' => $hoBusiness->id_business,
        ]);
    }

    /** @test */
    public function block_events_use_current_business_id_when_common_services_disabled()
    {
        session(['common_services' => 'No']);

        $response = $this->actingAs($this->user)->post('/block_events/save_or_update', [
            'block_event_name' => 'Branch Event',
            'block_event_desc' => 'Branch Description',
        ]);

        $response->assertStatus(200);
        $response->assertJson(['success' => true]);
        
        // Should be saved with current business_id
        $this->assertDatabaseHas('block_events', [
            'block_event_name' => 'Branch Event',
            'business_id' => $this->business->id_business,
        ]);
    }

    /** @test */
    public function block_events_list_shows_only_current_business_events()
    {
        BlockEvent::factory()->create([
            'business_id' => $this->business->id_business,
            'block_event_name' => 'Event 1',
        ]);

        $otherBusiness = Business::factory()->create();
        BlockEvent::factory()->create([
            'business_id' => $otherBusiness->id_business,
            'block_event_name' => 'Event 2',
        ]);

        $response = $this->actingAs($this->user)->get('/block_events');
        $response->assertStatus(200);
        
        $blockEvents = $response->viewData('blockEvents');
        $this->assertCount(1, $blockEvents);
        $this->assertEquals('Event 1', $blockEvents->first()->block_event_name);
    }

    /** @test */
    public function new_block_event_has_default_duration()
    {
        $response = $this->actingAs($this->user)->post('/block_events/save_or_update', [
            'block_event_name' => 'New Event',
            'block_event_desc' => 'New Description',
        ]);

        $response->assertStatus(200);
        
        $blockEvent = BlockEvent::where('block_event_name', 'New Event')->first();
        $this->assertEquals('00:00:00', $blockEvent->block_event_duration);
    }
}
