<?php

namespace Tests\Feature;

use App\Models\Business;
use App\Models\Customers;
use App\Models\Invoice;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Foundation\Testing\WithFaker;
use Tests\TestCase;

class CustomerControllerTest extends TestCase
{
    use RefreshDatabase;

    /**
     * Test that customer search returns matching results.
     */
    public function test_customer_search_returns_matching_results(): void
    {
        // Arrange: Create test data
        $business = Business::factory()->create();
        $user = User::factory()->create(['business_id' => $business->id_business]);
        $customer1 = Customers::factory()->create([
            'business_id' => $business->id_business,
            'customer_name' => 'John Doe',
            'customer_cell' => '1234567890',
        ]);
        $customer2 = Customers::factory()->create([
            'business_id' => $business->id_business,
            'customer_name' => 'Jane Smith',
            'customer_cell' => '0987654321',
        ]);

        // Act: Perform the search (route is GET, so use query parameter)
        $response = $this->actingAs($user)->get('/customer_search?term=John');

        // Assert: Verify the response
        $response->assertStatus(200);
        $response->assertJsonStructure([
            '*' => ['id', 'text', 'customer_name', 'customer_cell', 'customer_email', 'customer_card', 'customer_type']
        ]);
        
        // Check that John Doe is in the results
        $json = $response->json();
        $this->assertTrue(
            collect($json)->contains(fn($item) => $item['customer_name'] === 'John Doe')
        );
    }

    /**
     * Test that customer search returns empty when no matches found.
     */
    public function test_customer_search_returns_empty_when_no_matches(): void
    {
        // Arrange: Create a customer
        $business = Business::factory()->create();
        $user = User::factory()->create(['business_id' => $business->id_business]);
        Customers::factory()->create([
            'business_id' => $business->id_business,
            'customer_name' => 'John Doe',
        ]);

        // Act: Search for something that doesn't exist
        $response = $this->actingAs($user)->get('/customer_search?term=NonexistentCustomer');

        // Assert: Should return empty array or no matching results
        $response->assertStatus(200);
        $json = $response->json();
        $this->assertIsArray($json);
    }

    /**
     * Test that customer search works with phone number.
     */
    public function test_customer_search_finds_by_phone_number(): void
    {
        // Arrange
        $business = Business::factory()->create();
        $user = User::factory()->create(['business_id' => $business->id_business]);
        $customer = Customers::factory()->create([
            'business_id' => $business->id_business,
            'customer_cell' => '5551234567',
            'customer_name' => 'Test User',
        ]);

        // Act
        $response = $this->actingAs($user)->get('/customer_search?term=555123');

        // Assert
        $response->assertStatus(200);
        $json = $response->json();
        $this->assertNotEmpty($json);
        $this->assertTrue(
            collect($json)->contains(fn($item) => str_contains($item['customer_cell'], '555123'))
        );
    }

    /**
     * Test getting customer details includes recovery invoices.
     */
    public function test_get_customer_details_includes_recoveries(): void
    {
        // Arrange
        $business = Business::factory()->create();
        $user = User::factory()->create(['business_id' => $business->id_business]);
        $customer = Customers::factory()->create([
            'business_id' => $business->id_business,
        ]);
        
        // Create a recovery invoice
        $recoveryInvoice = Invoice::factory()->create([
            'customer_id' => $customer->id_customers,
            'business_id' => $business->id_business,
            'is_recovery' => 'Yes',
            'invoice_status' => 'valid',
            'balance' => 100.00,
        ]);

        // Act
        $response = $this->actingAs($user)->get('/get_customer_details?id_customer=' . $customer->id_customers);

        // Assert
        $response->assertStatus(200);
        $response->assertJsonStructure([
            'status',
            'customer_details',
            'recoveries' => [
                '*' => ['id_invoice', 'balance']
            ]
        ]);
        
        $json = $response->json();
        $this->assertEquals('success', $json['status']);
        $this->assertNotEmpty($json['recoveries']);
    }

    /**
     * Test that customer list page requires authentication.
     */
    public function test_customer_list_requires_authentication(): void
    {
        // Act: Try to access without authentication
        $response = $this->get('/customers');

        // Assert: Should redirect to login or return 401/403
        // Adjust based on your authentication setup
        $this->assertContains($response->status(), [302, 401, 403]);
    }

    /**
     * Test that customer list loads for authenticated user with business session.
     */
    public function test_customer_list_loads_for_authenticated_user(): void
    {
        // Arrange
        $business = Business::factory()->create();
        $user = User::factory()->create([
            'business_id' => $business->id_business,
        ]);

        // Act: Set session and authenticate
        session(['business_id' => $business->id_business]);
        $response = $this->actingAs($user)->get('/customers');

        // Assert
        $response->assertStatus(200);
        $response->assertViewIs('customers.customer_list');
    }
}
