<?php

namespace Tests\Feature;

use App\Models\Customers;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Support\Facades\DB;
use Tests\TestCase;

/**
 * Database Isolation Verification Tests
 * 
 * Run these tests to verify that your test setup is safe and isolated.
 * These tests ensure that:
 * 1. Tests run in the 'testing' environment
 * 2. Tests use a separate database (SQLite in-memory or test database)
 * 3. Data doesn't persist between tests
 * 4. RefreshDatabase trait is working correctly
 */
class DatabaseIsolationTest extends TestCase
{
    use RefreshDatabase;

    /**
     * Verify that tests are running in the 'testing' environment.
     * This is critical - if this fails, tests might be using your production database!
     */
    public function test_runs_in_testing_environment(): void
    {
        $appEnv = config('app.env');
        
        $this->assertEquals('testing', $appEnv, 
            'CRITICAL: Tests must run in "testing" environment. Current: ' . $appEnv
        );
    }

    /**
     * Verify that tests are using a test database, not production.
     * 
     * This test checks:
     * - For SQLite: Should use ":memory:" (in-memory database)
     * - For MySQL: Should use a test database (not production)
     * 
     * IMPORTANT: Update the assertion below with your production database name
     * to ensure tests never accidentally connect to it.
     */
    public function test_uses_test_database(): void
    {
        $connection = config('database.default');
        $database = config("database.connections.{$connection}.database");
        
        // For SQLite in-memory, database should be ":memory:"
        if ($connection === 'sqlite') {
            $this->assertEquals(':memory:', $database, 
                'SQLite tests should use in-memory database (:memory:)'
            );
        }
        
        // For MySQL/other databases, verify it's NOT your production database
        // TODO: Replace 'your_production_database' with your actual production database name
        $productionDatabaseNames = [
            'skedwise_production',
            'skedwise_prod',
            'production',
            // Add your production database names here
        ];
        
        foreach ($productionDatabaseNames as $prodDb) {
            $this->assertNotEquals($prodDb, $database, 
                "CRITICAL: Tests are using production database: {$prodDb}"
            );
        }
        
        // Uncomment the line below to see which database is being used
        // dump("Test database: {$database} (Connection: {$connection})");
    }

    /**
     * Verify that RefreshDatabase trait is working.
     * Data created in one test should NOT exist in the next test.
     */
    public function test_database_isolation_first_test(): void
    {
        // Create a unique customer that should NOT exist in other tests
        $customer = Customers::factory()->create([
            'customer_name' => 'Isolation Test Customer - ' . uniqid(),
        ]);
        
        // Verify it exists in this test
        $this->assertDatabaseHas('customers', [
            'id_customers' => $customer->id_customers,
        ]);
        
        // Store the ID for the next test to check
        $this->customerId = $customer->id_customers;
    }

    /**
     * This test runs AFTER test_database_isolation_first_test.
     * If RefreshDatabase is working, the customer from the previous test should NOT exist.
     */
    public function test_database_isolation_second_test(): void
    {
        // This should be empty or only contain data from this test
        // The customer from the previous test should be gone
        $customerCount = Customers::count();
        
        // Create a new customer in this test
        $newCustomer = Customers::factory()->create([
            'customer_name' => 'Isolation Test Customer 2 - ' . uniqid(),
        ]);
        
        // Verify only our new customer exists (or database is empty)
        $this->assertLessThanOrEqual(1, Customers::count(), 
            'Database should be isolated - previous test data should not exist'
        );
    }

    /**
     * Verify that database transactions are working.
     * Each test should be wrapped in a transaction that gets rolled back.
     */
    public function test_transaction_isolation(): void
    {
        $initialCount = Customers::count();
        
        // Create a customer
        $customer = Customers::factory()->create();
        
        // Verify it exists
        $this->assertDatabaseHas('customers', [
            'id_customers' => $customer->id_customers,
        ]);
        
        // After this test completes, RefreshDatabase should rollback this transaction
        // The next test will verify this
    }

    /**
     * Verify that the transaction from the previous test was rolled back.
     */
    public function test_transaction_rollback_verified(): void
    {
        // If transactions are working, the customer from the previous test should be gone
        // This test verifies that RefreshDatabase is properly isolating tests
        $this->assertTrue(true, 'If this test passes, transaction isolation is working');
    }

    /**
     * Verify that we can create and query data within a single test.
     * This ensures the database connection is working properly.
     */
    public function test_database_operations_work(): void
    {
        // Create test data
        $customer1 = Customers::factory()->create(['customer_name' => 'Test Customer 1']);
        $customer2 = Customers::factory()->create(['customer_name' => 'Test Customer 2']);
        
        // Verify we can query it
        $customers = Customers::whereIn('customer_name', ['Test Customer 1', 'Test Customer 2'])->get();
        
        $this->assertCount(2, $customers, 'Should be able to create and query data in tests');
    }

    /**
     * Summary test - displays current test configuration.
     * Run this to see what database/environment your tests are using.
     */
    public function test_display_test_configuration(): void
    {
        $appEnv = config('app.env');
        $connection = config('database.default');
        $database = config("database.connections.{$connection}.database");
        
        // This test will always pass, but shows the configuration
        $this->assertTrue(true);
        
        // Uncomment to see your test configuration:
        /*
        dump([
            'Environment' => $appEnv,
            'Database Connection' => $connection,
            'Database Name' => $database,
            'Using In-Memory' => $database === ':memory:',
        ]);
        */
    }
}
