<?php

namespace Tests\Feature;

use App\Models\DiscountReason;
use App\Models\DiscountPassword;
use App\Models\Business;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class DiscountConfigTest extends TestCase
{
    use RefreshDatabase;

    protected function setUp(): void
    {
        parent::setUp();
        $this->business = Business::factory()->create();
        $this->user = User::factory()->create(['business_id' => $this->business->id_business]);
        session(['business_id' => $this->business->id_business]);
    }

    /** @test */
    public function discount_config_page_loads_successfully()
    {
        $response = $this->actingAs($this->user)->get('/discount-configuration');
        $response->assertStatus(200);
        $response->assertViewIs('discount_configuration.list');
    }

    /** @test */
    public function discount_config_page_shows_discount_reasons_and_users()
    {
        DiscountReason::factory()->create(['business_id' => $this->business->id_business]);
        DiscountPassword::factory()->create(['business_id' => $this->business->id_business]);

        $response = $this->actingAs($this->user)->get('/discount-configuration');
        $response->assertStatus(200);
        
        $discountReasons = $response->viewData('discountReasons');
        $discountUsers = $response->viewData('discountUsers');
        
        $this->assertCount(1, $discountReasons);
        $this->assertCount(1, $discountUsers);
    }

    // Discount Reasons Tests
    /** @test */
    public function can_create_percentage_discount_reason()
    {
        $response = $this->actingAs($this->user)->post('/discount-configuration/save_or_update_reason', [
            'discount_reason' => 'VIP Discount',
            'type' => 'Percentage',
            'value' => 15,
            'active' => 'Yes',
        ]);

        $response->assertStatus(200);
        $response->assertJson(['success' => true]);
        
        $this->assertDatabaseHas('discount_reasons', [
            'discount_reason' => 'VIP Discount',
            'discount_perc' => 15,
            'fixed_amount' => 0,
            'active' => 'Yes',
            'business_id' => $this->business->id_business,
        ]);
    }

    /** @test */
    public function can_create_fixed_amount_discount_reason()
    {
        $response = $this->actingAs($this->user)->post('/discount-configuration/save_or_update_reason', [
            'discount_reason' => 'Flat Discount',
            'type' => 'Fixed Amount',
            'value' => 500,
            'active' => 'Yes',
        ]);

        $response->assertStatus(200);
        $response->assertJson(['success' => true]);
        
        $this->assertDatabaseHas('discount_reasons', [
            'discount_reason' => 'Flat Discount',
            'discount_perc' => 0,
            'fixed_amount' => 500,
            'active' => 'Yes',
            'business_id' => $this->business->id_business,
        ]);
    }

    /** @test */
    public function can_update_discount_reason()
    {
        $reason = DiscountReason::factory()->create([
            'business_id' => $this->business->id_business,
            'discount_reason' => 'Old Reason',
            'discount_perc' => 10,
            'fixed_amount' => 0,
        ]);

        $response = $this->actingAs($this->user)->post('/discount-configuration/save_or_update_reason', [
            'id_discount_reasons' => $reason->id_discount_reasons,
            'discount_reason' => 'Updated Reason',
            'type' => 'Fixed Amount',
            'value' => 1000,
            'active' => 'No',
        ]);

        $response->assertStatus(200);
        $response->assertJson(['success' => true]);
        
        $this->assertDatabaseHas('discount_reasons', [
            'id_discount_reasons' => $reason->id_discount_reasons,
            'discount_reason' => 'Updated Reason',
            'discount_perc' => 0,
            'fixed_amount' => 1000,
            'active' => 'No',
        ]);
    }

    /** @test */
    public function discount_reason_type_derivation_logic_is_correct()
    {
        // Test Percentage type
        $response1 = $this->actingAs($this->user)->post('/discount-configuration/save_or_update_reason', [
            'discount_reason' => 'Percentage Discount',
            'type' => 'Percentage',
            'value' => 20,
            'active' => 'Yes',
        ]);
        $response1->assertStatus(200);
        
        $reason1 = DiscountReason::where('discount_reason', 'Percentage Discount')->first();
        $this->assertEquals(20, $reason1->discount_perc);
        $this->assertEquals(0, $reason1->fixed_amount);

        // Test Fixed Amount type
        $response2 = $this->actingAs($this->user)->post('/discount-configuration/save_or_update_reason', [
            'discount_reason' => 'Fixed Discount',
            'type' => 'Fixed Amount',
            'value' => 750,
            'active' => 'Yes',
        ]);
        $response2->assertStatus(200);
        
        $reason2 = DiscountReason::where('discount_reason', 'Fixed Discount')->first();
        $this->assertEquals(0, $reason2->discount_perc);
        $this->assertEquals(750, $reason2->fixed_amount);
    }

    // Discount Users Tests
    /** @test */
    public function can_create_discount_user()
    {
        // Get max ID first
        $maxId = DiscountPassword::max('id') ?? 0;
        $expectedId = $maxId + 1;

        $response = $this->actingAs($this->user)->post('/discount-configuration/save_or_update_user', [
            'name' => 'John Doe',
            'username' => 'johndoe',
            'email' => 'john@example.com',
            'new_password' => 'password123',
            'confirm_password' => 'password123',
        ]);

        $response->assertStatus(200);
        $response->assertJson(['success' => true]);
        
        $this->assertDatabaseHas('discount_password', [
            'id' => $expectedId,
            'name' => 'John Doe',
            'username' => 'johndoe',
            'email' => 'john@example.com',
            'password' => md5('password123'),
            'business_id' => $this->business->id_business,
        ]);
    }

    /** @test */
    public function can_update_discount_user_without_password_change()
    {
        $user = DiscountPassword::factory()->create([
            'business_id' => $this->business->id_business,
            'password' => md5('oldpassword'),
        ]);

        $response = $this->actingAs($this->user)->post('/discount-configuration/save_or_update_user', [
            'id' => $user->id,
            'name' => 'Updated Name',
            'username' => 'updateduser',
            'email' => 'updated@example.com',
        ]);

        $response->assertStatus(200);
        $response->assertJson(['success' => true]);
        
        $this->assertDatabaseHas('discount_password', [
            'id' => $user->id,
            'name' => 'Updated Name',
            'username' => 'updateduser',
            'email' => 'updated@example.com',
            'password' => md5('oldpassword'), // Password unchanged
        ]);
    }

    /** @test */
    public function can_update_discount_user_with_password_change()
    {
        $user = DiscountPassword::factory()->create([
            'business_id' => $this->business->id_business,
            'password' => md5('oldpassword'),
        ]);

        $response = $this->actingAs($this->user)->post('/discount-configuration/save_or_update_user', [
            'id' => $user->id,
            'name' => $user->name,
            'username' => $user->username,
            'email' => $user->email,
            'old_password' => 'oldpassword',
            'new_password' => 'newpassword123',
            'confirm_password' => 'newpassword123',
        ]);

        $response->assertStatus(200);
        $response->assertJson(['success' => true]);
        
        $this->assertDatabaseHas('discount_password', [
            'id' => $user->id,
            'password' => md5('newpassword123'),
        ]);
    }

    /** @test */
    public function password_change_requires_old_password_verification()
    {
        $user = DiscountPassword::factory()->create([
            'business_id' => $this->business->id_business,
            'password' => md5('oldpassword'),
        ]);

        $response = $this->actingAs($this->user)->post('/discount-configuration/save_or_update_user', [
            'id' => $user->id,
            'name' => $user->name,
            'username' => $user->username,
            'email' => $user->email,
            'old_password' => 'wrongpassword',
            'new_password' => 'newpassword123',
            'confirm_password' => 'newpassword123',
        ]);

        $response->assertStatus(400);
        $response->assertJson(['success' => false]);
        $response->assertJsonFragment(['message' => 'Old password is incorrect']);
    }

    /** @test */
    public function password_change_requires_new_password_confirmation_match()
    {
        $user = DiscountPassword::factory()->create([
            'business_id' => $this->business->id_business,
            'password' => md5('oldpassword'),
        ]);

        $response = $this->actingAs($this->user)->post('/discount-configuration/save_or_update_user', [
            'id' => $user->id,
            'name' => $user->name,
            'username' => $user->username,
            'email' => $user->email,
            'old_password' => 'oldpassword',
            'new_password' => 'newpassword123',
            'confirm_password' => 'differentpassword',
        ]);

        $response->assertStatus(500);
        $response->assertJson(['success' => false]);
    }

    /** @test */
    public function discount_user_password_is_hashed_with_md5()
    {
        $response = $this->actingAs($this->user)->post('/discount-configuration/save_or_update_user', [
            'name' => 'Test User',
            'username' => 'testuser',
            'email' => 'test@example.com',
            'new_password' => 'testpassword',
            'confirm_password' => 'testpassword',
        ]);

        $response->assertStatus(200);
        
        $user = DiscountPassword::where('username', 'testuser')->first();
        $this->assertEquals(md5('testpassword'), $user->password);
    }

    /** @test */
    public function discount_reasons_are_scoped_to_business()
    {
        DiscountReason::factory()->create(['business_id' => $this->business->id_business]);
        
        $otherBusiness = Business::factory()->create();
        DiscountReason::factory()->create(['business_id' => $otherBusiness->id_business]);

        $response = $this->actingAs($this->user)->get('/discount-configuration');
        $discountReasons = $response->viewData('discountReasons');
        
        $this->assertCount(1, $discountReasons);
        $this->assertEquals($this->business->id_business, $discountReasons->first()->business_id);
    }

    /** @test */
    public function discount_users_are_scoped_to_business()
    {
        DiscountPassword::factory()->create(['business_id' => $this->business->id_business]);
        
        $otherBusiness = Business::factory()->create();
        DiscountPassword::factory()->create(['business_id' => $otherBusiness->id_business]);

        $response = $this->actingAs($this->user)->get('/discount-configuration');
        $discountUsers = $response->viewData('discountUsers');
        
        $this->assertCount(1, $discountUsers);
        $this->assertEquals($this->business->id_business, $discountUsers->first()->business_id);
    }
}
