<?php

namespace Tests\Feature;

use App\Models\EyelashRecord;
use App\Models\EyelashType;
use App\Models\EyelashThickness;
use App\Models\EyelashLength;
use App\Models\EyelashCurl;
use App\Models\Business;
use App\Models\Customers;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class EyelashRecordsTest extends TestCase
{
    use RefreshDatabase;

    protected function setUp(): void
    {
        parent::setUp();
        $this->business = Business::factory()->create();
        $this->user = User::factory()->create(['business_id' => $this->business->id_business]);
        session(['business_id' => $this->business->id_business]);
    }

    /** @test */
    public function eyelash_records_page_loads_successfully()
    {
        $response = $this->actingAs($this->user)->get('/eyelash-records');
        $response->assertStatus(200);
        $response->assertViewIs('eyelash_records.list');
    }

    /** @test */
    public function eyelash_records_data_endpoint_returns_json()
    {
        $customer = Customers::factory()->create(['business_id' => $this->business->id_business]);
        EyelashRecord::factory()->create([
            'business_id' => $this->business->id_business,
            'customer_id' => $customer->id_customers,
            'customer_name' => $customer->customer_name,
        ]);

        $response = $this->actingAs($this->user)->get('/eyelash-records/data');
        $response->assertStatus(200);
        $response->assertJsonStructure(['data']);
        $this->assertNotEmpty($response->json('data'));
    }

    /** @test */
    public function eyelash_records_column_filters_work_correctly()
    {
        $customer1 = Customers::factory()->create([
            'business_id' => $this->business->id_business,
            'customer_name' => 'John Doe',
        ]);
        $customer2 = Customers::factory()->create([
            'business_id' => $this->business->id_business,
            'customer_name' => 'Jane Smith',
        ]);

        EyelashRecord::factory()->create([
            'business_id' => $this->business->id_business,
            'customer_id' => $customer1->id_customers,
            'customer_name' => 'John Doe',
            'eyelash_type' => 'Classic',
        ]);

        EyelashRecord::factory()->create([
            'business_id' => $this->business->id_business,
            'customer_id' => $customer2->id_customers,
            'customer_name' => 'Jane Smith',
            'eyelash_type' => 'Volume',
        ]);

        // Filter by customer
        $response = $this->actingAs($this->user)->get('/eyelash-records/data?filter_customer=John');
        $data = $response->json('data');
        $this->assertCount(1, $data);
        $this->assertEquals('John Doe', $data[0]['customer_name']);

        // Filter by type
        $response = $this->actingAs($this->user)->get('/eyelash-records/data?filter_type=Volume');
        $data = $response->json('data');
        $this->assertCount(1, $data);
        $this->assertEquals('Volume', $data[0]['eyelash_type']);
    }

    /** @test */
    public function can_create_new_eyelash_record()
    {
        $customer = Customers::factory()->create(['business_id' => $this->business->id_business]);

        $response = $this->actingAs($this->user)->post('/eyelash-records/save_or_update', [
            'customer_id' => $customer->id_customers,
            'customer_name' => $customer->customer_name,
            'eyelash_type' => 'Classic',
            'thickness' => '0.15',
            'length' => '12mm',
            'curl' => 'C',
            'full_set_refill' => 'Full Set',
            'date' => '2024-01-15',
            'price' => '2500',
            'remarks' => 'Test remarks',
        ]);

        $response->assertStatus(200);
        $response->assertJson(['success' => true]);
        
        $this->assertDatabaseHas('eyelashes_record', [
            'customer_name' => $customer->customer_name,
            'eyelash_type' => 'Classic',
            'thickness' => '0.15',
        ]);
    }

    /** @test */
    public function can_update_existing_eyelash_record()
    {
        $customer = Customers::factory()->create(['business_id' => $this->business->id_business]);
        $record = EyelashRecord::factory()->create([
            'business_id' => $this->business->id_business,
            'customer_id' => $customer->id_customers,
            'customer_name' => $customer->customer_name,
            'eyelash_type' => 'Classic',
        ]);

        $response = $this->actingAs($this->user)->post('/eyelash-records/save_or_update', [
            'id_eyelashes' => $record->id_eyelashes,
            'customer_id' => $customer->id_customers,
            'customer_name' => $customer->customer_name,
            'eyelash_type' => 'Volume',
            'thickness' => '0.20',
            'length' => '14mm',
            'curl' => 'D',
            'full_set_refill' => 'Refill',
            'date' => '2024-01-20',
            'price' => '3000',
            'remarks' => 'Updated remarks',
        ]);

        $response->assertStatus(200);
        $response->assertJson(['success' => true]);
        
        $this->assertDatabaseHas('eyelashes_record', [
            'id_eyelashes' => $record->id_eyelashes,
            'eyelash_type' => 'Volume',
            'thickness' => '0.20',
        ]);
    }

    /** @test */
    public function can_save_eyelash_type_configuration()
    {
        $response = $this->actingAs($this->user)->post('/eyelash-records/save-type', [
            'eyelash_type' => 'Hybrid',
        ]);

        $response->assertStatus(200);
        $response->assertJson(['success' => true]);
        
        $this->assertDatabaseHas('eyelash_types', [
            'eyelash_type' => 'Hybrid',
            'business_id' => $this->business->id_business,
        ]);
    }

    /** @test */
    public function can_update_eyelash_type_configuration()
    {
        $type = EyelashType::factory()->create([
            'business_id' => $this->business->id_business,
            'eyelash_type' => 'Classic',
        ]);

        $response = $this->actingAs($this->user)->post('/eyelash-records/save-type', [
            'id_eyelash_types' => $type->id_eyelash_types,
            'eyelash_type' => 'Updated Classic',
        ]);

        $response->assertStatus(200);
        $response->assertJson(['success' => true]);
        
        $this->assertDatabaseHas('eyelash_types', [
            'id_eyelash_types' => $type->id_eyelash_types,
            'eyelash_type' => 'Updated Classic',
        ]);
    }

    /** @test */
    public function can_delete_eyelash_type_configuration()
    {
        $type = EyelashType::factory()->create([
            'business_id' => $this->business->id_business,
        ]);

        $response = $this->actingAs($this->user)->delete('/eyelash-records/delete-type/' . $type->id_eyelash_types);

        $response->assertStatus(200);
        $response->assertJson(['success' => true]);
        
        $this->assertDatabaseMissing('eyelash_types', [
            'id_eyelash_types' => $type->id_eyelash_types,
        ]);
    }

    /** @test */
    public function can_save_eyelash_thickness_configuration()
    {
        $response = $this->actingAs($this->user)->post('/eyelash-records/save-thickness', [
            'thickness' => '0.18',
        ]);

        $response->assertStatus(200);
        $response->assertJson(['success' => true]);
        
        $this->assertDatabaseHas('eyelash_thickness', [
            'thickness' => '0.18',
            'business_id' => $this->business->id_business,
        ]);
    }

    /** @test */
    public function can_save_eyelash_length_configuration()
    {
        $response = $this->actingAs($this->user)->post('/eyelash-records/save-length', [
            'length' => '16mm',
        ]);

        $response->assertStatus(200);
        $response->assertJson(['success' => true]);
        
        $this->assertDatabaseHas('eyelash_length', [
            'length' => '16mm',
            'business_id' => $this->business->id_business,
        ]);
    }

    /** @test */
    public function can_save_eyelash_curl_configuration()
    {
        $response = $this->actingAs($this->user)->post('/eyelash-records/save-curl', [
            'curl' => 'L',
        ]);

        $response->assertStatus(200);
        $response->assertJson(['success' => true]);
        
        $this->assertDatabaseHas('eyelash_curl', [
            'curl' => 'L',
            'business_id' => $this->business->id_business,
        ]);
    }

    /** @test */
    public function can_get_configuration_data()
    {
        EyelashType::factory()->create(['business_id' => $this->business->id_business]);
        EyelashThickness::factory()->create(['business_id' => $this->business->id_business]);
        EyelashLength::factory()->create(['business_id' => $this->business->id_business]);
        EyelashCurl::factory()->create(['business_id' => $this->business->id_business]);

        $response = $this->actingAs($this->user)->get('/eyelash-records/configuration-data');
        $response->assertStatus(200);
        $response->assertJsonStructure([
            'types',
            'thicknesses',
            'lengths',
            'curls',
        ]);
    }

    /** @test */
    public function eyelash_records_are_scoped_to_business()
    {
        $otherBusiness = Business::factory()->create();
        $customer = Customers::factory()->create(['business_id' => $this->business->id_business]);
        
        EyelashRecord::factory()->create([
            'business_id' => $this->business->id_business,
            'customer_id' => $customer->id_customers,
            'customer_name' => $customer->customer_name,
        ]);

        EyelashRecord::factory()->create([
            'business_id' => $otherBusiness->id_business,
            'customer_id' => $customer->id_customers,
            'customer_name' => $customer->customer_name,
        ]);

        $response = $this->actingAs($this->user)->get('/eyelash-records/data');
        $data = $response->json('data');

        // Should only return records for current business
        foreach ($data as $record) {
            $this->assertEquals($this->business->id_business, $record['business_id']);
        }
    }

    /** @test */
    public function eyelash_record_validation_requires_customer_name()
    {
        $response = $this->actingAs($this->user)->post('/eyelash-records/save_or_update', [
            'eyelash_type' => 'Classic',
        ]);

        $response->assertStatus(500);
        $response->assertJson(['success' => false]);
    }

    /** @test */
    public function eyelash_record_validation_requires_eyelash_type()
    {
        $customer = Customers::factory()->create(['business_id' => $this->business->id_business]);

        $response = $this->actingAs($this->user)->post('/eyelash-records/save_or_update', [
            'customer_name' => $customer->customer_name,
        ]);

        $response->assertStatus(500);
        $response->assertJson(['success' => false]);
    }
}
