<?php

namespace Tests\Feature;

use App\Models\FacialRecord;
use App\Models\Business;
use App\Models\Customers;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class FacialRecordsTest extends TestCase
{
    use RefreshDatabase;

    protected function setUp(): void
    {
        parent::setUp();
        $this->business = Business::factory()->create();
        $this->user = User::factory()->create(['business_id' => $this->business->id_business]);
        session(['business_id' => $this->business->id_business]);
    }

    /** @test */
    public function facial_records_page_loads_successfully()
    {
        $response = $this->actingAs($this->user)->get('/facial-records');
        $response->assertStatus(200);
        $response->assertViewIs('facial_records.list');
    }

    /** @test */
    public function facial_records_data_endpoint_returns_json()
    {
        $customer = Customers::factory()->create(['business_id' => $this->business->id_business]);
        FacialRecord::factory()->create([
            'business_id' => $this->business->id_business,
            'customer_id' => $customer->id_customers,
            'customer' => $customer->customer_name,
            'time' => '90', // 90 minutes
        ]);

        $response = $this->actingAs($this->user)->get('/facial-records/data');
        $response->assertStatus(200);
        $response->assertJsonStructure(['data']);
        
        $data = $response->json('data');
        $this->assertNotEmpty($data);
        $this->assertArrayHasKey('formatted_time', $data[0]);
        $this->assertEquals('1:30:00', $data[0]['formatted_time']); // 90 minutes = 1:30:00
    }

    /** @test */
    public function facial_records_time_conversion_displays_correctly()
    {
        $customer = Customers::factory()->create(['business_id' => $this->business->id_business]);
        
        // Test 45 minutes -> 0:45:00
        $record1 = FacialRecord::factory()->create([
            'business_id' => $this->business->id_business,
            'customer_id' => $customer->id_customers,
            'customer' => $customer->customer_name,
            'time' => '45',
        ]);

        // Test 90 minutes -> 1:30:00
        $record2 = FacialRecord::factory()->create([
            'business_id' => $this->business->id_business,
            'customer_id' => $customer->id_customers,
            'customer' => $customer->customer_name,
            'time' => '90',
        ]);

        // Test 120 minutes -> 2:00:00
        $record3 = FacialRecord::factory()->create([
            'business_id' => $this->business->id_business,
            'customer_id' => $customer->id_customers,
            'customer' => $customer->customer_name,
            'time' => '120',
        ]);

        $response = $this->actingAs($this->user)->get('/facial-records/data');
        $data = $response->json('data');

        $record1Data = collect($data)->firstWhere('id', $record1->id);
        $record2Data = collect($data)->firstWhere('id', $record2->id);
        $record3Data = collect($data)->firstWhere('id', $record3->id);

        $this->assertEquals('0:45:00', $record1Data['formatted_time']);
        $this->assertEquals('1:30:00', $record2Data['formatted_time']);
        $this->assertEquals('2:00:00', $record3Data['formatted_time']);
    }

    /** @test */
    public function facial_records_column_filters_work_correctly()
    {
        $customer1 = Customers::factory()->create([
            'business_id' => $this->business->id_business,
            'customer_name' => 'John Doe',
        ]);
        $customer2 = Customers::factory()->create([
            'business_id' => $this->business->id_business,
            'customer_name' => 'Jane Smith',
        ]);

        FacialRecord::factory()->create([
            'business_id' => $this->business->id_business,
            'customer_id' => $customer1->id_customers,
            'customer' => 'John Doe',
            'facial' => 'Deep Cleansing',
        ]);

        FacialRecord::factory()->create([
            'business_id' => $this->business->id_business,
            'customer_id' => $customer2->id_customers,
            'customer' => 'Jane Smith',
            'facial' => 'Hydrating Facial',
        ]);

        // Filter by customer
        $response = $this->actingAs($this->user)->get('/facial-records/data?filter_customer=John');
        $data = $response->json('data');
        $this->assertCount(1, $data);
        $this->assertEquals('John Doe', $data[0]['customer']);

        // Filter by facial
        $response = $this->actingAs($this->user)->get('/facial-records/data?filter_facial=Hydrating');
        $data = $response->json('data');
        $this->assertCount(1, $data);
        $this->assertEquals('Hydrating Facial', $data[0]['facial']);
    }

    /** @test */
    public function can_create_new_facial_record()
    {
        $customer = Customers::factory()->create(['business_id' => $this->business->id_business]);

        $response = $this->actingAs($this->user)->post('/facial-records/save_or_update', [
            'customer_id' => $customer->id_customers,
            'customer' => $customer->customer_name,
            'facial' => 'Test Facial',
            'exfoliant' => 'Test Exfoliant',
            'mask' => 'Test Mask',
            'cleanser' => 'Test Cleanser',
            'date' => '2024-01-15',
            'time_minutes' => 60,
            'charge' => 1500,
            'remarks' => 'Test remarks',
        ]);

        $response->assertStatus(200);
        $response->assertJson(['success' => true]);
        
        $this->assertDatabaseHas('facial_records', [
            'customer' => $customer->customer_name,
            'facial' => 'Test Facial',
            'time' => '60', // Stored as minutes string
            'charge' => '1500',
        ]);
    }

    /** @test */
    public function can_update_existing_facial_record()
    {
        $customer = Customers::factory()->create(['business_id' => $this->business->id_business]);
        $record = FacialRecord::factory()->create([
            'business_id' => $this->business->id_business,
            'customer_id' => $customer->id_customers,
            'customer' => $customer->customer_name,
            'facial' => 'Old Facial',
            'time' => '45',
        ]);

        $response = $this->actingAs($this->user)->post('/facial-records/save_or_update', [
            'id' => $record->id,
            'customer_id' => $customer->id_customers,
            'customer' => $customer->customer_name,
            'facial' => 'Updated Facial',
            'exfoliant' => 'Updated Exfoliant',
            'mask' => 'Updated Mask',
            'cleanser' => 'Updated Cleanser',
            'date' => '2024-01-20',
            'time_minutes' => 90,
            'charge' => 2000,
            'remarks' => 'Updated remarks',
        ]);

        $response->assertStatus(200);
        $response->assertJson(['success' => true]);
        
        $this->assertDatabaseHas('facial_records', [
            'id' => $record->id,
            'facial' => 'Updated Facial',
            'time' => '90',
            'charge' => '2000',
        ]);
    }

    /** @test */
    public function facial_record_validation_requires_customer()
    {
        $response = $this->actingAs($this->user)->post('/facial-records/save_or_update', [
            'facial' => 'Test Facial',
            'time_minutes' => 60,
        ]);

        $response->assertStatus(500);
        $response->assertJson(['success' => false]);
    }

    /** @test */
    public function facial_record_validation_requires_time_minutes()
    {
        $customer = Customers::factory()->create(['business_id' => $this->business->id_business]);

        $response = $this->actingAs($this->user)->post('/facial-records/save_or_update', [
            'customer' => $customer->customer_name,
            'facial' => 'Test Facial',
        ]);

        $response->assertStatus(500);
        $response->assertJson(['success' => false]);
    }

    /** @test */
    public function facial_record_time_minutes_must_be_non_negative()
    {
        $customer = Customers::factory()->create(['business_id' => $this->business->id_business]);

        $response = $this->actingAs($this->user)->post('/facial-records/save_or_update', [
            'customer' => $customer->customer_name,
            'facial' => 'Test Facial',
            'time_minutes' => -10,
        ]);

        $response->assertStatus(500);
        $response->assertJson(['success' => false]);
    }

    /** @test */
    public function facial_records_are_scoped_to_business()
    {
        $otherBusiness = Business::factory()->create();
        $customer = Customers::factory()->create(['business_id' => $this->business->id_business]);
        
        FacialRecord::factory()->create([
            'business_id' => $this->business->id_business,
            'customer_id' => $customer->id_customers,
            'customer' => $customer->customer_name,
        ]);

        FacialRecord::factory()->create([
            'business_id' => $otherBusiness->id_business,
            'customer_id' => $customer->id_customers,
            'customer' => $customer->customer_name,
        ]);

        $response = $this->actingAs($this->user)->get('/facial-records/data');
        $data = $response->json('data');

        // Should only return records for current business
        foreach ($data as $record) {
            $this->assertEquals($this->business->id_business, $record['business_id']);
        }
    }

    /** @test */
    public function facial_records_auto_increment_id_on_create()
    {
        $customer = Customers::factory()->create(['business_id' => $this->business->id_business]);

        // Create first record
        $response1 = $this->actingAs($this->user)->post('/facial-records/save_or_update', [
            'customer_id' => $customer->id_customers,
            'customer' => $customer->customer_name,
            'time_minutes' => 60,
        ]);

        $record1 = FacialRecord::latest('id')->first();
        $firstId = $record1->id;

        // Create second record
        $response2 = $this->actingAs($this->user)->post('/facial-records/save_or_update', [
            'customer_id' => $customer->id_customers,
            'customer' => $customer->customer_name,
            'time_minutes' => 90,
        ]);

        $record2 = FacialRecord::latest('id')->first();
        $secondId = $record2->id;

        // Second ID should be sequential (next auto-increment)
        $this->assertEquals($firstId + 1, $secondId);
    }
}
