<?php

namespace Tests\Feature;

use App\Models\FacialRecord;
use App\Models\EyelashRecord;
use App\Models\Department;
use App\Models\Floor;
use App\Models\BlockEvent;
use App\Models\DiscountReason;
use App\Models\DiscountPassword;
use App\Models\Business;
use App\Models\Customers;
use App\Models\User;
use App\Models\Staff;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

/**
 * Tests for Modal Behavior across all pages
 * 
 * This test suite verifies that:
 * 1. Save endpoints return success responses with correct structure
 * 2. Table reload endpoints exist and return data
 * 3. Backend supports the frontend modal auto-close and table reload behavior
 */
class ModalBehaviorTest extends TestCase
{
    use RefreshDatabase;

    protected function setUp(): void
    {
        parent::setUp();
        $this->business = Business::factory()->create();
        $this->user = User::factory()->create(['business_id' => $this->business->id_business]);
        session(['business_id' => $this->business->id_business]);
        session(['common_services' => 'No']);
    }

    // Facial Records Modal Behavior
    /** @test */
    public function facial_records_save_returns_success_response()
    {
        $customer = Customers::factory()->create(['business_id' => $this->business->id_business]);

        $response = $this->actingAs($this->user)->post('/facial-records/save_or_update', [
            'customer_id' => $customer->id_customers,
            'customer' => $customer->customer_name,
            'time_minutes' => 60,
        ]);

        $response->assertStatus(200);
        $response->assertJson(['success' => true]);
        $response->assertJsonStructure(['success', 'message', 'record']);
    }

    /** @test */
    public function facial_records_table_reload_endpoint_exists()
    {
        FacialRecord::factory()->create(['business_id' => $this->business->id_business]);

        $response = $this->actingAs($this->user)->get('/facial-records/data');
        $response->assertStatus(200);
        $response->assertJsonStructure(['data']);
    }

    // Eyelash Records Modal Behavior
    /** @test */
    public function eyelash_records_save_returns_success_response()
    {
        $customer = Customers::factory()->create(['business_id' => $this->business->id_business]);

        $response = $this->actingAs($this->user)->post('/eyelash-records/save_or_update', [
            'customer_id' => $customer->id_customers,
            'customer_name' => $customer->customer_name,
            'eyelash_type' => 'Classic',
        ]);

        $response->assertStatus(200);
        $response->assertJson(['success' => true]);
        $response->assertJsonStructure(['success', 'message', 'record']);
    }

    /** @test */
    public function eyelash_records_table_reload_endpoint_exists()
    {
        EyelashRecord::factory()->create(['business_id' => $this->business->id_business]);

        $response = $this->actingAs($this->user)->get('/eyelash-records/data');
        $response->assertStatus(200);
        $response->assertJsonStructure(['data']);
    }

    // Service Departments Modal Behavior
    /** @test */
    public function service_departments_save_returns_success_response()
    {
        $response = $this->actingAs($this->user)->post('/service_departments/save_or_update', [
            'service_department' => 'Test Department',
            'department_status' => 'Active',
        ]);

        $response->assertStatus(200);
        $response->assertJson(['success' => true]);
        $response->assertJsonStructure(['success', 'message', 'department']);
    }

    /** @test */
    public function service_departments_table_reload_endpoint_exists()
    {
        Department::factory()->create(['business_id' => $this->business->id_business]);

        $response = $this->actingAs($this->user)->get('/service_departments');
        $response->assertStatus(200);
        $response->assertViewHas('departments');
    }

    // Service Floors Modal Behavior
    /** @test */
    public function service_floors_save_returns_success_response()
    {
        $staff = Staff::factory()->create(['business_id' => $this->business->id_business]);

        $response = $this->actingAs($this->user)->post('/service_floors/save_or_update', [
            'floor_name' => 'Test Floor',
            'floor_workstations' => 5,
            'floormanager_id' => $staff->id_staff,
            'floor_status' => 'Active',
        ]);

        $response->assertStatus(200);
        $response->assertJson(['success' => true]);
        $response->assertJsonStructure(['success', 'message', 'floor']);
    }

    /** @test */
    public function service_floors_table_reload_endpoint_exists()
    {
        Floor::factory()->create(['business_id' => $this->business->id_business]);

        $response = $this->actingAs($this->user)->get('/service_floors');
        $response->assertStatus(200);
        $response->assertViewHas('floors');
    }

    // Block Events Modal Behavior
    /** @test */
    public function block_events_save_returns_success_response()
    {
        $response = $this->actingAs($this->user)->post('/block_events/save_or_update', [
            'block_event_name' => 'Test Event',
            'block_event_desc' => 'Test Description',
        ]);

        $response->assertStatus(200);
        $response->assertJson(['success' => true]);
        $response->assertJsonStructure(['success', 'message', 'block_event']);
    }

    /** @test */
    public function block_events_table_reload_endpoint_exists()
    {
        BlockEvent::factory()->create(['business_id' => $this->business->id_business]);

        $response = $this->actingAs($this->user)->get('/block_events');
        $response->assertStatus(200);
        $response->assertViewHas('blockEvents');
    }

    // Discount Config Modal Behavior
    /** @test */
    public function discount_reasons_save_returns_success_response()
    {
        $response = $this->actingAs($this->user)->post('/discount-configuration/save_or_update_reason', [
            'discount_reason' => 'Test Reason',
            'type' => 'Percentage',
            'value' => 10,
            'active' => 'Yes',
        ]);

        $response->assertStatus(200);
        $response->assertJson(['success' => true]);
        $response->assertJsonStructure(['success', 'message', 'reason']);
    }

    /** @test */
    public function discount_users_save_returns_success_response()
    {
        $response = $this->actingAs($this->user)->post('/discount-configuration/save_or_update_user', [
            'name' => 'Test User',
            'username' => 'testuser',
            'email' => 'test@example.com',
            'new_password' => 'password123',
            'confirm_password' => 'password123',
        ]);

        $response->assertStatus(200);
        $response->assertJson(['success' => true]);
        $response->assertJsonStructure(['success', 'message', 'user']);
    }

    /** @test */
    public function discount_config_table_reload_endpoint_exists()
    {
        DiscountReason::factory()->create(['business_id' => $this->business->id_business]);
        DiscountPassword::factory()->create(['business_id' => $this->business->id_business]);

        $response = $this->actingAs($this->user)->get('/discount-configuration');
        $response->assertStatus(200);
        $response->assertViewHas('discountReasons');
        $response->assertViewHas('discountUsers');
    }

    // Error Handling - Modal Should Remain Open on Failure
    /** @test */
    public function facial_records_save_returns_error_on_validation_failure()
    {
        $response = $this->actingAs($this->user)->post('/facial-records/save_or_update', [
            // Missing required fields
        ]);

        $response->assertStatus(500);
        $response->assertJson(['success' => false]);
        // Modal should remain open (handled by frontend)
    }

    /** @test */
    public function eyelash_records_save_returns_error_on_validation_failure()
    {
        $response = $this->actingAs($this->user)->post('/eyelash-records/save_or_update', [
            // Missing required fields
        ]);

        $response->assertStatus(500);
        $response->assertJson(['success' => false]);
        // Modal should remain open (handled by frontend)
    }

    /** @test */
    public function service_departments_save_returns_error_on_validation_failure()
    {
        $response = $this->actingAs($this->user)->post('/service_departments/save_or_update', [
            // Missing required fields
        ]);

        $response->assertStatus(500);
        $response->assertJson(['success' => false]);
        // Modal should remain open (handled by frontend)
    }

    /** @test */
    public function discount_reasons_save_returns_error_on_validation_failure()
    {
        $response = $this->actingAs($this->user)->post('/discount-configuration/save_or_update_reason', [
            // Missing required fields
        ]);

        $response->assertStatus(500);
        $response->assertJson(['success' => false]);
        // Modal should remain open (handled by frontend)
    }

    /** @test */
    public function discount_users_save_returns_error_on_validation_failure()
    {
        $response = $this->actingAs($this->user)->post('/discount-configuration/save_or_update_user', [
            // Missing required fields
        ]);

        $response->assertStatus(500);
        $response->assertJson(['success' => false]);
        // Modal should remain open (handled by frontend)
    }

    // Verify Success Response Structure for Modal Closing
    /** @test */
    public function all_save_endpoints_return_consistent_success_structure()
    {
        $customer = Customers::factory()->create(['business_id' => $this->business->id_business]);
        $staff = Staff::factory()->create(['business_id' => $this->business->id_business]);

        // Test all save endpoints return consistent structure
        $endpoints = [
            [
                'url' => '/facial-records/save_or_update',
                'data' => [
                    'customer_id' => $customer->id_customers,
                    'customer' => $customer->customer_name,
                    'time_minutes' => 60,
                ],
            ],
            [
                'url' => '/eyelash-records/save_or_update',
                'data' => [
                    'customer_id' => $customer->id_customers,
                    'customer_name' => $customer->customer_name,
                    'eyelash_type' => 'Classic',
                ],
            ],
            [
                'url' => '/service_departments/save_or_update',
                'data' => [
                    'service_department' => 'Test',
                    'department_status' => 'Active',
                ],
            ],
            [
                'url' => '/block_events/save_or_update',
                'data' => [
                    'block_event_name' => 'Test',
                    'block_event_desc' => 'Test',
                ],
            ],
        ];

        foreach ($endpoints as $endpoint) {
            $response = $this->actingAs($this->user)->post($endpoint['url'], $endpoint['data']);
            $response->assertStatus(200);
            $response->assertJson(['success' => true]);
            $response->assertJsonHasKey('message');
            // Frontend can use 'success' key to determine if modal should close
        }
    }
}
