<?php

namespace Tests\Feature;

use App\Models\SMSLog;
use App\Models\Business;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class SMSLogTest extends TestCase
{
    use RefreshDatabase;

    protected function setUp(): void
    {
        parent::setUp();
        $this->business = Business::factory()->create();
        $this->user = User::factory()->create(['business_id' => $this->business->id_business]);
        session(['business_id' => $this->business->id_business]);
    }

    /** @test */
    public function sms_log_page_loads_successfully()
    {
        $response = $this->actingAs($this->user)->get('/sms-log');
        $response->assertStatus(200);
        $response->assertViewIs('sms.sms_log');
    }

    /** @test */
    public function sms_log_data_endpoint_returns_datatables_format()
    {
        SMSLog::factory()->count(5)->create(['business_id' => $this->business->id_business]);

        $response = $this->actingAs($this->user)->get('/sms-log/data', [
            'draw' => 1,
            'start' => 0,
            'length' => 10,
        ]);

        $response->assertStatus(200);
        $response->assertJsonStructure([
            'draw',
            'recordsTotal',
            'recordsFiltered',
            'data' => [
                '*' => ['id_sms_log', 'msgdata', 'receiver', 'senton', 'using']
            ]
        ]);
    }

    /** @test */
    public function sms_log_filter_by_id()
    {
        $log1 = SMSLog::factory()->create([
            'business_id' => $this->business->id_business,
            'id_sms_log' => 100,
        ]);
        SMSLog::factory()->create([
            'business_id' => $this->business->id_business,
            'id_sms_log' => 200,
        ]);

        $response = $this->actingAs($this->user)->get('/sms-log/data', [
            'draw' => 1,
            'start' => 0,
            'length' => 10,
            'filter_id' => '100',
        ]);

        $data = $response->json('data');
        $this->assertCount(1, $data);
        $this->assertEquals(100, $data[0]['id_sms_log']);
    }

    /** @test */
    public function sms_log_filter_by_message()
    {
        SMSLog::factory()->create([
            'business_id' => $this->business->id_business,
            'msgdata' => 'Hello World',
        ]);
        SMSLog::factory()->create([
            'business_id' => $this->business->id_business,
            'msgdata' => 'Goodbye World',
        ]);

        $response = $this->actingAs($this->user)->get('/sms-log/data', [
            'draw' => 1,
            'start' => 0,
            'length' => 10,
            'filter_message' => 'Hello',
        ]);

        $data = $response->json('data');
        $this->assertCount(1, $data);
        $this->assertStringContainsString('Hello', $data[0]['msgdata']);
    }

    /** @test */
    public function sms_log_filter_by_receiver()
    {
        SMSLog::factory()->create([
            'business_id' => $this->business->id_business,
            'receiver' => '923001234567',
        ]);
        SMSLog::factory()->create([
            'business_id' => $this->business->id_business,
            'receiver' => '923009876543',
        ]);

        $response = $this->actingAs($this->user)->get('/sms-log/data', [
            'draw' => 1,
            'start' => 0,
            'length' => 10,
            'filter_receiver' => '3001',
        ]);

        $data = $response->json('data');
        $this->assertCount(1, $data);
        $this->assertStringContainsString('3001', $data[0]['receiver']);
    }

    /** @test */
    public function sms_log_filter_by_date()
    {
        $date = '2024-01-15';
        SMSLog::factory()->create([
            'business_id' => $this->business->id_business,
            'sent_date' => $date,
        ]);
        SMSLog::factory()->create([
            'business_id' => $this->business->id_business,
            'sent_date' => '2024-01-20',
        ]);

        $response = $this->actingAs($this->user)->get('/sms-log/data', [
            'draw' => 1,
            'start' => 0,
            'length' => 10,
            'filter_date' => $date,
        ]);

        $data = $response->json('data');
        $this->assertCount(1, $data);
    }

    /** @test */
    public function sms_log_filter_by_using()
    {
        SMSLog::factory()->create([
            'business_id' => $this->business->id_business,
            'using' => 'System',
        ]);
        SMSLog::factory()->create([
            'business_id' => $this->business->id_business,
            'using' => 'Manual',
        ]);

        $response = $this->actingAs($this->user)->get('/sms-log/data', [
            'draw' => 1,
            'start' => 0,
            'length' => 10,
            'filter_using' => 'System',
        ]);

        $data = $response->json('data');
        $this->assertCount(1, $data);
        $this->assertEquals('System', $data[0]['using']);
    }

    /** @test */
    public function sms_log_combined_filters_work_together()
    {
        SMSLog::factory()->create([
            'business_id' => $this->business->id_business,
            'msgdata' => 'Test Message',
            'receiver' => '923001234567',
            'using' => 'System',
        ]);
        SMSLog::factory()->create([
            'business_id' => $this->business->id_business,
            'msgdata' => 'Other Message',
            'receiver' => '923001234567',
            'using' => 'Manual',
        ]);

        $response = $this->actingAs($this->user)->get('/sms-log/data', [
            'draw' => 1,
            'start' => 0,
            'length' => 10,
            'filter_message' => 'Test',
            'filter_receiver' => '3001',
            'filter_using' => 'System',
        ]);

        $data = $response->json('data');
        $this->assertCount(1, $data);
        $this->assertStringContainsString('Test', $data[0]['msgdata']);
        $this->assertEquals('System', $data[0]['using']);
    }

    /** @test */
    public function sms_log_using_options_endpoint_returns_distinct_values()
    {
        SMSLog::factory()->create([
            'business_id' => $this->business->id_business,
            'using' => 'System',
        ]);
        SMSLog::factory()->create([
            'business_id' => $this->business->id_business,
            'using' => 'Manual',
        ]);
        SMSLog::factory()->create([
            'business_id' => $this->business->id_business,
            'using' => 'System', // Duplicate
        ]);

        $response = $this->actingAs($this->user)->get('/sms-log/using-options');
        $response->assertStatus(200);
        
        $options = $response->json();
        $this->assertCount(2, $options);
        $this->assertContains('System', $options);
        $this->assertContains('Manual', $options);
    }

    /** @test */
    public function sms_log_resend_requires_valid_id()
    {
        $log = SMSLog::factory()->create([
            'business_id' => $this->business->id_business,
        ]);

        $response = $this->actingAs($this->user)->post('/sms-log/resend', [
            'id' => $log->id_sms_log,
        ]);

        // May fail due to SMS service, but endpoint exists and validates
        $this->assertContains($response->status(), [200, 500]);
    }

    /** @test */
    public function sms_log_resend_validates_id_exists()
    {
        $response = $this->actingAs($this->user)->post('/sms-log/resend', [
            'id' => 99999, // Non-existent
        ]);

        $response->assertStatus(404);
        $response->assertJson(['success' => false]);
    }

    /** @test */
    public function sms_log_is_scoped_to_business()
    {
        SMSLog::factory()->create(['business_id' => $this->business->id_business]);
        
        $otherBusiness = Business::factory()->create();
        SMSLog::factory()->create(['business_id' => $otherBusiness->id_business]);

        $response = $this->actingAs($this->user)->get('/sms-log/data', [
            'draw' => 1,
            'start' => 0,
            'length' => 10,
        ]);

        $data = $response->json('data');
        $this->assertCount(1, $data);
        $this->assertEquals($this->business->id_business, SMSLog::find($data[0]['id_sms_log'])->business_id);
    }
}
