<?php

namespace Tests\Feature;

use App\Models\Business;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Support\Facades\DB;
use Tests\TestCase;

class ScheduledTasksTest extends TestCase
{
    use RefreshDatabase;

    protected function setUp(): void
    {
        parent::setUp();
        $this->business = Business::factory()->create();
        $this->user = User::factory()->create(['business_id' => $this->business->id_business]);
        session(['business_id' => $this->business->id_business]);
    }

    /** @test */
    public function scheduled_tasks_page_loads_successfully()
    {
        $response = $this->actingAs($this->user)->get('/scheduled-tasks');
        $response->assertStatus(200);
        $response->assertViewIs('sms.scheduled_tasks');
    }

    /** @test */
    public function scheduled_tasks_list_endpoint_returns_distinct_task_names()
    {
        // Insert test data directly into scheduled_tasks table
        DB::table('scheduled_tasks')->insert([
            [
                'business_id' => $this->business->id_business,
                'task_name' => 'Birthday Reminder',
                'status' => 'enabled',
                'msg' => 'Happy Birthday!',
                's_query' => 'SELECT * FROM customers WHERE DAY(birthday) = DAY(CURDATE())',
            ],
            [
                'business_id' => $this->business->id_business,
                'task_name' => 'Birthday Reminder', // Duplicate name
                'status' => 'enabled',
                'msg' => 'Happy Birthday!',
                's_query' => 'SELECT * FROM customers WHERE DAY(birthday) = DAY(CURDATE())',
            ],
            [
                'business_id' => $this->business->id_business,
                'task_name' => 'Appointment Reminder',
                'status' => 'enabled',
                'msg' => 'Your appointment is tomorrow',
                's_query' => 'SELECT * FROM appointments WHERE date = DATE_ADD(CURDATE(), INTERVAL 1 DAY)',
            ],
            [
                'business_id' => $this->business->id_business,
                'task_name' => 'Disabled Task',
                'status' => 'disabled',
                'msg' => 'This should not appear',
                's_query' => 'SELECT * FROM customers',
            ],
        ]);

        $response = $this->actingAs($this->user)->get('/scheduled-tasks/list');
        $response->assertStatus(200);
        
        $tasks = $response->json();
        $this->assertIsArray($tasks);
        
        // Should return distinct task names with status 'enabled' and non-null task_name
        $taskNames = collect($tasks)->pluck('task_name')->toArray();
        $this->assertContains('Birthday Reminder', $taskNames);
        $this->assertContains('Appointment Reminder', $taskNames);
        $this->assertNotContains('Disabled Task', $taskNames);
        
        // Should be distinct
        $this->assertCount(2, array_unique($taskNames));
    }

    /** @test */
    public function scheduled_tasks_list_only_includes_enabled_tasks()
    {
        DB::table('scheduled_tasks')->insert([
            [
                'business_id' => $this->business->id_business,
                'task_name' => 'Enabled Task',
                'status' => 'enabled',
                'msg' => 'Test',
                's_query' => 'SELECT * FROM customers',
            ],
            [
                'business_id' => $this->business->id_business,
                'task_name' => 'Disabled Task',
                'status' => 'disabled',
                'msg' => 'Test',
                's_query' => 'SELECT * FROM customers',
            ],
        ]);

        $response = $this->actingAs($this->user)->get('/scheduled-tasks/list');
        $tasks = $response->json();
        
        $taskNames = collect($tasks)->pluck('task_name')->toArray();
        $this->assertContains('Enabled Task', $taskNames);
        $this->assertNotContains('Disabled Task', $taskNames);
    }

    /** @test */
    public function scheduled_tasks_list_excludes_null_task_names()
    {
        DB::table('scheduled_tasks')->insert([
            [
                'business_id' => $this->business->id_business,
                'task_name' => 'Valid Task',
                'status' => 'enabled',
                'msg' => 'Test',
                's_query' => 'SELECT * FROM customers',
            ],
            [
                'business_id' => $this->business->id_business,
                'task_name' => null,
                'status' => 'enabled',
                'msg' => 'Test',
                's_query' => 'SELECT * FROM customers',
            ],
        ]);

        $response = $this->actingAs($this->user)->get('/scheduled-tasks/list');
        $tasks = $response->json();
        
        $taskNames = collect($tasks)->pluck('task_name')->toArray();
        $this->assertContains('Valid Task', $taskNames);
        $this->assertNotContains(null, $taskNames);
    }

    /** @test */
    public function scheduled_task_details_endpoint_returns_msg_and_day_difference()
    {
        $taskId = DB::table('scheduled_tasks')->insertGetId([
            'business_id' => $this->business->id_business,
            'task_name' => 'Test Task',
            'status' => 'enabled',
            'msg' => 'Test message',
            'day_difference' => 7,
            's_query' => 'SELECT * FROM customers',
        ]);

        $response = $this->actingAs($this->user)->get('/scheduled-tasks/details?id=' . $taskId);
        $response->assertStatus(200);
        
        $data = $response->json();
        $this->assertArrayHasKey('msg', $data);
        $this->assertArrayHasKey('day_difference', $data);
        $this->assertEquals('Test message', $data['msg']);
        $this->assertEquals(7, $data['day_difference']);
    }

    /** @test */
    public function scheduled_task_details_returns_404_for_nonexistent_task()
    {
        $response = $this->actingAs($this->user)->get('/scheduled-tasks/details?id=99999');
        $response->assertStatus(404);
        $response->assertJson(['error' => 'Task not found']);
    }

    /** @test */
    public function scheduled_task_query_endpoint_returns_s_query()
    {
        $taskId = DB::table('scheduled_tasks')->insertGetId([
            'business_id' => $this->business->id_business,
            'task_name' => 'Test Task',
            'status' => 'enabled',
            'msg' => 'Test',
            's_query' => 'SELECT id, name FROM customers WHERE active = 1',
        ]);

        $response = $this->actingAs($this->user)->get('/scheduled-tasks/query?id=' . $taskId);
        $response->assertStatus(200);
        
        $data = $response->json();
        $this->assertArrayHasKey('s_query', $data);
        $this->assertEquals('SELECT id, name FROM customers WHERE active = 1', $data['s_query']);
    }

    /** @test */
    public function scheduled_task_query_returns_404_for_nonexistent_task()
    {
        $response = $this->actingAs($this->user)->get('/scheduled-tasks/query?id=99999');
        $response->assertStatus(404);
        $response->assertJson(['error' => 'Query not found']);
    }

    /** @test */
    public function scheduled_task_query_returns_404_for_empty_query()
    {
        $taskId = DB::table('scheduled_tasks')->insertGetId([
            'business_id' => $this->business->id_business,
            'task_name' => 'Test Task',
            'status' => 'enabled',
            'msg' => 'Test',
            's_query' => '', // Empty query
        ]);

        $response = $this->actingAs($this->user)->get('/scheduled-tasks/query?id=' . $taskId);
        $response->assertStatus(404);
        $response->assertJson(['error' => 'Query not found']);
    }

    /** @test */
    public function scheduled_task_query_validation_requires_id()
    {
        $response = $this->actingAs($this->user)->get('/scheduled-tasks/query');
        $response->assertStatus(500); // Validation error
    }

    /** @test */
    public function scheduled_task_details_validation_requires_id()
    {
        $response = $this->actingAs($this->user)->get('/scheduled-tasks/details');
        $response->assertStatus(500); // Validation error
    }

    /** @test */
    public function scheduled_task_query_can_execute_dynamic_query()
    {
        // This test verifies that the query endpoint returns a valid SQL query
        // The actual execution would happen on the frontend
        $taskId = DB::table('scheduled_tasks')->insertGetId([
            'business_id' => $this->business->id_business,
            'task_name' => 'Dynamic Task',
            'status' => 'enabled',
            'msg' => 'Test',
            's_query' => 'SELECT id_customers, customer_name FROM customers LIMIT 10',
        ]);

        $response = $this->actingAs($this->user)->get('/scheduled-tasks/query?id=' . $taskId);
        $response->assertStatus(200);
        
        $data = $response->json();
        $this->assertNotEmpty($data['s_query']);
        $this->assertStringContainsString('SELECT', $data['s_query']);
    }
}
