<?php

namespace Tests\Feature;

use App\Models\Department;
use App\Models\Business;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Support\Facades\DB;
use Tests\TestCase;

class ServiceDepartmentsTest extends TestCase
{
    use RefreshDatabase;

    protected function setUp(): void
    {
        parent::setUp();
        $this->business = Business::factory()->create();
        $this->user = User::factory()->create(['business_id' => $this->business->id_business]);
        session(['business_id' => $this->business->id_business]);
        session(['common_services' => 'No']);
    }

    /** @test */
    public function service_departments_page_loads_successfully()
    {
        $response = $this->actingAs($this->user)->get('/service_departments');
        $response->assertStatus(200);
        $response->assertViewIs('service_department.service_departments_listing');
    }

    /** @test */
    public function can_create_new_service_department()
    {
        $response = $this->actingAs($this->user)->post('/service_departments/save_or_update', [
            'service_department' => 'Hair Services',
            'department_status' => 'Active',
        ]);

        $response->assertStatus(200);
        $response->assertJson(['success' => true]);
        
        $this->assertDatabaseHas('service_department', [
            'service_department' => 'Hair Services',
            'department_status' => 'Active',
            'business_id' => $this->business->id_business,
        ]);
    }

    /** @test */
    public function can_update_existing_service_department()
    {
        $department = Department::factory()->create([
            'business_id' => $this->business->id_business,
            'service_department' => 'Old Department',
            'department_status' => 'Active',
        ]);

        $response = $this->actingAs($this->user)->post('/service_departments/save_or_update', [
            'id_service_department' => $department->id_service_department,
            'service_department' => 'Updated Department',
            'department_status' => 'InActive',
        ]);

        $response->assertStatus(200);
        $response->assertJson(['success' => true]);
        
        $this->assertDatabaseHas('service_department', [
            'id_service_department' => $department->id_service_department,
            'service_department' => 'Updated Department',
            'department_status' => 'InActive',
        ]);
    }

    /** @test */
    public function service_department_validation_requires_name()
    {
        $response = $this->actingAs($this->user)->post('/service_departments/save_or_update', [
            'department_status' => 'Active',
        ]);

        $response->assertStatus(500);
        $response->assertJson(['success' => false]);
    }

    /** @test */
    public function service_department_validation_requires_status()
    {
        $response = $this->actingAs($this->user)->post('/service_departments/save_or_update', [
            'service_department' => 'Test Department',
        ]);

        $response->assertStatus(500);
        $response->assertJson(['success' => false]);
    }

    /** @test */
    public function service_department_status_must_be_active_or_inactive()
    {
        $response = $this->actingAs($this->user)->post('/service_departments/save_or_update', [
            'service_department' => 'Test Department',
            'department_status' => 'Invalid',
        ]);

        $response->assertStatus(500);
        $response->assertJson(['success' => false]);
    }

    /** @test */
    public function service_departments_use_ho_business_id_when_common_services_enabled()
    {
        // Create Head Office business
        $hoBusiness = Business::factory()->create(['ho' => 'Yes']);
        $this->business = Business::factory()->create(['ho' => 'No']);
        $this->user = User::factory()->create(['business_id' => $this->business->id_business]);
        
        session(['business_id' => $this->business->id_business]);
        session(['common_services' => 'Yes']);

        $response = $this->actingAs($this->user)->post('/service_departments/save_or_update', [
            'service_department' => 'Common Department',
            'department_status' => 'Active',
        ]);

        $response->assertStatus(200);
        $response->assertJson(['success' => true]);
        
        // Should be saved with HO business_id
        $this->assertDatabaseHas('service_department', [
            'service_department' => 'Common Department',
            'business_id' => $hoBusiness->id_business,
        ]);
    }

    /** @test */
    public function service_departments_use_current_business_id_when_common_services_disabled()
    {
        session(['common_services' => 'No']);

        $response = $this->actingAs($this->user)->post('/service_departments/save_or_update', [
            'service_department' => 'Branch Department',
            'department_status' => 'Active',
        ]);

        $response->assertStatus(200);
        $response->assertJson(['success' => true]);
        
        // Should be saved with current business_id
        $this->assertDatabaseHas('service_department', [
            'service_department' => 'Branch Department',
            'business_id' => $this->business->id_business,
        ]);
    }

    /** @test */
    public function service_departments_list_shows_only_current_business_departments()
    {
        Department::factory()->create([
            'business_id' => $this->business->id_business,
            'service_department' => 'Department 1',
        ]);

        $otherBusiness = Business::factory()->create();
        Department::factory()->create([
            'business_id' => $otherBusiness->id_business,
            'service_department' => 'Department 2',
        ]);

        $response = $this->actingAs($this->user)->get('/service_departments');
        $response->assertStatus(200);
        
        $departments = $response->viewData('departments');
        $this->assertCount(1, $departments);
        $this->assertEquals('Department 1', $departments->first()->service_department);
    }
}
